from django.apps import apps
from django.shortcuts import HttpResponse, get_object_or_404, render
from django.http import Http404
from django.contrib.auth.decorators import login_required
from django.views.generic import View, RedirectView
from django.views.generic.base import ContextMixin
from django.views.decorators.csrf import csrf_protect
from django.views.decorators.http import require_POST
from django.core.urlresolvers import reverse
from django.core.exceptions import ObjectDoesNotExist, ValidationError
from django.contrib.contenttypes.models import ContentType
from django.contrib.sites.models import Site
from django.conf import settings
from iconolab.models import Annotation, AnnotationRevision, Collection, Image, IconolabComment, MetaCategory, MetaCategoryInfo
from iconolab.forms.annotations import AnnotationRevisionForm
import datetime
import django_comments
from django_comments import signals
from django_comments.views.utils import next_redirect, confirmation_view


class GlobalHomepageView(View):
    def get(self, request, *args, **kwargs):
        # Handle homepage view here
        return render(request, 'iconolab/home.html');


class CollectionHomepageView(View, ContextMixin):
    def check_kwargs(self, kwargs):
        try:
            collection = Collection.objects.prefetch_related("items", "items__images").get(name=kwargs.get('collection_name'))
        except Collection.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        return True, (collection)
    
    def get(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection) = result
        else:
            return result(request)
        context = super(CollectionHomepageView, self).get_context_data(**kwargs)
        context['collection_name'] = self.kwargs.get('collection_name', '')
        context['collection'] = collection
        return render(request, 'iconolab/collection_home.html', context);


class ShowImageView(View, ContextMixin):
    
    def check_kwargs(self, kwargs):
        try:
            collection = Collection.objects.prefetch_related("items", "items__images").get(name=kwargs.get('collection_name'))
        except Collection.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            image = Image.objects.prefetch_related("annotations").get(image_guid=kwargs.get('image_guid'))
        except Image.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        return True, (collection, image)
    
    def get(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image) = result
        else:
            return result(request)
        context = super(ShowImageView, self).get_context_data(**kwargs)
        context['collection_name'] = self.kwargs.get('collection_name', '')
        context['image_guid'] = self.kwargs.get('image_guid', '')
        context['collection'] = collection
        context['image'] = image
        return render(request, 'iconolab/detail_image.html', context);
    
class CreateAnnotationView(View, ContextMixin):
    
    def get_context_data(self, **kwargs):
        context = super(CreateAnnotationView, self).get_context_data(**kwargs)
        context['collection_name'] = self.kwargs.get('collection_name', '')
        context['image_guid'] = self.kwargs.get('image_guid', '')
        return context
    
    def check_kwargs(self, kwargs):
        try:
            collection = Collection.objects.prefetch_related("items", "items__images").get(name=kwargs.get('collection_name'))
        except Collection.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            image = Image.objects.prefetch_related("annotations").get(image_guid=kwargs.get('image_guid'))
        except Image.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        return True, (collection, image)
    
    def get(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image) = result
        else:
            return result(request)
        annotation_form = AnnotationRevisionForm()
        context = self.get_context_data(**kwargs)
        context['image'] = image
        context['form'] = annotation_form
        context['tags_data'] = '[]'
        return render(request, 'iconolab/change_annotation.html', context) 
    
    def post(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image) = result
        else:
            return result(request)
        collection_name = kwargs['collection_name']
        image_guid = kwargs['image_guid']
        annotation_form = AnnotationRevisionForm(request.POST)
        if annotation_form.is_valid():
            author = request.user
            title = annotation_form.cleaned_data['title']
            description = annotation_form.cleaned_data['description']
            fragment = annotation_form.cleaned_data['fragment']
            tags_json = annotation_form.cleaned_data['tags']
            new_annotation = Annotation.objects.create_annotation(author, image, title=title, description=description, fragment=fragment, tags_json=tags_json)
            revision_comment = annotation_form.cleaned_data['comment']
            IconolabComment.objects.create(
                comment = revision_comment,
                revision = new_annotation.current_revision,
                content_type = ContentType.objects.get(app_label='iconolab', model='annotation'),
                content_object = new_annotation,
                site = Site.objects.get(id=settings.SITE_ID),
                object_pk = new_annotation.id,
                user = request.user,
                user_name = request.user.username
            )
            return RedirectView.as_view(url=reverse('annotation_detail', kwargs={'collection_name': collection_name, 'image_guid': image_guid, 'annotation_guid': new_annotation.annotation_guid}))(request)
        context = self.get_context_data(**kwargs)
        context['image'] = image
        context['form'] = annotation_form
        context['tags_data'] = '[]'
        render(request, 'iconolab/change_annotation.html', context)

class ShowAnnotationView(View, ContextMixin):
    
    def get_context_data(self, **kwargs):
        context = super(ShowAnnotationView, self).get_context_data(**kwargs)
        context['collection_name'] = self.kwargs.get('collection_name', '')
        context['image_guid'] = self.kwargs.get('image_guid', '')
        context['annotation_guid'] = self.kwargs.get('annotation_guid', '')
        return context
    
    def check_kwargs(self, kwargs):
        try:
            collection = Collection.objects.prefetch_related("items", "items__images").get(name=kwargs.get('collection_name'))
        except Collection.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            image = Image.objects.prefetch_related("annotations").get(image_guid=kwargs.get('image_guid'))
        except Image.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            annotation = Annotation.objects.select_related('current_revision').get(annotation_guid=kwargs.get('annotation_guid'))
        except Annotation.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        return True, (collection, image, annotation)
    
    def get(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image, annotation) = result
        else:
            return result(request)
        context = self.get_context_data(**kwargs)
        context['collection'] = collection
        context['image'] = image
        context['annotation'] = annotation
        context['tags_data'] = annotation.current_revision.get_tags_json()
        return render(request, 'iconolab/detail_annotation.html', context)


class EditAnnotationView(View, ContextMixin):
    
    def get_context_data(self, **kwargs):
        context = super(EditAnnotationView, self).get_context_data(**kwargs)
        context['collection_name'] = self.kwargs.get('collection_name', '')
        context['image_guid'] = self.kwargs.get('image_guid', '')
        context['annotation_guid'] = self.kwargs.get('annotation_guid', '')
        return context
    
    def check_kwargs(self, kwargs):
        try:
            collection = Collection.objects.prefetch_related("items", "items__images").get(name=kwargs.get('collection_name'))
        except Collection.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            image = Image.objects.prefetch_related("annotations").get(image_guid=kwargs.get('image_guid'))
        except Image.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            annotation = Annotation.objects.select_related('current_revision').get(annotation_guid=kwargs.get('annotation_guid'))
        except Annotation.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        return True, (collection, image, annotation)
    
    def get(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image, annotation) = result
        else:
            return result(request)
        annotation_form = AnnotationRevisionForm(instance=annotation.current_revision)
        context = self.get_context_data(**kwargs)
        context['image'] = image
        context['annotation'] = annotation
        context['form'] = annotation_form
        context['tags_data'] = annotation.current_revision.get_tags_json()
        return render(request, 'iconolab/change_annotation.html', context) 
    
    def post(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image, annotation) = result
        else:
            return result(request)
        collection_name = kwargs['collection_name']
        image_guid = kwargs['image_guid']
        annotation_guid = kwargs['annotation_guid']
        annotation_form = AnnotationRevisionForm(request.POST)
        if annotation_form.is_valid():
            revision_author = request.user
            revision_title = annotation_form.cleaned_data['title']
            revision_description = annotation_form.cleaned_data['description']
            revision_fragment = annotation_form.cleaned_data['fragment']
            revision_tags_json = annotation_form.cleaned_data['tags']
            new_revision = annotation.make_new_revision(revision_author, revision_title, revision_description, revision_fragment, revision_tags_json)
            revision_comment = annotation_form.cleaned_data['comment']
            comment = IconolabComment.objects.create(
                comment = revision_comment,
                revision = new_revision,
                content_type = ContentType.objects.get(app_label='iconolab', model='annotation'),
                content_object = annotation,
                site = Site.objects.get(id=settings.SITE_ID),
                object_pk = annotation.id,
                user = request.user,
                user_name = request.user.username
            )
            return RedirectView.as_view(url=reverse('annotation_detail', kwargs={'collection_name': collection_name, 'image_guid': image_guid, 'annotation_guid': annotation_guid}))(request)
        context = self.get_context_data(**kwargs)
        context['image'] = image
        context['form'] = annotation_form
        context['annotation'] = annotation
        context['tags_data'] = annotation.current_revision.get_tags_json()
        return render(request, 'iconolab/change_annotation.html', context)


class ShowRevisionView(View, ContextMixin):
    
    def get_context_data(self, **kwargs):
        context = super(ShowRevisionView, self).get_context_data(**kwargs)
        context['collection_name'] = self.kwargs.get('collection_name', '')
        context['image_guid'] = self.kwargs.get('image_guid', '')
        context['annotation_guid'] = self.kwargs.get('annotation_guid', '')
        context['revision_guid'] = self.kwargs.get('revision_guid', '')
        return context
    
    def check_kwargs(self, kwargs):
        try:
            collection = Collection.objects.prefetch_related("items", "items__images").get(name=kwargs.get('collection_name'))
        except Collection.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            image = Image.objects.prefetch_related("annotations").get(image_guid=kwargs.get('image_guid'))
        except Image.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            annotation = Annotation.objects.select_related('current_revision').get(annotation_guid=kwargs.get('annotation_guid'))
        except Annotation.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            revision = AnnotationRevision.objects.select_related('parent_revision').get(revision_guid=kwargs.get('revision_guid'))
        except AnnotationRevision.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        return True, (collection, image, annotation, revision)
    
    def get(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image, annotation, revision) = result
        else:
            return result(request)
        context = self.get_context_data(**kwargs)
        context['collection'] = collection
        context['image'] = image
        context['annotation'] = annotation
        context['revision'] = revision
        context['tags_data'] = revision.get_tags_json()
        context['comment'] = revision.creation_comment.first()
        return render(request, 'iconolab/detail_revision.html', context)

        
class MergeProposalView(View, ContextMixin):
    
    def get_context_data(self, **kwargs):
        context = super(MergeProposalView, self).get_context_data(**kwargs)
        context['collection_name'] = self.kwargs.get('collection_name', '')
        context['image_guid'] = self.kwargs.get('image_guid', '')
        context['annotation_guid'] = self.kwargs.get('annotation_guid', '')
        context['revision_guid'] = self.kwargs.get('revision_guid', '')
        return context
    
    def check_kwargs(self, kwargs):
        try:
            collection = Collection.objects.prefetch_related("items", "items__images").get(name=kwargs.get('collection_name'))
        except Collection.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            image = Image.objects.prefetch_related("annotations").get(image_guid=kwargs.get('image_guid'))
        except Image.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            annotation = Annotation.objects.select_related('current_revision').get(annotation_guid=kwargs.get('annotation_guid'))
        except Annotation.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        try:
            revision = AnnotationRevision.objects.select_related('parent_revision').get(revision_guid=kwargs.get('revision_guid'))
        except AnnotationRevision.DoesNotExist:
            return False, RedirectView.as_view(url=reverse('404error'))
        return True, (collection, image, annotation, revision)
    
    def get(self, request, *args, **kwargs):
        success, result = self.check_kwargs(kwargs)
        if success:
            (collection, image, annotation, revision) = result
        else:
            return result(request)
        # Only show merge form if there is a revision to merge AND the current user is the annotation author
        if revision.state != AnnotationRevision.AWAITING or request.user != annotation.author:
            return RedirectView.as_view(
                url=reverse('revision_detail', 
                    kwargs={
                        'collection_name': collection.name, 
                        'image_guid': image.image_guid, 
                        'annotation_guid': annotation.annotation_guid,
                        'revision_guid': revision.revision_guid
                    }
                )
            )
        context = self.get_context_data(**kwargs)
        context['collection'] = collection
        context['image'] = image
        context['annotation'] = annotation
        # Proposal data
        context['proposal_revision'] = revision
        context['proposal_tags_data'] = revision.get_tags_json()
        context['proposal_comment'] = revision.creation_comment.first()
        # Parent data
        context['parent_revision'] = revision.parent_revision
        context['parent_tags_data'] = revision.parent_revision.get_tags_json()
        context['parent_comment'] = revision.parent_revision.creation_comment.first()
        # Current data
        context['current_revision'] = revision
        context['current_tags_data'] = revision.get_tags_json()
        context['current_comment'] = revision.creation_comment.first()
        return render(request, 'iconolab/merge_revision.html', context)
    
    def post(self, request, *args, **kwargs):
        # Handle merge form submit here
        pass
    
    
class NotFoundErrorView(View):
    def get(self, request, *args, **kwargs):
        # Handle image display here
        pass
    

@csrf_protect
@require_POST
def post_comment_iconolab(request, next=None, using=None):
    '''
    Post a comment.
    HTTP POST is required. If ``POST['submit'] == 'preview'`` or if there are
    errors a preview template, ``comments/preview.html``, will be rendered.
    '''
    # Fill out some initial data fields from an authenticated user, if present
    data = request.POST.copy()
    if request.user.is_authenticated():
        if not data.get('name', ''):
            data['name'] = request.user.get_full_name() or request.user.get_username()
        if not data.get('email', ''):
            data['email'] = request.user.email

    # Look up the object we're trying to comment about
    ctype = data.get('content_type')
    object_pk = data.get('object_pk')
    if ctype is None or object_pk is None:
        return CommentPostBadRequest('Missing content_type or object_pk field.')
    try:
        model = apps.get_model(*ctype.split('.', 1))
        target = model._default_manager.using(using).get(pk=object_pk)
    except TypeError:
        return CommentPostBadRequest(
            'Invalid content_type value: %r' % escape(ctype))
    except AttributeError:
        return CommentPostBadRequest(
            'The given content-type %r does not resolve to a valid model.' % escape(ctype))
    except ObjectDoesNotExist:
        return CommentPostBadRequest(
            'No object matching content-type %r and object PK %r exists.' % (
                escape(ctype), escape(object_pk)))
    except (ValueError, ValidationError) as e:
        return CommentPostBadRequest(
            'Attempting go get content-type %r and object PK %r exists raised %s' % (
                escape(ctype), escape(object_pk), e.__class__.__name__))

    # Do we want to preview the comment?
    preview = 'preview' in data

    # Construct the comment form
    form = django_comments.get_form()(target, data=data)

    # Check security information
    if form.security_errors():
        return CommentPostBadRequest(
            'The comment form failed security verification: %s' % escape(str(form.security_errors())))

    # If there are errors or if we requested a preview show the comment
    if form.errors:
        return render(request, 'iconolab/detail_annotation.html', {
                'comment_form': form,
                'next': data.get('next', next),
                'annotation': target,
                'annotation_guid': target.annotation_guid,
                'image_guid': target.image.image_guid,
                'collection_name': target.image.item.collection.name,
                'tags_data': target.current_revision.get_tags_json()
            },
        )

    # Otherwise create the comment
    comment = form.get_comment_object()
    comment.ip_address = request.META.get('REMOTE_ADDR', None)
    if request.user.is_authenticated():
        comment.user = request.user

    # Signal that the comment is about to be saved
    responses = signals.comment_will_be_posted.send(
        sender=comment.__class__,
        comment=comment,
        request=request
    )

    for (receiver, response) in responses:
        if response is False:
            return CommentPostBadRequest(
                'comment_will_be_posted receiver %r killed the comment' % receiver.__name__)

    # Save the comment and signal that it was saved
    comment.save()
    
    
    signals.comment_was_posted.send(
        sender=comment.__class__,
        comment=comment,
        request=request
    )
    
    # Creating metacategories here as apparently there is no way to make it work easily woth django_comments_xtd
    for metacategory in form.cleaned_data.get("metacategories", []):
        if 'xtd_comment' in comment:
            metacategory_info = MetaCategoryInfo.objects.create(
                comment = comment['xtd_comment'],
                metacategory = metacategory
            )

    return next_redirect(request, fallback=next or 'comments-comment-done',
                         c=comment._get_pk_val())