from django.db import models
from django.contrib.auth.models import User

from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from reversion import revisions as reversion
from reversion.models import Revision


class Tag(models.Model):
	label = models.SlugField()
	link = models.URLField()
	description = models.TextField()

	def __str__(self):
		return '%d:%s' % (self.id, self.label)


# fonds Ingres - Musee de la Poste 
class Collection(models.Model):
	name =	models.CharField(max_length=50)
	description = models.CharField(max_length=255)

	def __str__(self):
		return self.name

# class image_metadata
class Image(models.Model):
	name = models.CharField(max_length=200)
	media = models.ImageField(upload_to='uploads/', height_field='height', width_field='width')
	height = models.IntegerField(null=False, blank=False)
	width = models.IntegerField(null=False, blank=False)
	mimetype = models.CharField(null=True, blank=True, max_length=1024)
	exif = models.TextField(null=True, blank=True) 
	collection = models.ForeignKey(Collection)
	date_created = models.DateTimeField(auto_now_add=True, null=True)
	date_modified = models.DateTimeField(auto_now=True, null=True)
	
	def __str__(self):
		return self.name

# Folders
class Folder(models.Model):
	label = models.CharField(max_length=255)
	owner = models.ForeignKey(User)
	images = models.ManyToManyField(Image)

	def __str__(self):
		return label

@reversion.register()
class Annotation(models.Model):
	title = models.CharField(max_length=255)
	description = models.TextField(null=True)
	fragment = models.TextField() # path string
	tags = models.ManyToManyField(Tag)
	image = models.ForeignKey(Image, default=0, on_delete=models.CASCADE)
	author = models.ForeignKey(User, null=True)
	date_created = models.DateTimeField(auto_now_add=True, null=True)
	date_modified = models.DateTimeField(auto_now=True, null=True)

	def __str__(self):
		return self.title

# comments

LINK = 1
IMAGE = 2
PDF = 3

COMMENT_CHOICES = (
	(LINK, 'link'),
	(IMAGE, 'image'),
	(PDF, 'pdf')
	)


class MetaCategory(models.Model):
	collection = models.ForeignKey(Collection)
	label = models.CharField(max_length=200)

	def __str__(self):
		return self.label

	class Meta:
		verbose_name_plural = "Metacategories"

class Comment(models.Model):
	author =  models.ForeignKey(User)
	created_date = models.DateTimeField(blank=False, null=False, auto_now_add=True)
	annotation = models.ForeignKey(Annotation, null=True) # revision de l'utilisateur?
	target = models.ForeignKey("self") #
	#thread_id #django-contrib-comment #threadedcomment
	content = models.TextField(blank=True)
	metacategories = models.ManyToManyField(MetaCategory) 


class CommentAttachement(models.Model):
	comment = models.ForeignKey(Comment, on_delete=models.CASCADE, related_name='attachments')
	main_annotation = models.ForeignKey(Annotation)
	attachment_type = models.IntegerField(choices=COMMENT_CHOICES, default=1)
	created_date = models.DateTimeField(auto_now_add=True)
	data = models.TextField(blank=False)


# Activity & Notification

class Activity(models.Model):

	NEW_COMMENT = 1
	NEW_REVISION = 2
	NEW_REVISION_COMMENT = 3
	NEW_EXPERT_ANSWER = 4
	NEW_REFERENCE = 5
	EXPERT_CALL = 6

	ACTIVITY_VERBS =(
		(NEW_COMMENT, 'Nouveau commentaire'),
		(NEW_REVISION, 'Nouvelle revision'),
		(NEW_REVISION_COMMENT, 'Nouveau commentaire de revision'),
		(NEW_EXPERT_ANSWER, 'New expert ans')
		) 

	verb = models.IntegerField(choices=ACTIVITY_VERBS)
	actor = models.ForeignKey(User)

	target_content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
	target_object_id = models.PositiveIntegerField()
	target = GenericForeignKey('target_content_type', 'target_object_id')

	action_content_type = models.ForeignKey(ContentType, related_name='activity_action', on_delete=models.CASCADE, null=True, blank=True)
	action_object_id = models.PositiveIntegerField(null=True, blank=True)
	action_content = GenericForeignKey('action_content_type', 'action_object_id') 
	
	created_date = models.DateTimeField(auto_now_add=True)

	def __str__(self):
		return '%s:%s' % (author.name, verbe)


class Notification(models.Model):

	UNREAD = 0
	READ = 1
	DELETED = 2

	STATUS = (
		(READ, 'Lu'),
		(UNREAD, 'Non lu'),
		(DELETED, 'Efface')
		)

	activity = models.ForeignKey(Activity)
	user = models.ForeignKey(User)
	status = models.IntegerField(choices=STATUS, default=UNREAD)
	created_date = models.DateTimeField(auto_now_add=True)

