/* enable zoom */
import { eventEmitter } from '../utils'

class ZoomHandler {

	constructor (params) {
		this.zoomFactor = 0.1 || params.zoomFactor;
		this.paper = params.paper;
		this.viewport = {width: 529, height: 800};
		this.scale = 1;
		this.disableDrag = false;
		this.paper.attr({"preserveAspectRatio": "xMidYMid meet"});
		this.currentViewBox = [0, 0, this.viewport.width, this.viewport.height];
		this.paper.drag(this.onDrag.bind(this), this.onStart.bind(this), this.onStop.bind(this));
		this.lastPosition = []; 
	}

	zoomIn () {
		var viewBoxW = this.viewport.width - (this.viewport.width * this.zoomFactor * this.scale);
		var viewBoxH = this.viewport.height - (this.viewport.height * this.zoomFactor * this.scale);
		if (!viewBoxW || !viewBoxH) { return false; }

		/* center the viewbox */
		var topX = (this.viewport.width - viewBoxW) / 2; 
		var topY = (this.viewport.height - viewBoxH) / 2; 

		this.currentViewBox[0] = topX; 
		this.currentViewBox[1] = topY;

		this.currentViewBox[2] = viewBoxW;
		this.currentViewBox[3] = viewBoxH;

		this.updateViewBox();
		this.scale ++;
	}
	
	updateViewBox () {
		this.paper.attr({"viewBox": this.currentViewBox});
		eventEmitter.emit("zoomChanged", {
			"zoomFactor": this.getZoomFactor(),
			"viewport": this.viewport,
			"currentViewBox": this.currentViewBox
		});
	}
	
	getZoomFactor () {
		return {
			x: this.viewport.width / this.currentViewBox[2],
			y: this.viewport.height / this.currentViewBox[3]	
		};
	}

	onStart (x, y, e) {
		
		this.lastPosition[0] = this.currentViewBox[0];
		this.lastPosition[1] = this.currentViewBox[1];

		if (e.target.className.baseVal === "drawingHandler") {
			this.disableDrag = true;
		}
	}

	canDrag () {
		return !this.disableDrag;
	}

	onStop () {
		this.disableDrag = false;
	}

	onDrag (dx, dy, x, y, event) {

		if (!this.canDrag()) { return true; }

		var newX = this.lastPosition[0] - dx;
		var newY = this.lastPosition[1] - dy;

		/* maxX bound */
		if (newX + this.currentViewBox[2] >= this.viewport.width) {
			newX = this.viewport.width - this.currentViewBox[2];
		}

		/* maxY bound */
		if (newY + this.currentViewBox[3] >= this.viewport.height) {
			newY = this.viewport.height - this.currentViewBox[3];
		}

		if (newX <= 0) { newX = 0; }

		if(newY <= 0) { newY = 0; }

		this.currentViewBox[0] = newX;
		this.currentViewBox[1] = newY;

		this.updateViewBox();
	}

	reset () {
		this.scale = 1;
		this.currentViewBox = [0, 0, this.viewport.width, this.viewport.height];
		this.updateViewBox();
	}

	zoomOut () {
		if (this.scale === 1) { return false; }
		var viewBoxW = this.currentViewBox[2] + (this.viewport.width * this.zoomFactor);
		var viewBoxH = this.currentViewBox[3] + (this.viewport.height * this.zoomFactor);
		if (viewBoxW > this.viewport.width || viewBoxW > this.viewport.height) {
			return false;
		}

		var topX = (this.viewport.width - viewBoxW) / 2;
		var topY = (this.viewport.height - viewBoxH) / 2; 
  
		this.currentViewBox[0] = topX;
		this.currentViewBox[1] = topY; 
		this.currentViewBox[2] = viewBoxW;
		this.currentViewBox[3] = viewBoxH;

		this.scale--;
		this.updateViewBox();
		
	}
}

export default {

	enable_zoom: function (params) {
		return new ZoomHandler(params);
	}
}