from django.shortcuts import HttpResponse, get_object_or_404, render
from django.http import Http404
from django.contrib.auth.decorators import login_required
from django.views.generic import View, RedirectView
from django.views.generic.base import ContextMixin
from django.core.urlresolvers import reverse
from iconolab.models import Annotation, Collection, Image
from iconolab.forms.annotations import AnnotationRevisionForm
import json

def make_tags_json(annotation_revision):
	final_list = []
	for tagging_info in annotation_revision.tagginginfo_set.all():
		final_list.append({
			"tag_label": tagging_info.tag.label,
			"tag_link": tagging_info.tag.link,
			"accuracy": tagging_info.accuracy,
			"relevancy": tagging_info.relevancy
		})
	return json.dumps(final_list) 

class GlobalHomepageView(View):
	def get(self, request, *args, **kwargs):
		# Handle homepage view here
		return render(request, 'iconolab/home.html');


class CollectionHomepageView(View, ContextMixin):
	def get(self, request, *args, **kwargs):
		context = super(CollectionHomepageView, self).get_context_data(**kwargs)
		context["collection_name"] = self.kwargs.get("collection_name", "")
		return render(request, 'iconolab/collection_home.html', context);


class ShowImageView(View, ContextMixin):
	
	def check_kwargs(self, kwargs):
		try:
			collection = Collection.objects.get(name=kwargs.get("collection_name", ""))
		except Collection.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		try:
			image = Image.objects.get(image_guid=kwargs.get("image_guid", ""))
		except Image.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		return collection, image
	
	def get(self, request, *args, **kwargs):
		collection, image = self.check_kwargs(kwargs)
		context = super(ShowImageView, self).get_context_data(**kwargs)
		context["collection_name"] = self.kwargs.get("collection_name", "")
		context["image_guid"] = self.kwargs.get("image_guid", "")
		context["collection"] = collection
		context["image"] = image
		return render(request, 'iconolab/detail_image.html', context);
    
class CreateAnnotationView(View, ContextMixin):
	
	def get_context_data(self, **kwargs):
		context = super(CreateAnnotationView, self).get_context_data(**kwargs)
		context["collection_name"] = self.kwargs.get("collection_name", "")
		context["image_guid"] = self.kwargs.get("image_guid", "")
		return context
	
	def check_kwargs(self, kwargs):
		try:
			collection = Collection.objects.get(name=kwargs.get("collection_name", ""))
		except Collection.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		try:
			image = Image.objects.get(image_guid=kwargs.get("image_guid", ""))
		except Image.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		return collection, image
	
	def get(self, request, *args, **kwargs):
		collection, image = self.check_kwargs(kwargs)
		annotation_form = AnnotationRevisionForm()
		context = self.get_context_data(**kwargs)
		context["image"] = image
		context["form"] = annotation_form
		return render(request, 'iconolab/change_annotation.html', context) 
	
	def post(self, request, *args, **kwargs):
		collection, image = self.check_kwargs(kwargs)
		collection_name = kwargs["collection_name"]
		image_guid = kwargs["image_guid"]
		annotation_form = AnnotationRevisionForm(request.POST)
		if annotation_form.is_valid():
			author = request.user
			title = annotation_form.cleaned_data["title"]
			description = annotation_form.cleaned_data["description"]
			fragment = annotation_form.cleaned_data["fragment"]
			tags_json = annotation_form.cleaned_data["tags"]
			new_annotation = Annotation.objects.create_annotation(author, image, title=title, description=description, fragment=fragment, tags_json=tags_json)
			return RedirectView.as_view(url=reverse("annotation_detail", kwargs={'collection_name': collection_name, 'image_guid': image_guid, 'annotation_guid': new_annotation.annotation_guid}))(request)


class ShowAnnotationView(View, ContextMixin):
	
	def get_context_data(self, **kwargs):
		context = super(ShowAnnotationView, self).get_context_data(**kwargs)
		context["collection_name"] = self.kwargs.get("collection_name", "")
		context["image_guid"] = self.kwargs.get("image_guid", "")
		context["annotation_guid"] = self.kwargs.get("annotation_guid", "")
		return context
	
	def check_kwargs(self, kwargs):
		try:
			collection = Collection.objects.get(name=kwargs.get("collection_name", ""))
		except Collection.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		try:
			image = Image.objects.get(image_guid=kwargs.get("image_guid", ""))
		except Image.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		try:
			annotation = Annotation.objects.select_related("current_revision").get(annotation_guid=kwargs.get("annotation_guid", ""))
		except Annotation.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		return collection, image, annotation
	
	def get(self, request, *args, **kwargs):
		collection, image, annotation = self.check_kwargs(kwargs)
		self.check_kwargs(kwargs)
		context = self.get_context_data(**kwargs)
		context["annotation"] = annotation
		return render(request, 'iconolab/detail_annotation.html', context)


class EditAnnotationView(View, ContextMixin):
	
	def get_context_data(self, **kwargs):
		context = super(EditAnnotationView, self).get_context_data(**kwargs)
		context["collection_name"] = self.kwargs.get("collection_name", "")
		context["image_guid"] = self.kwargs.get("image_guid", "")
		context["annotation_guid"] = self.kwargs.get("annotation_guid", "")
		return context
	
	def check_kwargs(self, kwargs):
		try:
			collection = Collection.objects.get(name=kwargs.get("collection_name", ""))
		except Collection.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		try:
			image = Image.objects.get(image_guid=kwargs.get("image_guid", ""))
		except Image.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		try:
			annotation = Annotation.objects.select_related("current_revision").get(annotation_guid=kwargs.get("annotation_guid", ""))
		except Annotation.DoesNotExist:
			return RedirectView.as_view(url=reverse("404error"))
		return collection, image, annotation
	
	def get(self, request, *args, **kwargs):
		collection, image, annotation = self.check_kwargs(kwargs)
		annotation_form = AnnotationRevisionForm(instance=annotation.current_revision)
		context = self.get_context_data(**kwargs)
		context["image"] = image
		context["annotation"] = annotation
		context["form"] = annotation_form
		context["tags_data"] = make_tags_json(annotation.current_revision)
		return render(request, 'iconolab/change_annotation.html', context) 
	
	def post(self, request, *args, **kwargs):
		collection, image, annotation = self.check_kwargs(kwargs)
		collection_name = kwargs["collection_name"]
		image_guid = kwargs["image_guid"]
		annotation_guid = kwargs["annotation_guid"]
		annotation_form = AnnotationRevisionForm(request.POST)
		if annotation_form.is_valid():
			revision_author = request.user
			revision_title = annotation_form.cleaned_data["title"]
			revision_description = annotation_form.cleaned_data["description"]
			revision_fragment = annotation_form.cleaned_data["fragment"]
			revision_tags_json = annotation_form.cleaned_data["tags"]
			annotation.make_new_revision(revision_author, revision_title, revision_description, revision_fragment, revision_tags_json)
			return RedirectView.as_view(url=reverse("annotation_detail", kwargs={'collection_name': collection_name, 'image_guid': image_guid, 'annotation_guid': annotation_guid}))(request)
		print(annotation_form.errors)
		return HttpResponse("wow")
		
class MergeProposalView(View):
	
	def get(self, request, *args, **kwargs):
		# Handle merge form display here
		pass
	
	def post(self, request, *args, **kwargs):
		# Handle merge form submit here
		pass
	
	
class NotFoundErrorView(View):
	def get(self, request, *args, **kwargs):
		# Handle image display here
		pass