/* Enabling us to resize a shape width a handler 
	#http://stackoverflow.com/questions/32390028/how-to-drag-and-resize-svg-rectangle-using-cursor-types
*/
import { eventEmitter } from "../utils"

var ShapeResizer = function (paper, shape) {
	this.paper = paper;
	this.shape = shape;
	this.handlers = [];
	this.isResizing = false;
	this.currentPosition = {};
	this.HANDLER_SIZE = 8;
	this.SHAPE_MIN_SIZE = 20;
	this.states = {};
	this.noop = function (){}
	this.init();
}

var api = ShapeResizer.prototype = {

	init: function () {
		this.currentZoomFactor = {x: 1, y: 1};
		this.showHandlers();
	},

	showHandlers: function () {
		/* show handler here */
		var bbox = this.shape.getBBox();
		var handleX = bbox.x - this.HANDLER_SIZE/2;
		var handleY = bbox.y - this.HANDLER_SIZE/2;
		var handler = this.paper.rect(handleX, handleY, this.HANDLER_SIZE, this.HANDLER_SIZE).attr({fill: 'red'});
		handler.addClass("drawingHandler");
		this.shape.addClass("drawingHandler");
		var handlerInfos = {position: "t_r", handler: handler};
		this.handlers.push(handlerInfos);
		this.shapesGroup = this.paper.g(this.shape, handler);
		this.attachEvents();
	},

	/*one handlers */
	updateShapePositions: function (handlerData, dx, dy) {
		//start
		var handlerBBox = handlerData.handler.getBBox();
		var shapeBBox = this.shape.data("origBbox");
		var newX = handlerBBox.x + this.HANDLER_SIZE / 2;
		var newY = handlerBBox.y + this.HANDLER_SIZE / 2;

		/*to the right => reduce the size */
		var newWidth = (dx > 0) ? shapeBBox.width - dx : shapeBBox.width + Math.abs(dx); 
		var newHeight = (dy > 0) ? shapeBBox.height - dy : shapeBBox.height + Math.abs(dy);
		
		var transformValue = this.shape.data('origTransform') + (this.shape.data('origTransform') ? "T" : "t") + [dx, dy];
		this.shape.attr({'transform': transformValue, width: newWidth, height: newHeight});
	},

	dragEvents: {
		onStart: function (handlerData, dx, dy, e) {
			this.startPosition = {x: e.clientX, y: e.clientY};
			this.isResizing = true;
			this.currentPosition = {};
			handlerData.handler.data("origTransform", handlerData.handler.transform().local);
			this.shape.data("origBbox", this.shape.getBBox());
			this.shape.data("origTransform", this.shape.transform().local);
		},

		onMove: function (handlerData, dx, dy, x, y, e) {
			dx = dx / this.currentZoomFactor.x;
			dy = dy / this.currentZoomFactor.y;
			var transformValue = handlerData.handler.data('origTransform') + (handlerData.handler.data('origTransform') ? "T" : "t") + [dx, dy];
			this.currentPosition.x = e.clientX;
			this.currentPosition.y = e.clientY;
			/* deal with boundaries */
			if (! this.checkBondaries(dx, dy)) { return; }
			handlerData.handler.attr({ transform: transformValue});
			this.updateShapePositions(handlerData, dx, dy);
		},

		onStop: function () {
			this.isResizing = false;
			this.startPosition = {};
			this.currentPosition = {};
		}
	},
	
	checkBondaries: function (dx, dy) {
		var result = true;

		if (this.shape.data("origBbox").width - dx <=  this.SHAPE_MIN_SIZE) {
			result = false;
		}

		if (this.shape.data("origBbox").height - dy <= this.SHAPE_MIN_SIZE) {
			result = false;
		}

		return result;
	},

	destroy: function () {
		this.handlers.map(function (handlerData) {
			handlerData.handler.remove();
		});
		delete this;
	},

	attachEvents: function () {
		var self = this;
		this.handlers.map(function (handlerData) {
			handlerData.handler.drag(
				self.dragEvents.onMove.bind(self, handlerData),
			 	self.dragEvents.onStart.bind(self, handlerData),
				self.dragEvents.onStop.bind(self, handlerData));
		});

		eventEmitter.on("cutout:clear", function () {
			self.destroy();
		});

		eventEmitter.on("zoomChanged", function (zoomInfos) {
			self.currentZoomFactor = zoomInfos.zoomFactor; 
		});

		this.shapesGroup.drag(function (dx, dy) {
			if (self.isResizing) { return; }
			dx = dx / self.currentZoomFactor.x;
			dy = dy / self.currentZoomFactor.y;
			var transformValue = this.data('origTransform') + (this.data('origTransform') ? "T" : "t") + [dx, dy];
			this.transform(transformValue);
		}, function () {
			this.data('origTransform', this.transform().local);
		}, this.noop);
	},
}

export default {

	enable_resizer : function (paper, rect) {
		new ShapeResizer(paper, rect);
	}
}