# -*- coding: UTF-8 -*-
import json
import logging
import os
import imghdr

from django.conf import settings

from iconolab.management.commands.importimages import BaseImportImagesCommand
from iconolab.models import (Collection, ItemMetadata)

if settings.IMPORT_LOGGER_NAME and settings.LOGGING['loggers'].get(settings.IMPORT_LOGGER_NAME, ''):
    logger = logging.getLogger(settings.IMPORT_LOGGER_NAME)
else:
    logger = logging.getLogger(__name__)

class Command(BaseImportImagesCommand):
    help = 'import images from a directory into the media folder and creates item and image objects'

    def add_arguments(self, parser):
        parser.add_argument('source_dir')
        parser.add_argument(
            '--collection-name',
            dest='collection_name',
            default=False,
            help='insert extracted data into the specified collection'

        )
        parser.add_argument(
            '--no-jpg-conversion',
            dest='no-jpg-conversion',
            default=False,
            help='use this option if you only want the image copied and not converted'
        )


    def handle(self, *args, **options):

        print('# Logging with logger '+logger.name)
        logger.debug('# Initializing command with args: %r', options)
      
        self.source_dir = options.get('source_dir')

        collection_name = options.get('collection_name')

        if collection_name:
            print('## Finding collection with id ' + 
                  collection_name)
            try:
                collection = Collection.objects.get(
                    name=collection_name)
            except Collection.DoesNotExist:
                raise ValueError('!!! Collection with primary key ' +
                                 collection_name +' was not found, aborting !!!')
        else:
            raise ValueError(
                '!!! No collection fixture or collection id, aborting because we can\'t properly generate data. !!!')


        '''Listing image files in target directory'''

        print(
            '## Converting image and moving it to static dir, creating Image and Item objects')
        print('### Images will be stored in ' + os.path.join(settings.MEDIA_ROOT, 'uploads'))

        for dirname, _, files in os.walk(self.source_dir):
            for filename in files:
                filename_without_extension, _ = os.path.splitext(filename)
                if imghdr.what(os.path.join(dirname, filename)) is None:
                    continue

                json_path = os.path.join(dirname, filename_without_extension + ".json")
                if not os.path.isfile(json_path):
                    continue

                with open(json_path) as json_data:
                    item_data = json.load(json_data)
                    item_image = item_data['image']
                
                    path_images = os.path.join(dirname, filename)
                    image_list = [path_images]

                    natural_key = ItemMetadata.get_natural_key(collection, item_image['id'])

                    if ItemMetadata.objects.filter(
                            item__collection=collection, natural_key=natural_key).exists():
                        print('#### An item with ' +
                              natural_key +' for natural key, already exists in database in the import collection')
                    else:
                        self.create_item_and_metadata(
                            natural_key, collection, item_data, image_list, options)

        print('# All done!')