# -*- coding: utf-8 -*-
'''
Created on Jan 27, 2012

@author: ymh
'''
from hdabo.models import Tag, Datasheet
from hdalab import fields

from django.db import models


class TagYears(models.Model):
    """
    Représente une période (début - fin) associée à un Tag.

    :cvar object tag: Le tag
    :cvar int start_year: L'année de début de la période.
    :cvar int end_year: L'année de fin de période.
    """
    tag = models.OneToOneField(Tag, related_name="years", blank=False, null=False, db_index=True)
    start_year = models.IntegerField(blank=False, null=False, db_index=True)
    end_year = models.IntegerField(blank=False, null=False, db_index=True)

    class Meta:
        app_label = 'hdalab'


class TagLinks(models.Model):
    """
    Représente un lien entre tag.

    :cvar object subject: Le tag sujet (source) du lien.
    :cvar object object: Le tag object (cible) du lien.
    """

    subject = models.ForeignKey(Tag, blank=False, null=False, db_index=True, related_name="taglinks_subjects")
    object = models.ForeignKey(Tag, blank=False, null=False, db_index=True, related_name="taglinks_objects")

    class Meta:
        app_label = 'hdalab'

class Country(models.Model):
    """
    Représente un pays.

    :cvar str dbpedia_uri: L'URI DBPedia de ce pays.
    """

    dbpedia_uri = models.URLField(max_length=255, blank=False, null=False, db_index=True, unique=True)

    class Meta:
        app_label = 'hdalab'

class GeoInclusion(models.Model):
    """
    Représente une relation entre un tag et un pays. (par exemple, le tage "Paris" est inclus dans le pays "France")
    Un tag ne peut être relié qu'a un seul pays.

    :cvar object tag: le tag.
    :cvar object country: Le pays.

    """

    tag = models.OneToOneField(Tag, related_name="locatedin", db_index=True)
    country = models.ForeignKey(Country, blank=False, null=False, related_name="includes", db_index=True)

    class Meta:
        app_label = 'hdalab'

class DbpediaFields(models.Model):
    """
    Informations DBPedia pour un tag.
    Un tag ne peut être lié qu'à une seule instance de cette classe.

    :cvar str dbpedia_uri: L'URI DBPedia.
    :cvar object tag: Le tag.
    :cvar str abstract: Texte résumé du tag issu de DBPedia.
    :cvar str thumbnail: URL d'un thumbnail pour le tag.
    :cvar str label: Label DBPedia pour le tag.

    """

    dbpedia_uri = models.URLField(max_length=2048, blank=False, null=False, db_index=True, unique=False)
    tag = fields.OneToOneField(Tag, db_index=True, related_name="dbpedia_fields", related_default=lambda instance: None)
    abstract = models.TextField(blank=True, null=True)
    thumbnail = models.URLField(max_length=2048, blank=True, null=True, db_index=False)
    label = models.CharField(max_length=2048, unique=False, blank=True, null=True)

    class Meta:
        app_label = 'hdalab'


class DbpediaFieldsTranslation(models.Model):
    """
    Les traductions pour un :class:DbpediaField.

    :cvar object master: Le DBPediafield
    :cvar str language_code: Le code de la langue de la traduction.
    :cvar str label: Le label traduit. Attention si une traduction du label n'a pas été trouvée, on trouvera là le même label que dans "master".
    :cvar str abstract: Le résumé traduit. Attention si une traduction du résumé n'a pas été trouvée, on trouvera là le même label que dans "master".
    :cvar bool is_label_translated: Indique si le label est effectivement traduit.
    :cvar bool is_abstract_translated: Indicque si le résumé est effectivement traduit.
    """

    master = models.ForeignKey(DbpediaFields, blank=False, null=False, db_index=True, related_name="translations")
    language_code = models.CharField(max_length=15, blank=False, null=False, db_index=True)
    label = models.CharField(max_length=2048, unique=False, blank=True, null=False)
    abstract = models.TextField(blank=True, null=True)
    is_label_translated = models.BooleanField(default=True)
    is_abstract_translated = models.BooleanField(default=True)
    class Meta:
        app_label = 'hdalab'
        unique_together = ('master', 'language_code')


class HdaSession(models.Model):
    """
    Classe destinée à sauvagarder des session d'exploration sur HDALab (état de l'interface de recherche par facette).
    Cet objet n'est pas utilisé.
    """

    sessionid = models.CharField(max_length=36, unique=True, blank=False, null=False, db_index=True)
    data = models.TextField(blank=True, null=True)

    class Meta:
        app_label = 'hdalab'

class InseeCoords(models.Model):
    """
    Coordonnées géographiques et nom de l'entité pour un code INSEE.

    :cvar integer insee: code insee.
    :cvar str city_name: Nom de la commune.
    :cvar float latitude: Latitude.
    :cvar float longitude: Longitude.
    """

    insee = models.IntegerField(primary_key=True)
    city_name = models.CharField(max_length=255, blank=False, null=False)
    latitude = models.FloatField(blank=False, null=False)
    longitude = models.FloatField(blank=False, null=False)

    class Meta:
        app_label = 'hdalab'

class DatasheetExtras(models.Model):
    """
    Information supplémentaire pour und fiche hda.

    :cvar object datasheet: La fiche HDA.
    :cvar object insee: Les information insee (:class:InseeCoords) pour la fiche.
    """

    datasheet = models.OneToOneField(Datasheet, related_name="extras", db_index=True)
    insee = models.ForeignKey(InseeCoords, blank=True, null=True, db_index=True)

    class Meta:
        app_label = 'hdalab'
