# -*- coding: utf-8 -*-

from django.conf import settings
from django.contrib.auth.decorators import login_required #@UnusedImport
from django.core.paginator import Paginator
from django.db.models import Max
from django.http import HttpResponseBadRequest
from django.shortcuts import render_to_response, redirect
from django.template import RequestContext
from haystack.constants import DJANGO_ID
from haystack.query import SearchQuerySet
from hdabo.wp_utils import process_tag
from hdabo.utils import OrderedDict
from hdabo.wp_utils import (normalize_tag, query_wikipedia_title,
    get_or_create_tag)
from models import Datasheet, Organisation, Tag, TagCategory, TaggedSheet
from wikitools import wiki
import django.utils.simplejson as json



#@login_required
def home(request):
    
    # Get all organizations
    orgas = Organisation.objects.all().order_by('name')
    # Count all validated, unvalidated sheets for each organisation
    org_list = []
    for orga in orgas :
        all_datasheets = Datasheet.objects.filter(organisation=orga)
        nb_all = len(all_datasheets)
        nb_val = len(all_datasheets.filter(validated=True))
        nb_unval = len(all_datasheets.filter(validated=False))
        org_list.append({'organisation':orga, 'nb_all':nb_all, 'nb_val':nb_val, 'nb_unval':nb_unval})
    
    return render_to_response("organisation_list.html",
                              {'organisations':org_list},
                              context_instance=RequestContext(request))


#@login_required
def list_for_orga(request, orga_id=None, valid=None, start_index=None):
    
    orga = Organisation.objects.get(id=orga_id)
    orga_name = orga.name
    
    if start_index :
        try:
            start_index = int(start_index)
        except :
            start_index = 0
    else :
        start_index = 0
    
    # If valid = 0, we search unvalidated sheets
    # If valid = 1, we search validated sheets
    # If valid = 2, we search AND DISPLAY all sheets
    if valid == "1" :
        # We count all the validated sheets
        datasheets = Datasheet.objects.filter(organisation=orga).filter(validated=True)
        nb_sheets = len(datasheets)
        # And select the current one
        datasheets = [datasheets[start_index]]
    elif valid != "2":
        valid = "0"
        # We count all the validated sheets
        datasheets = Datasheet.objects.filter(organisation=orga).filter(validated=False)
        nb_sheets = len(datasheets)
        # And select the current one
        datasheets = [datasheets[start_index]]
    else :
        datasheets = Datasheet.objects.filter(organisation=orga)
        nb_sheets = len(datasheets)
    
    # We get the ORDERED tags if we display one sheet (case valid = 0 and 1)
    ordered_tags = None
    if valid != "2" :
        ordered_tags = TaggedSheet.objects.filter(datasheet=datasheets[0]).order_by('order')
    
    displayed_index = start_index + 1;
    prev_index = max(start_index - 1, 0);
    next_index = min(nb_sheets - 1, start_index + 1);
    last_index = max(nb_sheets - 1, 0);
    
    return render_to_response("list_for_orga.html",
                              {'datasheets':datasheets, 'orga_name':orga_name,
                               'nb_sheets':nb_sheets, 'orga_id':orga_id, 'ordered_tags':ordered_tags,
                               'prev_index':prev_index, 'next_index':next_index, 'last_index':last_index,
                               'start_index':start_index, 'displayed_index':displayed_index, 'valid':valid,
                               'categories':json.dumps(get_categories())},
                              context_instance=RequestContext(request))


#@login_required
def all_tags(request, num_page=None, nb_by_page=None):
    
    alltags = Tag.objects.order_by('label')
    nb_total = len(alltags)
    # We build the paginator for the requested list
    if nb_by_page :
        try:
            nb_by_page = int(nb_by_page)
        except :
            nb_by_page = 25
    else :
        nb_by_page = 25
    if num_page :
        try:
            num_page = int(num_page)
        except :
            num_page = 1
    else :
        num_page = 1
    p = Paginator(alltags, nb_by_page)
    current_page = p.page(num_page)
    
    prev_page = max(num_page - 1, 1)
    next_page = min(num_page + 1, p.num_pages)
    last_page = p.num_pages
    
    return render_to_response("all_tags.html",
                              {'nb_total':nb_total, 'tags':current_page.object_list, 'current_page':current_page,
                               'prev_page':prev_page, 'next_page':next_page, 'last_page':last_page,
                               'num_page':num_page, 'nb_by_page':nb_by_page, 'categories':json.dumps(get_categories())},
                              context_instance=RequestContext(request))
    

#@login_required
def tag_up_down(request):
    ds_id = request.POST["datasheet_id"]
    # post vars new_order and old_order indicate the position (from 1) of the tag in the list.
    # NB : it is different from the TagSheet.order in the database.
    new_order = int(request.POST["new_order"]) - 1
    old_order = int(request.POST["old_order"]) - 1
    # First we get the datasheet's TaggedSheets (list to force evaluation)
    ordered_tags = list(TaggedSheet.objects.filter(datasheet=Datasheet.objects.get(id=ds_id)).order_by('order'))
    # We change the moved TaggedSheets's order
    new_ts_order = ordered_tags[new_order].order
    moved_ts = ordered_tags[old_order]
    moved_ts.order = new_ts_order
    moved_ts.save()
    # We move the TaggedSheets's order
    if new_order > old_order :
        # And we decrease the other ones
        for i in range(old_order + 1, new_order + 1) :
            ts = ordered_tags[i]
            ts.order = ts.order - 1
            ts.save()
    else :
        # And we increase the other ones
        for i in range(new_order, old_order) :
            ts = ordered_tags[i]
            ts.order = ts.order + 1
            ts.save()
    ds = Datasheet.objects.get(id=ds_id)
    ds.manual_order = True
    ds.save()
    
    return get_tag_table(request=request, ds_id=ds_id, valid=0)


#@login_required
def get_tag_table(request=None, ds_id=None, valid=None):
    
    ordered_tags = TaggedSheet.objects.filter(datasheet=Datasheet.objects.filter(id=ds_id)[0]).order_by('order')
    
    return render_to_response("partial/tag_table.html",
                              {'ordered_tags':ordered_tags, 'valid':valid},
                              context_instance=RequestContext(request))


#@login_required
def get_all_tags_table(request, num_page=None, nb_by_page=None):
    
    alltags = Tag.objects.order_by('label')
    # We build the paginator for the requested list
    if nb_by_page :
        try:
            nb_by_page = int(nb_by_page)
        except :
            nb_by_page = 25
    else :
        nb_by_page = 25
    if num_page :
        try:
            num_page = int(num_page)
        except :
            num_page = 1
    else :
        num_page = 1
    p = Paginator(alltags, nb_by_page)
    current_page = p.page(num_page)
    
    return render_to_response("partial/all_tags_table.html",
                              {'tags':current_page.object_list},
                              context_instance=RequestContext(request))


#@login_required
def remove_tag_from_list(request=None):
    
    ds_id = request.POST["datasheet_id"]
    tag_id = request.POST["tag_id"]
    # First we get the datasheet's TaggedSheets
    ds_tags = TaggedSheet.objects.filter(datasheet=Datasheet.objects.filter(id=ds_id)[0])
    # We get the current TaggedSheet and we delete it
    ts = ds_tags.filter(tag=Tag.objects.filter(id=tag_id))[0]
    ts.delete()
    
    ds = Datasheet.objects.get(id=ds_id)
    ds.manual_order = True
    ds.save()

    return get_tag_table(request=request, ds_id=ds_id, valid=0)


#@login_required
def modify_tag(request):

    tag_id = request.POST["id"]
    tag_label = request.POST["value"]
        
    tag = Tag.objects.get(id=tag_id)
    
    if tag.label != tag_label:

        tag.label = tag_label
        
        site = wiki.Wiki(settings.WIKIPEDIA_API_URL) #@UndefinedVariable
        wp_res = query_wikipedia_title(site, tag_label)
        status, url, pageid, dbpedia_uri = (wp_res['status'], wp_res['wikipedia_url'], wp_res['pageid'], wp_res["dbpedia_uri"])

        if status is not None:
            tag.url_status = status

        tag.wikipedia_url = url
        tag.wikipedia_pageid = pageid            
        tag.dbpedia_uri = dbpedia_uri 
            
        tag.save()
    
    return get_all_tags_table(request=request, num_page=request.POST["num_page"], nb_by_page=request.POST["nb_by_page"])


#@login_required
def modify_tag_datasheet(request):

    tag_id = request.POST["id"]
    tag_label = request.POST["value"]
    ds_id = request.POST["datasheet_id"]
        
    tag = Tag.objects.get(id=tag_id)
    
    ds = Datasheet.objects.get(id=ds_id)
        
    if tag.label != tag_label:
        
        if tag_label.lower() in [t.label.lower() for t in ds.tags.all()]:
            return  HttpResponseBadRequest(json.dumps({'error': 'duplicate_tag', 'message': u"Le tag %s existe déjà pour cette fiche." % (tag_label)}), mimetype="application/json")

        tag, created = get_or_create_tag(tag_label) #@UnusedVariable

        ts = TaggedSheet.objects.get(tag=tag_id, datasheet=ds_id)
        ts.tag = tag
        kwargs = {DJANGO_ID + "__exact": unicode(ds_id)}
        results = SearchQuerySet().filter(title=tag_label).filter_or(description=tag_label).filter(**kwargs)
        if len(results) > 0:
            ts.index_note = results[0].score        

        ts.save()
        
        ds.manual_order = True
        ds.save()        
    
    return get_tag_table(request=request, ds_id=ds_id, valid=0)



#@login_required
def reset_wikipedia_info(request):
    # 2 cases : 
    # - ordered tag for one datasheet : POST["datasheet_id"] is not null
    # - all tags list : POST["datasheet_id"] is null and POST["num_page"] and POST["nb_by_page"] are not null
    tag_id = request.POST["tag_id"]
    
    tag = Tag.objects.get(id=tag_id)
    site = wiki.Wiki(settings.WIKIPEDIA_API_URL) #@UndefinedVariable
    
    tag.label = normalize_tag(tag.original_label)
    
    #reset tag info
    tag.wikipedia_url = None
    tag.wikipedia_pageid = None
    
    process_tag(site, tag, 0)
    
    if u"datasheet_id" in request.POST :
        return get_tag_table(request=request, ds_id=request.POST["datasheet_id"], valid=0)
    else :
        return get_all_tags_table(request=request, num_page=request.POST["num_page"], nb_by_page=request.POST["nb_by_page"])


#@login_required
def add_tag(request=None):
    
    ds_id = request.POST["datasheet_id"]
    tag_label = request.POST["value"]
    
    ds = Datasheet.objects.get(id=ds_id)
    if tag_label.lower() in [t.label.lower() for t in ds.tags.all()]:
        return  HttpResponseBadRequest(json.dumps({'error': 'duplicate_tag', 'message': u"Le tag %s existe déjà pour cette fiche." % (tag_label)}), mimetype="application/json")


    tag, created = get_or_create_tag(tag_label)    
    # We put the tag at the bottom of the datasheet's tag list 
    # if the tag is created or if the tag is not in the list
    
    list_ts = TaggedSheet.objects.filter(datasheet=ds)
    if created or len(list_ts.filter(tag=tag)) == 0 :
        new_order = list_ts.aggregate(Max('order'))['order__max'] + 1
        ts = TaggedSheet.objects.create(datasheet=ds, tag=tag, original_order=new_order, order=new_order)
        ts.save()
        ds.manual_order = True
        ds.save()
    
    return get_tag_table(request=request, ds_id=ds_id, valid=0)


#@login_required
def remove_wp_link(request=None):
    # 2 cases : 
    # - ordered tag for one datasheet : POST["datasheet_id"] is not null
    # - all tags list : POST["datasheet_id"] is null and POST["num_page"] and POST["nb_by_page"] are not null
    tag_id = request.POST["tag_id"]
    tag = Tag.objects.filter(id=tag_id)[0]
    tag.wikipedia_url = None
    tag.wikipedia_pageid = None
    tag.url_status = 0;
    tag.save()
    
    if u"datasheet_id" in request.POST :
        return get_tag_table(request=request, ds_id=request.POST["datasheet_id"], valid=0)
    else :
        return get_all_tags_table(request=request, num_page=request.POST["num_page"], nb_by_page=request.POST["nb_by_page"])


#@login_required
def validate_datasheet(request=None, ds_id=None, valid=None):
    # We set if valid is true of false, function of the url parameters
    if valid == "1" or valid == "true" or not valid :
        valid = True
    else :
        valid = False
    # We validate or unvalidate the requester datasheet
    
    if request.user.is_authenticated():
        user = request.user
    else:
        user = None
    
    ds = Datasheet.objects.get(id=ds_id)
    if valid :
        ds.validate(user)
    else :
        ds.unvalidate()
    ds.save()
    # If there are still some unvalidated/validated ds for the ds's orga, we display the first one
    same_organisation_ds = Datasheet.objects.filter(organisation=ds.organisation).filter(validated=not valid)
    if valid :
        # We ask to display the unvalidated ds
        valid_req = 0
    else :
        # We ask to display the validated ds
        valid_req = 1
    if len(same_organisation_ds) > 0 :
        return redirect('list_for_orga', orga_id=ds.organisation.id, valid=valid_req)
    else :
        return redirect('home')
    

#@login_required
def update_tag_alias(request):
    # 2 cases : 
    # - ordered tag for one datasheet : POST["datasheet_id"] is not null
    # - all tags list : POST["datasheet_id"] is null and POST["num_page"] and POST["nb_by_page"] are not null
    tag_id = request.POST["id"]
    tag_alias = request.POST["value"]
    tag = Tag.objects.get(id=tag_id)
    tag.alias = tag_alias
    tag.save()
    
    if u"datasheet_id" in request.POST :
        return get_tag_table(request=request, ds_id=request.POST["datasheet_id"], valid=0)
    else :
        return get_all_tags_table(request=request, num_page=request.POST["num_page"], nb_by_page=request.POST["nb_by_page"])
    

#@login_required
def get_categories():
    # List of categories in an OrderedDict
    categories = OrderedDict({"":""})
    for c in TagCategory.objects.order_by('label') :
        categories.update({c.label:c.label})
    return categories
    

#@login_required
def update_tag_category(request):
    
    tag_id = request.POST["id"]
    cat = request.POST["value"]
    tag = Tag.objects.get(id=tag_id)
    if cat and cat != "" :
        tag.category = TagCategory.objects.get(label=cat)
    else :
        tag.category = None
    tag.save()
    # This function is available only in all_tags_table context
    return get_all_tags_table(request=request, num_page=request.POST["num_page"], nb_by_page=request.POST["nb_by_page"])
    
        
