# -*- coding: utf-8 -*-

from django.contrib.auth.models import User
from django.db import models
from hdabo.utils import Property
import datetime

class Organisation(models.Model):
    hda_id = models.CharField(max_length=512, unique=True, blank=False, null=False)
    name = models.CharField(max_length=512, unique=False, blank=False, null=False)
    location = models.CharField(max_length=512, unique=False, blank=True, null=True)
    website = models.CharField(max_length=1024, unique=False, blank=True, null=True)
    

class Author(models.Model):
    hda_id = models.CharField(max_length=512, unique=True, blank=False, null=False)
    lastname = models.CharField(max_length=512, unique=False, blank=True, null=True)
    firstname = models.CharField(max_length=512, unique=False, blank=True, null=True)

class TimePeriod(models.Model):
    TIME_PERIOD_CHOICES = (
        (1,u'Primaire'),
        (2,u'Collège'),
        (3,u'Lycée'),
    )
    TIME_PERIOD_DICT = {
        u'Primaire': 1,
        u'Collège': 2,
        u'Lycée': 3,
    }
    label = models.CharField(max_length=512, unique=False, blank=False, null=False)
    school_period = models.IntegerField(choices=TIME_PERIOD_CHOICES)
    
    class Meta:
        unique_together = ("label", "school_period")

    def __unicode__(self):
        return unicode(self.label)


class Domain(models.Model):
    DOMAIN_PERIOD_CHOICES = (
        (0,u'Global'),
        (1,u'Primaire'),
        (2,u'Collège'),
        (3,u'Lycée'),
    )
    DOMAIN_PERIOD_DICT = {
        u'Global': 0,
        u'Primaire': 1,
        u'Collège': 2,
        u'Lycée': 3,
    }
    label = models.CharField(max_length=512, unique=False, blank=False, null=False)
    school_period = models.IntegerField(choices=DOMAIN_PERIOD_CHOICES)

    class Meta:
        unique_together = ("label", "school_period")

    def __unicode__(self):
        return unicode(self.label)


class DocumentFormat(models.Model):
    label = models.CharField(max_length=512, unique=True, blank=False, null=False)

    def __unicode__(self):
        return unicode(self.label)

class Tag(models.Model):
    label = models.CharField(max_length=1024, unique=True, blank=False, null=False)
    original_label = models.CharField(max_length=1024, unique=False, blank=True, null=True, editable=False)
    alias = models.CharField(max_length=1024, unique=False, blank=True, null=True)
    wikipedia_url = models.URLField(verify_exists=False, max_length=512, blank=True, null=True)
    dbpedia_uri = models.URLField(verify_exists=False, max_length=512, blank=True, null=True)
    wikipedia_activated = models.BooleanField(default=False)


class TagCategory(models.Model):
    label = models.CharField(max_length=512, unique=True, blank=False, null=False)
    
    def __unicode__(self):
        return unicode(self.label)
    
class Location(models.Model):
    name = models.CharField(max_length=512, unique=False, blank=False, null=False)
    insee = models.CharField(max_length=5, unique=True, blank=False, null=False)

    def __unicode__(self):
        return unicode("%s : %s"%(self.name, self.insee))

class Datasheet(models.Model):
    hda_id = models.CharField(max_length=512, unique=True, blank=False, null=False)
    author = models.ForeignKey(Author, null=True, blank=True)
    organisation = models.ForeignKey(Organisation)
    title = models.CharField(max_length=2048, unique=False, blank=False, null=False)
    description = models.TextField(blank=True, null=True)
    url = models.URLField(verify_exists=False, max_length=512, blank=True, null=True)
    domains = models.ManyToManyField(Domain, limit_choices_to={'school_period':Domain.DOMAIN_PERIOD_DICT[u'Global']}, related_name="datasheets")
    primary_periods = models.ManyToManyField(TimePeriod, limit_choices_to={'school_period':TimePeriod.TIME_PERIOD_DICT[u'Primaire']}, related_name="primary_periods_datasheets")
    college_periods = models.ManyToManyField(TimePeriod, limit_choices_to={'school_period':TimePeriod.TIME_PERIOD_DICT[u'Collège']}, related_name="college_periods_datasheets")
    highschool_periods = models.ManyToManyField(TimePeriod, limit_choices_to={'school_period':TimePeriod.TIME_PERIOD_DICT[u'Lycée']}, related_name="highschool_periods_datasheets")
    primary_themes = models.ManyToManyField(Domain, limit_choices_to={'school_period':Domain.DOMAIN_PERIOD_DICT[u'Primaire']}, related_name="primary_themes_datasheets")
    college_themes = models.ManyToManyField(Domain, limit_choices_to={'school_period':Domain.DOMAIN_PERIOD_DICT[u'Collège']}, related_name="college_themes_datasheets")
    highschool_themes = models.ManyToManyField(Domain, limit_choices_to={'school_period':Domain.DOMAIN_PERIOD_DICT[u'Lycée']}, related_name="highschool_themes_datasheets")
    town = models.ForeignKey(Location, null=True, blank=True)
    format = models.ForeignKey(DocumentFormat, null=True, blank=True)
    original_creation_date = models.DateField()
    original_modification_date = models.DateField()
    modification_datetime = models.DateTimeField(auto_now=True)
    validation_date = models.DateTimeField()
    validated = models.BooleanField()
    validator = models.ForeignKey(User)
    tags = models.ManyToManyField(Tag, through='TaggedSheet')
    
    
    def validate(self, user):
        self.validation_date = datetime.datetime.now()
        self.validated = True
        self.validator = user
        self.save()
    
    def unvalidate(self):
        self.validation_date = datetime.datetime.min
        self.validated = False
        self.validator = None
        self.save()

    @Property
    def domain_text(): #@NoSelf
        def fget(self):
            return "; ".join([d.label for d in self.domains.all()])
        
        return locals() 
    
    @Property
    def primary_periods_text(): #@NoSelf
        def fget(self):
            return "; ".join([d.label for d in self.primary_periods.all()]) 

        return locals() 


    @Property
    def college_periods_text(): #@NoSelf
        def fget(self):
            return "; ".join([d.label for d in self.college_periods.all()]) 

        return locals() 

    @Property
    def highschool_periods_text(): #@NoSelf
        def fget(self):
            return "; ".join([d.label for d in self.highschool_periods.all()]) 

        return locals() 


    @Property
    def primary_themes_text(): #@NoSelf
        def fget(self):
            return "; ".join([d.label for d in self.primary_themes.all()]) 

        return locals() 

    @Property
    def college_themes_text(): #@NoSelf
        def fget(self):
            return "; ".join([d.label for d in self.college_themes.all()]) 

        return locals() 

    @Property
    def highschool_themes_text(): #@NoSelf
        def fget(self):
            return "; ".join([d.label for d in self.highschool_themes.all()]) 

        return locals()

    @Property
    def town_text(): #@NoSelf
        def fget(self):
            return self.town.name if self.town else ""
        
        return locals()


class TaggedSheet(models.Model):
    datasheet = models.ForeignKey(Datasheet)
    tag = models.ForeignKey(Tag)
    original_order = models.IntegerField()
    order = models.IntegerField()
    index_note = models.FloatField()
    categories = models.ManyToManyField(TagCategory)
    