# -*- coding: utf-8 -*-
'''
Created on Jan 31, 2012

@author: ymh
'''
from django.conf import settings
from django.db import connection
from django.db.models import Q, Count, Sum
from django.http import HttpResponse
from hdabo.models import Tag, Datasheet, TaggedSheet
from hdalab.models import TagLinks, HdaSession, Country, GeoInclusion, TagYears
import django.utils.simplejson as json
import hmac
import uuid

def taginfo(request):
    label = request.GET.get('label', None)
    
    resobj = {'requested_label' : label}
        
    resobj["content_count"] = Datasheet.objects.filter(taggedsheet__tag__label__iexact = label).distinct().count()
    
    res = Tag.objects.select_related('dbpedia_fields').filter(label__iexact = label).order_by('-dbpedia_uri')[0:1]
    if len(res) == 1:
        restag = res.get()
        resobj["dbpedia_uri"] = restag.dbpedia_uri
        if resobj["dbpedia_uri"] is not None and restag.dbpedia_fields is not None:
            dbfield = restag.dbpedia_fields
            resobj["abstract"] = dbfield.abstract
            resobj["dbpedia_label"] = dbfield.label
            resobj["thumbnail"] = dbfield.thumbnail
    res = Tag.objects.filter(label__iexact = label).order_by('-wikipedia_url')[0:1]
    if len(res) == 1:
        resobj["wikipedia_url"] = res.get().wikipedia_url
        
    resobj["links"] = [{'subject':tl.subject.label, 'object':tl.object.label} for tl in TagLinks.objects.select_related().filter(Q(subject__label__iexact = label) | Q(object__label__iexact = label))]

    return HttpResponse(content=json.dumps(resobj), mimetype='application/json')


def sessioninfo(request):
    
    data = json.loads(request.GET.get('data', "{}"))
    write = False
    
    if 'sessionid' in request.GET:
        request.session['sessionid'] = request.GET['sessionid']
    if 'sessionkey' in request.GET:
        request.session['sessionkey'] = request.GET['sessionkey']
        
    if 'sessionid' in request.session:
        sessionid = request.session['sessionid']
        
        if HdaSession.objects.filter(sessionid=sessionid).count() == 1:
            sessionkey = request.session.get('sessionkey',None)
            hm = hmac.new(settings.SECRET_KEY, sessionid)
            if hm.hexdigest() == sessionkey:
                write = True            
        else:
            del request.session['sessionid']
        
    if 'sessionid' not in request.session:
        sessionid = unicode(uuid.uuid1())
        HdaSession.objects.create(sessionid=sessionid, data=json.dumps({}))
        write = True
        request.session['sessionid'] = sessionid
        request.session['sessionkey'] = hmac.new(settings.SECRET_KEY, sessionid).hexdigest()
        
    if write and data:
        HdaSession.objects.filter(sessionid=sessionid).update(data=json.dumps(data))
    else:
        data = HdaSession.objects.get(sessionid=sessionid).data
        data = json.loads(data) if data else {}           
         
    resobj = {'data': data, "write_allowed" : write, "sessionid": sessionid }
    if write:
        resobj['sessionkey'] = request.session['sessionkey']
        
    return HttpResponse(content=json.dumps(resobj), mimetype='application/json')


def tagsearch(request):
    
    q = request.GET.get('term',None)
    qs = Tag.objects.filter(label__icontains = q) if q else Tag.objects.all()        
    res = qs.annotate(nb=Count('datasheet')).order_by('-nb')[:20]
    
    return HttpResponse(content=json.dumps([{'label':t.label,'nb':t.nb} for t in res]), mimetype='application/json')


def filter(request):
    periode = request.GET.get('period',None)
    label = request.GET.get('label', None)
    contentlist = request.GET.get('contentlist', None)
    max_tag_order = request.GET.get('mto', 12)
    content_count = request.GET.get('contentcount', 12)
    tag_count = request.GET.get('tagcount', 30)
    cont_count = 0;
    contenus = {}


    if label or periode or contentlist :

        qs = Datasheet.objects.filter(validated=True)

        globalids = []
        
        if periode:
            years = periode.split(",")
            start_year = int(years[0])
            end_year = int(years[0:2][-1])
            delta = max(1, (end_year-start_year)/2)
            minstart = start_year - delta;
            maxend = end_year + delta;
            
            tagqs = Tag.objects.filter(years__end_year__gte = start_year, 
                                       years__start_year__lte = end_year,
                                       years__end_year__lte = maxend,
                                       years__start_year__gte = minstart)
            
            globalids += [t.id for t in tagqs]
            
            qs = qs.filter(taggedsheet__tag__in = tagqs)
            
        if label:
            for txtlbl in label.split(","):
                taglblqs = Tag.objects.filter(label__iexact = txtlbl)
                globalids += [t.id for t in taglblqs if t.id not in globalids]
                qs = qs.filter(taggedsheet__tag__in = taglblqs)

        if label or periode:
            qs = qs.filter(taggedsheet__order__lte = max_tag_order)

        if contentlist:
            qs = qs.filter(id__in = contentlist.split(","))
        
#        if contentlist is None:
#            qs = qs.order_by('?')
        
        qs = qs.distinct()
        
        contenus = dict([(content.id, {'score' : 0, 'tags' : [], 'id':content.id, 'title': content.title, 'description': content.description, 'url': content.url}) for content in qs])

        #countries = dict([ (cc.label, {'isocode':cc.isocode, 'score':0}) for cc in CountryCode.objects.all() ])
        #disciplines = dict([ (d.label, {'label':d.label, 'score':0}) for d in Tag.objects.filter(category__label = "Discipline artistique")])
        
        countries = {}
        disciplines = {}
        
        tags = {}
        
        contentids = contenus.keys()
        cont_count = len(contentids)
        
        qs = TaggedSheet.objects.select_related('tag','tag__category','tag__locatedin__country').filter(datasheet__in = contentids, order__lte = max_tag_order).order_by('order')
        for ts in qs:
            match_tag = ts.tag.id in globalids
            contenutags = {'id': ts.tag.id, 'label':ts.tag.label, 'order':ts.order, 'match': match_tag }
            
#            if ts.tag.category is not None:
#                contenutags['category'] = ts.tag.category.label
            
            contenus[ts.datasheet.id]['tags'].append(contenutags)
                
            tagscore = 2*max_tag_order - ts.order
            if ts.tag.category is not None and ts.tag.category.label not in [u'Discipline artistique', u'Datation', u'Localisation']:
                if ts.tag.id not in tags:
                    tags[ts.tag.id] = {'id':ts.tag.id,'label':ts.tag.label, 'score':0}
                tags[ts.tag.id]['score'] += 1
                if match_tag:
                    tags[ts.tag.id]['match'] = True
                
            if match_tag:
                contenus[ts.datasheet.id]['score'] += tagscore
                
            if ts.tag.category is not None and ts.tag.category.label == u'Discipline artistique':
                if ts.tag.label not in disciplines:
                    disciplines[ts.tag.label] = {'label':ts.tag.label, 'score':0}
                disciplines[ts.tag.label]['score'] += 1
                
            if ts.tag.locatedin is not None:
                country_id = ts.tag.locatedin.country.id
                if country_id not in countries:
                    countries[country_id] = {'id':country_id, 'dbpedia_uri':ts.tag.locatedin.country.dbpedia_uri, 'score': 0}
                countries[country_id]['score'] += 1
            
        if contentlist is None:
            contenus = sorted(contenus.values(),key=lambda e: -e['score'])[0:content_count]
            contenus = [contenu for contenu in contenus if contenu['score']]
            
        countries = countries.values()
        tags = sorted(tags.values(), key=lambda e: -e['score'])[0:tag_count]
        disciplines = sorted(disciplines.values(), key=lambda e: -e['score'])[0:10]
        years = {}
        
        if contentids:
            qs = TagYears.objects.filter(tag__taggedsheet__datasheet__in = contentids).annotate(nb=Count('tag__taggedsheet'))
            for ty in qs:
                for year in range(ty.start_year, ty.end_year):
                    years[year] = ty.nb + years[year] if year in years else 0

    else:
        
        for ds in Datasheet.objects.order_by("?")[:content_count]:
            contenus[ds.id] = {'id':ds.id, 'title':ds.title, 'description':ds.description, 'url':ds.url, 'tags':[]}
            
        cont_count = Datasheet.objects.count()
        
        qs = TaggedSheet.objects.select_related('tag','datasheet').filter(datasheet__id__in = contenus.keys(), order__lte = max_tag_order).order_by("order").only('order','tag__label','tag__id','datasheet__id')
        for ts in qs:
            contenus[ts.datasheet.id]['tags'].append({'id': ts.tag.id, 'label':ts.tag.label, 'order':ts.order})
        contenus = contenus.values()
        
        qs = Tag.objects.exclude(category__label = u"Localisation").exclude(category__label = u"Datation").exclude(category__label = u"Discipline artistice").annotate(nb=Count('datasheet')).order_by('-nb')[:tag_count]
        tags = [{'id': tag.id, 'label': tag.label, 'score': tag.nb} for tag in qs]
            
        qs = Country.objects.annotate(nb=Count('includes__tag__taggedsheet')).order_by('-nb')
        countries = [{'id': country.id, 'dbpedia_uri': country.dbpedia_uri, 'score': country.nb} for country in qs]

        qs = Tag.objects.annotate(nb=Count('taggedsheet')).filter(category = 5).order_by('-nb')[:10]
        disciplines = [{'label':tag.label,'score':tag.nb} for tag in qs]

        years = {}
        qs = TagYears.objects.annotate(nb=Count('tag__taggedsheet'))
        for ty in qs:
            for year in range(ty.start_year, ty.end_year):
                years[year] = ty.nb + years[year] if year in years else 0

    yearchange = []
    for year in sorted(years.keys()):
        score = years[year]
        if year < 2011:
            if (year-1 not in years and score != 0) or (year-1 in years and years[year-1] != score):
                yearchange.append({'year': year, 'score': score})
            if year+1 not in years and year != -1 and score != 0:
                yearchange.append({'year': year+1, 'score': 0})

    output = {'count': cont_count, 'contents': contenus, 'tags':tags, 'sparkline':yearchange, 'countries':countries, 'disciplines':disciplines}
    
    return HttpResponse(content=json.dumps(output), mimetype='application/json')
