# -*- coding: utf-8 -*-
'''
Created on Jul 17, 2014

@author: tc
'''
import logging
from renkanmanager.models import Renkan

from django.conf import settings
from django.db import models
from django.utils.translation import ugettext_lazy as _


logger = logging.getLogger(__name__)

class HdalabRenkan(models.Model):
    """
    Classe representant un Renkan HDALab.
    Est utilisée pour ajouter au projet renkan des informations sur l'état d'édition ou bien si il est marqué comme favori.

    :cvar object renkan: Le renkan lui même.
    :cvar int state: État du renkan (1: en édition, 2: en modération, 3: publié, 4: rejeté).
    :cvar bool favorite: Le renkan est marqué comme favori ou pas.
    """

    EDITION = 1
    MODERATED = 2
    PUBLISHED = 3
    REJECTED = 4

    STATE_CHOICES = (
    (EDITION, 'edition'),
    (MODERATED, 'moderated'),
    (PUBLISHED, 'published'),
    (REJECTED, 'rejected'),
    )

    STATE_CHOICES_DICT = {
        EDITION: _('edition_state'),
        MODERATED: _('moderated_state'),
        PUBLISHED: _('published_state'),
        REJECTED: _('rejected_state')
    }

    renkan = models.ForeignKey(Renkan, blank=False, null=False)
    state = models.IntegerField(choices=STATE_CHOICES, default=1)
    favorite = models.BooleanField(null=False, blank=False, default=False)

    class Meta:
        app_label = 'hdalab'


class HdalabRenkanStateTransition(models.Model):
    """
    Modélise un changement de statut du renkan.
    Permet de garder l'historique des changements de statut.

    :cvar object renkan: Le renkan.
    :cvar int from_state: le statut initial.
    :cvar int to_state: le statut final.
    :cvar datetime ts: date/heure du changement.
    :cvar str message: Message (optionel) associé au changement de statut.
    :cvar object author: Utilisateur ayant initié le changement de statut.
    """

    renkan = models.ForeignKey(HdalabRenkan, blank=False, null=False, related_name='states')
    from_state = models.IntegerField(choices=HdalabRenkan.STATE_CHOICES, default=1, blank=False, null=False)
    to_state = models.IntegerField(choices=HdalabRenkan.STATE_CHOICES, default=1, blank=False, null=False)
    ts = models.DateTimeField(auto_now_add=True, blank=False, null=False)
    message = models.TextField(blank=True, null=True)
    author = models.ForeignKey(settings.AUTH_USER_MODEL, blank=False, null=False)

    class Meta:
        app_label = 'hdalab'
        ordering = ["-ts"]
