from django.core.exceptions import ImproperlyConfigured, ObjectDoesNotExist
from django.http import HttpResponseNotFound
from django.views.generic.base import View, TemplateResponseMixin
from ldt.api.ldt.resources import ProjectResource
from ldt.ldt_utils.models import Project
import logging
from hashcut.models import Mashup, Branding


class MashupHome(TemplateResponseMixin, View):
    
    # iri = default brand name
    branding = "iri"
    template_suffix = "mashup_home.html"
    template_name = "iri_mashup_home.html"
    
    def get_template_names(self):
        """
        Return a list of template names to be used for the request. Must return
        a list. May not be called if get_template is overridden.
        """
        try:
            names = super(MashupHome, self).get_template_names()
        except ImproperlyConfigured:
            raise ImproperlyConfigured("Class MashupHome requires either a definition of 'template_name'")

        # the branding template is supposed to override the default template. So we insert instead of append
        if self.branding and self.branding != "":
            names.insert(0,"%s_%s" % (self.branding, self.template_suffix))
        
        return names
    
    
    def get(self, request, branding=None, **kwargs):
        if branding and branding!="":
            self.branding = branding
        projects = Project.safe_objects.all()[:10]
        p1 = projects[0]
        p2 = projects[1]
        projects = projects[2:]
        context = {"branding":self.branding, "projects":projects, "p1":p1, "p2":p2}
        return self.render_to_response(context)
    


class MashupEdit(TemplateResponseMixin, View):
    
    # iri = default brand name
    branding = "iri"
    template_suffix = "mashup_edit.html"
    template_name = "iri_mashup_edit.html"
    
    def get_template_names(self):
        """
        Return a list of template names to be used for the request. Must return
        a list. May not be called if get_template is overridden.
        """
        try:
            names = super(MashupEdit, self).get_template_names()
        except ImproperlyConfigured:
            raise ImproperlyConfigured("Class MashupEdit requires either a definition of 'template_name'")

        # the branding template is supposed to override the default template. So we insert instead of append
        if self.branding and self.branding != "":
            names.insert(0,"%s_%s" % (self.branding, self.template_suffix))
        
        return names
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        context = {"branding":self.branding, "creator":request.user.username}
        return self.render_to_response(context)


def save_mashup(request, branding="iri"):
    # First we use the project api to save the project
    logging.debug("save_mashup 1")
    proj_api = ProjectResource()
    logging.debug("save_mashup 2 " + repr(proj_api))
    rtrn = proj_api.dispatch_list(request)
    logging.debug("save_mashup 3 " + repr(rtrn))
    if rtrn and rtrn['Location']:
        logging.debug("save_mashup 4 " + rtrn['Location'])
        # The api return the api location of the created project (i.e. /a/b/ldt_id/). So we just get the id to get the project object.
        proj_ldt_id = rtrn['Location'].split("/")[-2]
        logging.debug("save_mashup 5 proj_ldt_id = " + proj_ldt_id)
        proj = None
        try:
            proj = Project.objects.get(ldt_id=proj_ldt_id)
        except Project.DoesNotExist:
            raise ObjectDoesNotExist("Save Mashup : project not found. project_ldt_id = " + proj_ldt_id)
        # Now that we have the project object, we can save the mashup object
        brd = Branding.objects.get(name=branding)
        logging.debug("save_mashup 6" + repr(brd))
        new_mashup = Mashup()
        new_mashup.branding = brd
        new_mashup.project = proj
        logging.debug("save_mashup 7 new_mashup = " + repr(new_mashup))
        new_mashup.save()
        logging.debug("save_mashup 8")
    return rtrn
    


class MashupHashcut(TemplateResponseMixin, View):
    
    # iri = default brand name
    branding = "iri"
    template_suffix = "mashup_hashcut.html"
    template_name = "iri_mashup_hashcut.html"
    
    def get_template_names(self):
        """
        Return a list of template names to be used for the request. Must return
        a list. May not be called if get_template is overridden.
        """
        try:
            names = super(MashupHashcut, self).get_template_names()
        except ImproperlyConfigured:
            raise ImproperlyConfigured("Class MashupHashcut requires either a definition of 'template_name'")

        # the branding template is supposed to override the default template. So we insert instead of append
        if self.branding and self.branding != "":
            names.insert(0,"%s_%s" % (self.branding, self.template_suffix))
        
        return names
    
    def get(self, request, branding="iri", ldt_id=None, **kwargs):
        self.branding = branding
        if not ldt_id:
            return HttpResponseNotFound("A project id must be given.")
        context = {"branding":self.branding, "ldt_id":ldt_id}
        return self.render_to_response(context)
    


class MashupContent(TemplateResponseMixin, View):
    
    # iri = default brand name
    branding = "iri"
    template_suffix = "mashup_content.html"
    template_name = "iri_mashup_content.html"
    
    def get_template_names(self):
        """
        Return a list of template names to be used for the request. Must return
        a list. May not be called if get_template is overridden.
        """
        try:
            names = super(MashupContent, self).get_template_names()
        except ImproperlyConfigured:
            raise ImproperlyConfigured("Class MashupContent requires either a definition of 'template_name'")

        # the branding template is supposed to override the default template. So we insert instead of append
        if self.branding and self.branding != "":
            names.insert(0,"%s_%s" % (self.branding, self.template_suffix))
        
        return names
    
    def get(self, request, branding="iri", ctt_id=None, **kwargs):
        self.branding = branding
        if not ctt_id:
            return HttpResponseNotFound("A content id must be given.")
        context = {"branding":self.branding, "ctt_id":ctt_id}
        return self.render_to_response(context)
    
    
    
    
    
    
    