# -*- coding: utf-8 -*-
'''
Created on Feb 20, 2013

@author: ymh
'''
from fabric.api import run, local, env, cd, put, prefix, sudo, lcd
from fabric.colors import green
from fabric.context_managers import settings
from fabric.contrib.files import exists, upload_template
from fabric.contrib.project import rsync_project
from fabric.tasks import Task
import imp
import os.path
import re
import shutil
import sys
import urlparse


__all__ = ["check_folder_access", "syncdb", "collectstatic", "do_relaunch_server",
    "export_version", "do_sync_web", "create_config", "clean_export_folder", 
    "sync_install_build", "do_create_virtualenv", "clean_rsync_folder", "rsync_export",    
    "do_sync_comp", "get_comp_versions_dict", "SyncComp"] 

def get_export_path(version):
    base_path = os.path.join(env.base_export_path,env.export_prefix).rstrip("/")
    return os.path.expanduser(base_path) + "_%s" % (str(version))

def clean_export_folder(path):
    print("Removing %s" % path)
    if os.path.isdir(path):
        shutil.rmtree(path, ignore_errors=True)

def do_export_version(path, **export_keys):
    print("Export version %s : %s " % (path,repr(export_keys)))

    for export_key, version in export_keys.items():
        export_path = os.path.join(path,export_key)
        
        repo_url = env.repos[export_key]['repo']
        url_part = urlparse.urlparse(repo_url)
        if url_part.scheme or url_part.netloc:
            # this is a remote repo. Let's clone first
            clone_path = os.path.join(path,'clone',export_keys)
            os.makedirs(clone_path)
            
            scm = "hg"            
            with settings(warn_only=True):
                output = local('git ls-remote \"%s\"' % repo_url)
                scm = "git" if output.failed else "hg" 
            if scm == "hg":
                output = local("hg clone \"%s\" \"%s\"" % (repo_url,clone_path))
            else:
                local("git clone \"%s\" \"%s\"" % (repo_url,clone_path))            
        else:
            clone_path = repo_url
        
        with lcd(clone_path):
            # detetct .git or .hg subfolder
            if os.path.exists(os.path.join(clone_path,".git")):
                os.makedirs(export_path)
                cmd_str = "git archive \'%s\' | tar -x -C \"%s\""
            else:
                cmd_str = "hg archive -r \'%s\' \"%s\""
            local(cmd_str % (str(version),export_path))
    
    print("Export version %s done"%repr(export_keys))

    
def get_remote_env(remotepath, remotevirtualenvpath, platform_web_module, settings_key):
    activate_path = os.path.join(remotevirtualenvpath, "bin/activate")
    with prefix("source \"%s\"" % activate_path), prefix("export PYTHONPATH=\"%s\"" % remotepath), cd(remotepath):
        tempfilepath = run("mktemp -t fablib.XXXXXX")
        with settings(warn_only=True):
            run("echo \"import os\" > %s" % (tempfilepath))
            map(lambda str_echo: run("echo \"%s\" >> %s" % (str_echo, tempfilepath)),
                ["os.environ.setdefault('DJANGO_SETTINGS_MODULE', '%s.settings')" % (platform_web_module),
                 "from django.conf import settings",
                 "print settings.%s" % (settings_key)])
            res = run("python < %s" % (tempfilepath))
        run("rm -f \"%s\"" % (tempfilepath))
    return res
    
    
    
def rsync_export(path, remotepath, filters):
    print("Rsync %s to %s",(path,remotepath))
    
    filter_option_str = "--progress --stats"
    if filters:
        filter_option_str += " " + " ".join(["--filter \"%s\"" % (f) for f in filters])
    
    run("mkdir -p \"%s\"" % remotepath)
    rsync_project(remotepath, local_dir=path, extra_opts=filter_option_str, delete=True)
    print("Rsync %s to %s done",(path,remotepath))
    
def clean_rsync_folder(remotepath):
    print("clean rsync folder %s" % remotepath)
    run("rm -fr \"%s\"" % remotepath)
    
def build_src(path):
    print("Build source dist at %s" % path)
    f = None
    sys.path.append(path)
    current_path = os.getcwdu()
    try:    
        os.chdir(path)    
        try:    
            f, pathname, description = imp.find_module("setup", [path])
            print("Build source dist at %s : found setup" % path)
            setup_mod = imp.load_module("setup", f, pathname, description)
            print("Build source dist at %s : setup loaded" % path)
        finally:
            if f:
                f.close()
            
            setup_mod.launch_setup("setup.py", ['sdist'])
    finally:
        os.chdir(current_path)
    print("Build source dist at %s done" % path)

def get_version(version):
    version_str = '%s.%s' % (version[0], version[1])
    if version[2]:
        version_str = '%s.%s' % (version_str, version[2])
    if version[3:] == ('alpha', 0):
        version_str = '%s pre-alpha' % version_str
    else:
        if version[3] != 'final':
            version_str = '%s %s %s' % (version_str, version[3], version[4])
    return version_str
    

def get_src_version(key, path):
        
    print("get src version for %s at %s" % (key,path))
    
    mod_name = env.repos[key].get('module', key) or key
    
    f = None
    sys.path.append(path)
    current_path = os.getcwdu()
    os.chdir(path)
    try:
        f, pathname, description = imp.find_module(mod_name, [path])
        src_mod = imp.load_module(mod_name, f, pathname, description)
    except:
        src_mod = None
        print("Could not import module, trying to parse")        
    finally:
        os.chdir(current_path)
        if f:
            f.close()
    version = None
    if src_mod is None:
        with open(os.path.join(path,mod_name,"__init__.py"),'r') as init_file:
            for line in init_file:
                m = re.search('VERSION\s+=\s+\((.+)\)', line, re.I)
                if m:                    
                    version = tuple([re.sub('[\s\"\']','', item) for item in m.group(1).split(',')])
                    break
    elif hasattr(src_mod, "VERSION"):
        version = src_mod.VERSION
    elif hasattr(src_mod, "__version__"):
        version = src_mod.__version__

    print("VERSION : %s" % repr(version))
     
    if version is None:
        version = ""

    if not isinstance(version, basestring):
        if src_mod and hasattr(src_mod, "get_version"):
            version_str = src_mod.get_version()
        elif isinstance(version, tuple):
            version_str = get_version(version)
        else:
            version_str = str(version)
    else:
        version_str = version
        
    print("VERSION str : %s" % repr(version_str))
    return (version, version_str) 
    

def sync_build(path):
    print("Sync build %s" % path)
    with cd(env.remote_path['build_export']):
        filename = os.path.basename(path)
        res_trans = put(path, os.path.join(env.remote_path['build_export'], filename))
        print("Sync build %s to %s" % (path,repr(res_trans)))
        return res_trans

def remove_build(path):
    print("remove build build %s" % path)
    run("rm \"%s\"" % path)
        

def install_build(remotepath, remotevirtualenvpath, module_to_uninstall= None):
    print("Install build %s in %s" % (remotepath, remotevirtualenvpath))
    activate_path = os.path.join(remotevirtualenvpath, "bin/activate")
    
    with prefix("source %s" % activate_path):
        if module_to_uninstall:
            with settings(warn_only=True):
                run("pip uninstall -y %s" % module_to_uninstall)
        run("pip install \"%s\"" % remotepath)

def collectstatic(remotepath, remotevirtualenvpath, platform_web_module):
    print("Collect static in %s with %s" % (remotepath, remotevirtualenvpath))
    remotestaticsitepath = get_remote_env(remotepath, remotevirtualenvpath, platform_web_module, "STATIC_ROOT")
    activate_path = os.path.join(remotevirtualenvpath, "bin/activate")
    with prefix("source \"%s\"" % activate_path), prefix("export PYTHONPATH=\"%s\"" % remotepath), cd(remotepath):
        #remocve old files optio -c of collect static fail !
        run("rm -fr \"%s\"" % (remotestaticsitepath))
        run("python manage.py collectstatic --noinput")

def syncdb(remotepath, remotevirtualenvpath):
    activate_path = os.path.join(remotevirtualenvpath, "bin/activate")
    with prefix("source \"%s\"" % activate_path), prefix("export PYTHONPATH=\"%s\"" % remotepath), cd(remotepath):
        run("python manage.py syncdb --migrate --noinput")
        
def create_config(export_path):    
    print("Create config from %s" % (export_path,))
    remotepath = env.remote_path['src']
    remote_config_path = os.path.join(remotepath, env.platform_web_module, "config.py")
    template_path = os.path.join(export_path, "src", env.platform_web_module, "config.py.tmpl")
    
    context = {
        'base_dir': os.path.join(remotepath, env.platform_web_module).rstrip("/")+"/",
        'asctime': '%(asctime)s',
        'levelname': '%(levelname)s',
        'message': '%(message)s',
        'module': '%(module)s',
    }
    context.update(env.config['web'])
    
    if not exists(remote_config_path, verbose=True):
        upload_template(template_path, remote_config_path, context=context)

def export_version(**kwargs):
    print("export version %s" % (repr(kwargs)))
    
    export_path = kwargs.get('path', None)
    
    if not export_path:    
        export_path = get_export_path("_".join(["%s_%s" % (k,v) for k,v in kwargs.items()]))
    
    clean_export_folder(export_path)
    
    do_export_version(export_path,**kwargs)
    
    return export_path

def do_create_virtualenv(remote_venv_export_path, remotevirtualenvpath):
    print("Create virtualenv export_path : %s - remote venvpath : %s" % (remote_venv_export_path, remotevirtualenvpath))
    activate_path = os.path.join(remotevirtualenvpath, "bin/activate")
    if "remote_baseline_venv" in env and env.remote_baseline_venv:
        prefix_str = "source \"%s\"" % os.path.join(env.remote_baseline_venv, "bin/activate")
    else:
        prefix_str = "echo"
    with settings(warn_only=True):
        run("rm -fr \"%s\"" % remotevirtualenvpath)
    run("mkdir -p \"%s\"" % remotevirtualenvpath)
    with prefix(prefix_str), cd(os.path.join(remote_venv_export_path,"virtualenv","web")):
        run("python create_python_env.py")
        run("python project-boot.py \"%s\"" % remotevirtualenvpath)
    with prefix("source \"%s\"" % activate_path):
        run("pip install -r \"%s\"" % os.path.join(remote_venv_export_path,"virtualenv","web","res","srvr_requirements.txt"))

def do_sync_comp(key, export_path):
    print("do_sync_comp with  path %s" % (export_path))
    
    src_path = os.path.join(export_path,env.repos[key]['src_root'])
    # find setup.py
    for root, _, files in os.walk(src_path):
        if "setup.py" in files:
            src_path = root
    
    build_src(src_path)
    (_,version_str) = get_src_version(key, src_path)
    build_path = os.path.join(src_path,"dist","%s-%s.tar.gz" % (key,version_str))
    sync_install_build(build_path, key)
        

def sync_install_build(build_path, module_to_uninstall=None):
    res_trans = None
    try:
        res_trans = sync_build(build_path)
        install_build(res_trans[0], env.remote_path['virtualenv'], module_to_uninstall)        
    finally:
        if res_trans:
            remove_build(res_trans[0])


def do_sync_web(version, export_path):
    print("do_sync_web with version %s and path %s" % (version,export_path))
    #sync src
    src_path = os.path.join(export_path,"src/") 
    rsync_export(src_path, env.remote_path['src'], env.rsync_filters['src'])
        
    
def check_folder_access():
    print("Check folder access")
    # get remote user
    for folder_path in env.folders:
        if not os.path.isabs(folder_path):
            folder_path = env.remote_path['web'].rstrip("/")+ "/" + folder_path
            with settings(warn_only=True):
                if not exists(folder_path):
                    run("mkdir -p \"%s\"" % folder_path)
                run("chown -R :%s \"%s\"" % (env.web_group, folder_path))
                run("chmod -R g+w \"%s\"" % folder_path)

def get_comp_versions_dict(export_path_web):
    comp_versions = {}
    requirement_file_path = os.path.join(export_path_web, 'src', 'requirement.txt')
    if not os.path.exists(requirement_file_path):
        return comp_versions
    with open(requirement_file_path) as f:
        for line in f:
            m = re.match('^([\w-]+)\s+\(\s*\=\=\s*([\.\d\w]+)\s*\)', line)
            if m:
                key, version_req = m.groups()
                if "." in version_req:
                    version_req = "V" + ".".join(["%02d" % (int(s)) if s.isdigit() else s for s in version_req.split(".")])
                comp_versions[key] = version_req
    
    return comp_versions

def do_relaunch_server(do_collectstatic, do_syncdb):
    check_folder_access()
    if do_syncdb:
        syncdb(env.remote_path['src'], env.remote_path['virtualenv'])
    if do_collectstatic:
        collectstatic(env.remote_path['src'], env.remote_path['virtualenv'], env.platform_web_module)
    sudo(env.web_relaunch_cmd, shell=False)


class SyncComp(Task):
    
    def __init__(self, key):
        self.key = key

    def __get_name(self):
        return "sync_" + self.key
    
    name = property(__get_name)
    
    def run(self, version):
        print(green("sync %s with version %s" % (self.key, version)))
        export_path_web = export_version(web=version)    
        export_path_web_full = os.path.join(export_path_web,'web')
        comp_versions = get_comp_versions_dict(export_path_web_full)

        export_path = export_version(**{self.key:comp_versions[self.key]})
        export_path_full = os.path.join(export_path,self.key)
        do_sync_comp(self.key, export_path_full)
        clean_export_folder(export_path)
        clean_export_folder(export_path_web)
        
        do_relaunch_server(do_collectstatic=True, do_syncdb=True)
