<?php

use CorpusParole\Libraries\Filters\CorpusFilterManager;

class CorpusFilterManagerTest extends TestCase
{
    /**
     * test prepare languages noop
     *
     * @return void
     */
    public function testPrepareLanguagesNoOp()
    {
        $languagesInput = ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/oci'];
        $filterManager = new CorpusFilterManager();
        $languagesOutput = $filterManager->prepareLanguages($languagesInput);
        $this->assertEquals($languagesInput, $languagesOutput);
    }

    /**
     * test prepare languages unknown
     *
     * @return void
     */
    public function testPrepareLanguagesUnkown()
    {
        $languagesInput = ['foobar'];
        $filterManager = new CorpusFilterManager();
        $languagesOutput = $filterManager->prepareLanguages($languagesInput);
        $this->assertEquals(['http://lexvo.org/id/iso639-3/foobar'], $languagesOutput);
    }

    /**
     * test prepare languages addPrefix
     *
     * @return void
     */
    public function testPrepareLanguagesAddPrefix()
    {
        $languagesInput = ['fra', 'gsw', 'bre', 'oci'];
        $filterManager = new CorpusFilterManager();
        $languagesOutput = $filterManager->prepareLanguages($languagesInput);
        $this->assertEquals(['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/oci'], $languagesOutput);
    }

    /**
     * test prepare languages mix
     *
     * @return void
     */
    public function testPrepareLanguagesMix()
    {
        $languagesInput = ['fra', 'http://lexvo.org/id/iso639-3/gsw', 'bre', 'http://lexvo.org/id/iso639-3/oci'];
        $filterManager = new CorpusFilterManager();
        $languagesOutput = $filterManager->prepareLanguages($languagesInput);
        $this->assertEquals(['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/oci'], $languagesOutput);
    }

    /**
     * test prepare languages recursion simple
     *
     * @return void
     */
    public function testPrepareLanguagesRecusionSimple()
    {
        $languagesInput = ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'corpus-oil'];
        $filterManager = new CorpusFilterManager();
        $languagesOutput = $filterManager->prepareLanguages($languagesInput);
        $this->assertEquals(['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/pcd'], $languagesOutput);
    }

    /**
     * test prepare languages recursion simple
     *
     * @return void
     */
    public function testPrepareLanguagesRecusionComplex()
    {
        $languagesInput = ['fra', 'http://lexvo.org/id/iso639-3/gsw', 'corpus-oil', 'corpus-regionals'];
        $filterManager = new CorpusFilterManager();
        $languagesOutput = $filterManager->prepareLanguages($languagesInput);
        sort($languagesOutput);
        $languagesExpected = ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/pcd',
             'http://lexvo.org/id/iso639-3/frp', 'http://lexvo.org/id/iso639-3/cos', 'http://lexvo.org/id/iso639-3/rcf',
             'http://lexvo.org/id/iso639-3/gcf'];
        sort($languagesExpected);
        $this->assertEquals($languagesExpected, $languagesOutput);
    }

    /**
     * test prepare location no operation
     *
     * @return void
     */
    public function testPrepareLocationNoOp()
    {
        $locationInput = '3030293';
        $filterManager = new CorpusFilterManager();
        $locationOutput = $filterManager->prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }

    /**
     * test prepare location array
     *
     * @return void
     */
    public function testPrepareLocationArray()
    {
        $locationInput = ['3030293', 'foobar'];
        $filterManager = new CorpusFilterManager();
        $locationOutput = $filterManager->prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }


    /**
     * test prepare location unknown
     *
     * @return void
     */
    public function testPrepareLocationUnknown()
    {
        $locationInput = 'foobar';
        $filterManager = new CorpusFilterManager();
        $locationOutput = $filterManager->prepareLocation($locationInput);
        $this->assertEquals('foobar', $locationOutput);
    }

    /**
     * test prepare location geonames
     *
     * @return void
     */
    public function testPrepareLocationGeonames()
    {
        $locationInput = 'http://sws.geonames.org/3030293';
        $filterManager = new CorpusFilterManager();
        $locationOutput = $filterManager->prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }

    /**
     * test prepare location geonames array
     *
     * @return void
     */
    public function testPrepareLocationGeonamesArray()
    {
        $locationInput = ['http://sws.geonames.org/3030293', 'http://www.geonames.org/3017382'];
        $filterManager = new CorpusFilterManager();
        $locationOutput = $filterManager->prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }


    /**
     * test prepare themes noop
     *
     * @return void
     */
    public function testPrepareThemesNoOp()
    {
        $themesInput = ['ark:/12148/cb11937931x', 'ark:/12148/cb11946662b', 'ark:/12148/cb13318415c'];
        $themesExpectedOutput = ['bnf|ark:/12148/cb11937931x', 'bnf|ark:/12148/cb11946662b', 'bnf|ark:/12148/cb13318415c'];
        $filterManager = new CorpusFilterManager();
        $themesOutput = $filterManager->prepareTheme($themesInput);
        $this->assertEquals($themesExpectedOutput, $themesOutput);
    }

    /**
     * test prepare themes full url
     *
     * @return void
     */
    public function testPrepareThemesFullUrl()
    {
        $themesInput = ['http://ark.bnf.fr/ark:/12148/cb11937931x', 'http://data.bnf.fr/ark:/12148/cb11946662b', 'https://ark.bnf.fr/ark:/12148/cb13318415c'];
        $filterManager = new CorpusFilterManager();
        $themesOutput = $filterManager->prepareTheme($themesInput);
        $this->assertEquals(['bnf|ark:/12148/cb11937931x', 'bnf|ark:/12148/cb11946662b', 'bnf|ark:/12148/cb13318415c'], $themesOutput);
    }


    /**
     * test prepare themes full url
     *
     * @return void
     */
    public function testPrepareThemesUnknown()
    {
        $themesInput = ['ark:/12148/cb11937931x', 'foo', 'ark:/12148/cb11946662b', 'ark:/12148/cb13318415c', 'bar'];
        $filterManager = new CorpusFilterManager();
        $themesOutput = $filterManager->prepareTheme($themesInput);
        $this->assertEquals(['bnf|ark:/12148/cb11937931x', 'txt|foo', 'bnf|ark:/12148/cb11946662b', 'bnf|ark:/12148/cb13318415c', 'txt|bar'], $themesOutput);
    }

    /**
     * test prepare themes mix
     *
     * @return void
     */
    public function testPrepareThemesMix()
    {
        $themesInput = ['ark:/12148/cb11937931x', 'foo', 'http://data.bnf.fr/ark:/12148/cb11946662b', 'ark:/12148/cb13318415c', 'bar'];
        $filterManager = new CorpusFilterManager();
        $themesOutput = $filterManager->prepareTheme($themesInput);
        $this->assertEquals(['bnf|ark:/12148/cb11937931x', 'txt|foo', 'bnf|ark:/12148/cb11946662b', 'bnf|ark:/12148/cb13318415c', 'txt|bar'], $themesOutput);
    }

    /**
     * test prepareDate
     *
     */
    public function testPrepareDate()
    {
        $datesInput = [ "1961", "1950-1960"];
        $filterManager = new CorpusFilterManager();
        $dateOutput = $filterManager->prepareDate($datesInput);
        $this->assertEquals(['1950-1960', '1961'], $dateOutput);
    }

    /**
     * test prepareDate bad format
     *
     */
    public function testPrepareDateBadFormat()
    {
        $datesInput = [ "1961", "1950-1960", "foo"];
        $filterManager = new CorpusFilterManager();
        $dateOutput = $filterManager->prepareDate($datesInput);
        $this->assertEquals(['1950-1960', '1961'], $dateOutput);
    }


    /**
     * test prepareDate null
     *
     */
    public function testPrepareDateNull()
    {
        $datesInput = null;
        $filterManager = new CorpusFilterManager();
        $dateOutput = $filterManager->prepareDate($datesInput);
        $this->assertEquals([], $dateOutput);
    }


    /**
     * test prepareDate empty
     *
     */
    public function testPrepareDateEmpty()
    {
        $datesInput = [];
        $filterManager = new CorpusFilterManager();
        $dateOutput = $filterManager->prepareDate($datesInput);
        $this->assertEquals([], $dateOutput);
    }


}
