<?php

use CorpusParole\Libraries\Mergers\CocoonSoundRdfMerger;
use CorpusParole\Libraries\CocoonUtils;

use EasyRdf\Graph;
use EasyRdf\Resource;
use EasyRdf\Literal;

use Illuminate\Foundation\Testing\WithoutMiddleware;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;

class CocoonSoundRdfMergerTest extends TestCase
{

    const TEST_INPUT_DOCS = [
        'BASE' => <<<EOT
        @prefix ore: <http://www.openarchives.org/ore/terms/> .
        @prefix edm: <http://www.europeana.eu/schemas/edm/> .
        @prefix dc11: <http://purl.org/dc/elements/1.1/> .
        @prefix olac: <http://www.language-archives.org/OLAC/1.1/> .
        @prefix dc: <http://purl.org/dc/terms/> .
        @prefix geo: <http://www.w3.org/2003/01/geo/wgs84_pos#> .
        @prefix xsd: <http://www.w3.org/2001/XMLSchema#> .
        @prefix skos: <http://www.w3.org/2004/02/skos/core#> .
        @prefix owl: <http://www.w3.org/2002/07/owl#> .

        <http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND>
          a ore:Aggregation ;
          edm:aggregatedCHO <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND> ;
          edm:provider "Corpus de la Parole"@fr ;
          edm:dataProvider <http://viaf.org/viaf/142432638> ;
          edm:isShownAt <http://corpusdelaparole.huma-num.fr/corpus-app#/detail/crdo-CFPP2000_35_SOUND> ;
          edm:isShownBy <http://cocoon.huma-num.fr/data/archi/masters/372593.wav> ;
          edm:rights <http://creativecommons.org/licenses/by-nc-sa/4.0/> ;
          edm:hasView <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav>, <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3> .

        <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND>
          a edm:ProvidedCHO ;
          dc11:description "Enregistrement issu du Corpus de Français Parlé Parisien des années 2000 (CFPP2000)"@fr, "Enregistrement: CaractÃ©ristiques de l'enregistrement; IdentifiantÂ : CFPP2000 [03-01] Ozgur_Kilic_H_32_alii_3e; LangueÂ : franÃ§ais; EditeurÂ : UniversitÃ© Paris III â Syled ; Date : 17.11.2010; Lieu de l'enregistrementÂ : au domicile de lâenquÃªtÃ© Ãzgur KILIÃ ; EnquÃªteurÂ : Sonia BRANCA-ROSOFF; Quartier(s) concernÃ©(s)Â : Paris 3e, et 20e (pour lâÃ¢ge adulte); TranscriptionÂ : Noalig TANGUY; AnonymisationÂ : Noalig TANGUY; DurÃ©eÂ : 0h46â; "@fr, "EnquÃªteur: Etat civil; PrÃ©nom et NOMÂ : Sonia BRANCA-ROSOFF; SexeÂ : fÃ©minin; Lieu de naissanceÂ : New-York (USA); Lieu la petite enfanceÂ : Bretagne, puis rÃ©gion parisienne, puis de 8 Ã  16 ans Nice; Date de naissance : 16.12.1947; Situation familialeÂ : mariÃ©e, deux enfants; MobilitÃ© gÃ©ographiqueÂ : Paris 1967-1971, Aix-en-Provence 1971-1996, Paris 1996-2012; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : Doctorat dâÃtat (sciences du langage) 1987; Travail; ActivitÃ© actuelleÂ : Enseignante-chercheuse (linguistique franÃ§aise), UniversitÃ© de la Sorbonne Nouvelle-Paris III, Paris; ActivitÃ©s passÃ©esÂ : ; Langues; Langue premiÃ¨reÂ : franÃ§ais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : nÃ©e en 1916 Ã  Nice (France), interprÃ¨te de confÃ©rencesÂ ; licence dâanglais; PÃ¨reÂ : nÃ© en 1913 Ã  Mananjary (Madagascar), ingÃ©nieur.; "@fr, "EnquÃªtÃ© 1: Etat civil; PrÃ©nom et NOM fictifsÂ : Ãzgur KILIÃ ; SexeÂ : masculin; Lieu de naissance: Paris 11e, France; Lieu la petite enfanceÂ : Paris 11e, France; Date de naissance : 07.02.1979; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : pacsÃ©, deux enfants; MobilitÃ© gÃ©ographiqueÂ : 3e, 18e, 20e arrondissements; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec autres enquÃªtÃ©sÂ : amis; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire  st Merri 75003Â ; Secondaire : Victor Hugo, Turgot 75003, Jean LurÃ§at 75013 gÃ©nÃ©ral bac ES SupÃ©rieur. Puis St Denis universitÃ© (cinÃ©ma); Travail; ActivitÃ© actuelleÂ : ingÃ©nieur du son pour l'audiovisuel; ActivitÃ©s passÃ©esÂ : restauration, assistant son pour l'audiovisuel ; Langues; Langues pratiquÃ©esÂ : franÃ§ais, anglais, turc; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : psychothÃ©rapeute, nÃ© a Houilles rÃ©gion parisienne France; PÃ¨reÂ : professeur d'universitÃ©, nÃ© a Istanbul en Turquie.  ; "@fr, "EnquÃªtÃ© 2: Etat civil; PrÃ©nom et NOM fictifsÂ : Michel CHEVRIER; SexeÂ : masculin; Lieu de naissance : naissance Ã  Boulogne-Billancourt (92); Lieu la petite enfance : rue du Temple, 3e arrondissement, Paris; Date de naissance : 01.06.1979; Age au moment de lâenregistrementÂ : 31 ans; Situation familialeÂ : concubinage; MobilitÃ© gÃ©ographiqueÂ : 3e, puis 20e (entre 2005 et 2009) puis Romainville (depuis 2009); Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; PrÃ©nom et NOM rÃ©elsÂ : ; RÃ©seau par lequel contactÃ©Â : ; Adresse (si connue)Â : ; TÃ©lÃ©phone (si connu)Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ãcole Ã  aire ouverte rue St Merri 75003Â ; Secondaire (CollÃ¨ge Victor Hugo 75003) puis LycÃ©e AutogÃ©rÃ© de Paris (75015) Bac LittÃ©raire. Deux annÃ©es au CIM (Ã©cole de jazz) dans le 75018, puis 3 ans au conservatoire du 9eme; Travail; ActivitÃ© actuelleÂ : Musicien, Compositeur, Professeur de piano jazz ; ActivitÃ©s passÃ©esÂ : Musicien; Langues; Langues pratiquÃ©esÂ : franÃ§ais, anglais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : ancienne mÃ©decin gÃ©nÃ©raliste pneumologue, puis sociologue, nÃ©e en France; PÃ¨reÂ : enseignant en sciences de l'Ã©ducation Ã  Paris 8 nÃ© en Belgique; "@fr, "EnquÃªtÃ© 3: Etat civil; PrÃ©nom et NOM fictifsÂ : BenoÃ®t DU BREUIL-DE-PONT-EN-AUGE; SexeÂ : masculin; Lieu de naissance : Paris 14e; Lieu la petite enfance :  Paris 3e; Date de naissance : 28.11.1978; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : concubinage avÃ©rÃ©, avec papier de la mairie tamponnÃ© et signÃ©!; MobilitÃ© gÃ©ographiqueÂ : Ã  2 ans vit dans le 9Â°, puis dÃ©mÃ©nage pour le 3Â°, est restÃ© dans la mÃªme rue pendant 20 ans tout en changeant d'immeuble. Ensuite le 19Â° pendant 4 ans, puis Pantin 6 mois puis Les Lilas. ; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec autres enquÃªtÃ©sÂ : voir ScolaritÃ©, PrÃ©cisions; PrÃ©nom et NOM rÃ©elsÂ : ; RÃ©seau par lequel contactÃ©Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [ ] Post-licence [ ] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : CAP; PrÃ©cisionsÂ : Primaire Ã©cole Ã  aire ouverte (St Merri) dans le 4Â° avec Augustin, Ãzgur, Michel. Secondaire : collÃ¨ge Victor-Hugo dans le 3Â° avec Ãzgur ; puis CAP ; Travail; ActivitÃ© actuelleÂ : ; ActivitÃ©s passÃ©esÂ : Ã©bÃ©niste agenceur puis Ã©bÃ©niste restaurateur et enfin constructeur dÃ©cors (ainsi que de nombreux petits jobs ayant durÃ© moins d'un an); Langues; Langues pratiquÃ©esÂ : franÃ§ais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : Ã  la retraite ; secrÃ©taire de rÃ©daction dans un grand journal, baccalaurÃ©at ; nÃ©e Ã  Montrouge; PÃ¨reÂ : conseiller Ã  travail Ã  pole emploi, nÃ© Ã  Boulogne Billancourt, baccalaurÃ©at prÃ©sentÃ© 3 ou 4 fois, puis obtenu par Ã©quivalence. ; "@fr, "EnquÃªtÃ© 4: Etat civil; EnquÃªtÃ© 4; PrÃ©nom et NOM fictifsÂ : ; SexeÂ : fÃ©minin / masculin; Lieu de naissance : Paris 14e; Lieu de la petite enfance : Paris 4e et 3e ; Date de naissance : 06.12.1976; Age au moment de lâenregistrementÂ : 34 ans; Situation familialeÂ : ; MobilitÃ© gÃ©ographiqueÂ : Rue la BruyÃ¨re 75009 (1976-1978), Rambuteau 75004 (1978-1987/88) & 75003 (chgt de cÃ´tÃ© de rue 1988-1998), a quittÃ© Paris de 1998 Ã  2005Â ; Rue Rambuteau 75003 (2006-2010), rue Gossec 75012 (2011); Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© 2Â : ami; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; RÃ©seau par lequel contactÃ©Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ecole St Merri 75003. Secondaire : CollÃ¨ge FranÃ§ois Couperin 75004, Institut St Pierre Fourier 75012Â ; Cours Charlemagne 75014 ; Travail; ActivitÃ© actuelleÂ : Superviseur Centre d'appels ; ActivitÃ©s passÃ©esÂ : Animateur Club Enfant, RÃ©ceptionniste ; Langues; Langues pratiquÃ©esÂ :; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : retraitÃ©e, Paris, bac + 1/2 (?) ; PÃ¨reÂ : Conseiller PÃ´le Emploi, Paris, bac. ; "@fr, "EnquÃªtÃ© 5: Etat civil; PrÃ©nom et NOM fictifsÂ : Ãtienne SALVEGAS; SexeÂ : masculin; Lieu de naissance : Paris 12e; Lieu de la petite enfance : Paris 3e Le Marais; Date de naissance :  16.06.1978; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : mariÃ© 1 enfant, 12 ans de vie commune ; MobilitÃ© gÃ©ographiqueÂ : 3e (1978-1999) 19e (1999-2002) 9e (2002-2011) ; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© 2 : ami ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; PrÃ©nom et NOM rÃ©elsÂ : ; ScolaritÃ©; Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [ ] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ecole Vertus, rue des Vertus 3Ã¨me. Secondaire CollÃ¨ge Montgolfier / LycÃ©e Turgot 3Ã¨me. SupÃ©rieur Droit Univ. Paris PanthÃ©on Sorbonne ; Travail; ActivitÃ© actuelleÂ : Ã©vÃ¨nementiel; ActivitÃ©s passÃ©esÂ : stagiaire journaliste sportif / relations publiques, accueil, agent de  sÃ©curitÃ©, remplacement gardien, rÃ©ceptionniste hÃ´tellerie.; Langues; Langues pratiquÃ©esÂ :; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : nÃ© Ã  Paris, lycÃ©e ; PÃ¨reÂ : nÃ© Ã  Oloron (64), lycÃ©e ; "@fr ;
          dc11:language <http://lexvo.org/id/iso639-3/fra> ;
          dc11:publisher <http://viaf.org/viaf/142432638> ;
          dc11:type <http://purl.org/dc/dcmitype/Sound>, "primary_text"^^olac:linguistic-type, "narrative"^^olac:discourse-type, "report"^^olac:discourse-type, "unintelligible_speech"^^olac:discourse-type ;
          dc:license <http://creativecommons.org/licenses/by-nc-sa/3.0/> ;
          dc11:subject "lexicography"^^olac:linguistic-field, "phonetics"^^olac:linguistic-field, "anthropological_linguistics"^^olac:linguistic-field, "general_linguistics"^^olac:linguistic-field, <http://lexvo.org/id/iso639-3/fra>, "text_and_corpus_linguistics"^^olac:linguistic-field, "phonology"^^olac:linguistic-field, "semantics"^^olac:linguistic-field, "sociolinguistics"^^olac:linguistic-field, "syntax"^^olac:linguistic-field, "typology"^^olac:linguistic-field, "discourse_analysis"^^olac:linguistic-field, "historical_linguistics"^^olac:linguistic-field, "language_documentation"^^olac:linguistic-field, "mathematical_linguistics"^^olac:linguistic-field ;
          dc11:title "CFPP2000 [03-01] Ozgur_Kilic_H_32_alii_3e Entretien de Ozgur KILIÇ 2"@fr ;
          dc:accessRights "Freely available for non-commercial use" ;
          dc:extent "PT48M26S" ;
          dc:isPartOf <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE>, <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_CFPP2000> ;
          dc11:source "CFPP2000"@fr ;
          dc11:coverage "Quartier concerné : 3e"@fr ;
          olac:depositor <http://viaf.org/viaf/93752300> ;
          dc11:contributor <http://viaf.org/viaf/93752300>, "Branca-Rosoff, Sonia", "Kiliç Ozgur", "Chevrier Michel", "du-Breuil-de-Pont-en-Auge Benoît", "du-Breuil-de-Pont-en-Auge Augustin", "Salvegas Etienne", "Tanguy Noalig" ;
          olac:interviewer "Branca-Rosoff, Sonia" ;
          olac:responder "Kiliç Ozgur", "Chevrier Michel", "du-Breuil-de-Pont-en-Auge Benoît", "du-Breuil-de-Pont-en-Auge Augustin", "Salvegas Etienne" ;
          olac:transcriber "Tanguy Noalig" ;
          dc:available "2013-10-12"^^dc:W3CDTF ;
          dc:created "2010-11-17"^^dc:W3CDTF ;
          dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF ;
          dc:spatial [
            a edm:Place ;
            geo:lat "48.83975"^^xsd:float ;
            geo:long "2.3542"^^xsd:float ;
            skos:note "FR"^^dc:ISO3166, "France, Paris, Université Sorbonne Nouvelle Paris 3, site Censier"@fr ;
            owl:sameAs <http://vocab.getty.edu/tgn/7008038>
          ] .

        <http://cocoon.huma-num.fr/data/archi/masters/372593.wav>
          a edm:WebResource ;
          dc:extent "PT48M26S" ;
          dc11:format "audio/x-wav"^^dc:IMT ;
          dc:created "2010-11-17"^^dc:W3CDTF ;
          dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF .

        <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav>
          a edm:WebResource ;
          dc:extent "PT48M26S" ;
          dc11:format "audio/x-wav"^^dc:IMT ;
          edm:isDerivativeOf <http://cocoon.huma-num.fr/data/archi/masters/372593.wav> ;
          dc:created "2010-11-17"^^dc:W3CDTF ;
          dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF .

        <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3>
          a edm:WebResource ;
          dc:extent "PT48M26S" ;
          dc11:format "audio/mpeg"^^dc:IMT ;
          edm:isDerivativeOf <http://cocoon.huma-num.fr/data/archi/masters/372593.wav> ;
          dc:created "2010-11-17"^^dc:W3CDTF ;
          dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF .
EOT
        ,
        'SOURCE' => <<<EOT
        @prefix ore: <http://www.openarchives.org/ore/terms/> .
        @prefix edm: <http://www.europeana.eu/schemas/edm/> .
        @prefix dc11: <http://purl.org/dc/elements/1.1/> .
        @prefix olac: <http://www.language-archives.org/OLAC/1.1/> .
        @prefix dc: <http://purl.org/dc/terms/> .
        @prefix geo: <http://www.w3.org/2003/01/geo/wgs84_pos#> .
        @prefix xsd: <http://www.w3.org/2001/XMLSchema#> .
        @prefix skos: <http://www.w3.org/2004/02/skos/core#> .
        @prefix owl: <http://www.w3.org/2002/07/owl#> .

        <http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND>
          a ore:Aggregation ;
          edm:aggregatedCHO <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND> ;
          edm:provider "Corpus de la Parole"@fr ;
          edm:dataProvider <http://viaf.org/viaf/142432638> ;
          edm:isShownAt <http://corpusdelaparole.huma-num.fr/corpus-app#/detail/crdo-CFPP2000_35_SOUND> ;
          edm:isShownBy <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xml> ;
          edm:rights <http://creativecommons.org/licenses/by-nc-sa/4.0/> ;
          edm:hasView <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xml>, <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xhtml>, <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav> .

        <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND>
          a edm:ProvidedCHO ;
          dc11:description "Enregistrement issu du Corpus de Français Parlé Parisien des années 2000 (CFPP2000)"@fr, "Enregistrement: CaractÃ©ristiques de l'enregistrement; IdentifiantÂ : CFPP2000 [03-01] Ozgur_Kilic_H_32_alii_3e; LangueÂ : franÃ§ais; EditeurÂ : UniversitÃ© Paris III â Syled ; Date : 17.11.2010; Lieu de l'enregistrementÂ : au domicile de lâenquÃªtÃ© Ãzgur KILIÃ ; EnquÃªteurÂ : Sonia BRANCA-ROSOFF; Quartier(s) concernÃ©(s)Â : Paris 3e, et 20e (pour lâÃ¢ge adulte); TranscriptionÂ : Noalig TANGUY; AnonymisationÂ : Noalig TANGUY; DurÃ©eÂ : 0h46â; "@fr, "EnquÃªteur: Etat civil; PrÃ©nom et NOMÂ : Sonia BRANCA-ROSOFF; SexeÂ : fÃ©minin; Lieu de naissanceÂ : New-York (USA); Lieu la petite enfanceÂ : Bretagne, puis rÃ©gion parisienne, puis de 8 Ã  16 ans Nice; Date de naissance : 16.12.1947; Situation familialeÂ : mariÃ©e, deux enfants; MobilitÃ© gÃ©ographiqueÂ : Paris 1967-1971, Aix-en-Provence 1971-1996, Paris 1996-2012; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : Doctorat dâÃtat (sciences du langage) 1987; Travail; ActivitÃ© actuelleÂ : Enseignante-chercheuse (linguistique franÃ§aise), UniversitÃ© de la Sorbonne Nouvelle-Paris III, Paris; ActivitÃ©s passÃ©esÂ : ; Langues; Langue premiÃ¨reÂ : franÃ§ais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : nÃ©e en 1916 Ã  Nice (France), interprÃ¨te de confÃ©rencesÂ ; licence dâanglais; PÃ¨reÂ : nÃ© en 1913 Ã  Mananjary (Madagascar), ingÃ©nieur.; "@fr, "EnquÃªtÃ© 1: Etat civil; PrÃ©nom et NOM fictifsÂ : Ãzgur KILIÃ ; SexeÂ : masculin; Lieu de naissance: Paris 11e, France; Lieu la petite enfanceÂ : Paris 11e, France; Date de naissance : 07.02.1979; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : pacsÃ©, deux enfants; MobilitÃ© gÃ©ographiqueÂ : 3e, 18e, 20e arrondissements; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec autres enquÃªtÃ©sÂ : amis; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire  st Merri 75003Â ; Secondaire : Victor Hugo, Turgot 75003, Jean LurÃ§at 75013 gÃ©nÃ©ral bac ES SupÃ©rieur. Puis St Denis universitÃ© (cinÃ©ma); Travail; ActivitÃ© actuelleÂ : ingÃ©nieur du son pour l'audiovisuel; ActivitÃ©s passÃ©esÂ : restauration, assistant son pour l'audiovisuel ; Langues; Langues pratiquÃ©esÂ : franÃ§ais, anglais, turc; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : psychothÃ©rapeute, nÃ© a Houilles rÃ©gion parisienne France; PÃ¨reÂ : professeur d'universitÃ©, nÃ© a Istanbul en Turquie.  ; "@fr, "EnquÃªtÃ© 2: Etat civil; PrÃ©nom et NOM fictifsÂ : Michel CHEVRIER; SexeÂ : masculin; Lieu de naissance : naissance Ã  Boulogne-Billancourt (92); Lieu la petite enfance : rue du Temple, 3e arrondissement, Paris; Date de naissance : 01.06.1979; Age au moment de lâenregistrementÂ : 31 ans; Situation familialeÂ : concubinage; MobilitÃ© gÃ©ographiqueÂ : 3e, puis 20e (entre 2005 et 2009) puis Romainville (depuis 2009); Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; PrÃ©nom et NOM rÃ©elsÂ : ; RÃ©seau par lequel contactÃ©Â : ; Adresse (si connue)Â : ; TÃ©lÃ©phone (si connu)Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ãcole Ã  aire ouverte rue St Merri 75003Â ; Secondaire (CollÃ¨ge Victor Hugo 75003) puis LycÃ©e AutogÃ©rÃ© de Paris (75015) Bac LittÃ©raire. Deux annÃ©es au CIM (Ã©cole de jazz) dans le 75018, puis 3 ans au conservatoire du 9eme; Travail; ActivitÃ© actuelleÂ : Musicien, Compositeur, Professeur de piano jazz ; ActivitÃ©s passÃ©esÂ : Musicien; Langues; Langues pratiquÃ©esÂ : franÃ§ais, anglais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : ancienne mÃ©decin gÃ©nÃ©raliste pneumologue, puis sociologue, nÃ©e en France; PÃ¨reÂ : enseignant en sciences de l'Ã©ducation Ã  Paris 8 nÃ© en Belgique; "@fr, "EnquÃªtÃ© 3: Etat civil; PrÃ©nom et NOM fictifsÂ : BenoÃ®t DU BREUIL-DE-PONT-EN-AUGE; SexeÂ : masculin; Lieu de naissance : Paris 14e; Lieu la petite enfance :  Paris 3e; Date de naissance : 28.11.1978; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : concubinage avÃ©rÃ©, avec papier de la mairie tamponnÃ© et signÃ©!; MobilitÃ© gÃ©ographiqueÂ : Ã  2 ans vit dans le 9Â°, puis dÃ©mÃ©nage pour le 3Â°, est restÃ© dans la mÃªme rue pendant 20 ans tout en changeant d'immeuble. Ensuite le 19Â° pendant 4 ans, puis Pantin 6 mois puis Les Lilas. ; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec autres enquÃªtÃ©sÂ : voir ScolaritÃ©, PrÃ©cisions; PrÃ©nom et NOM rÃ©elsÂ : ; RÃ©seau par lequel contactÃ©Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [ ] Post-licence [ ] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : CAP; PrÃ©cisionsÂ : Primaire Ã©cole Ã  aire ouverte (St Merri) dans le 4Â° avec Augustin, Ãzgur, Michel. Secondaire : collÃ¨ge Victor-Hugo dans le 3Â° avec Ãzgur ; puis CAP ; Travail; ActivitÃ© actuelleÂ : ; ActivitÃ©s passÃ©esÂ : Ã©bÃ©niste agenceur puis Ã©bÃ©niste restaurateur et enfin constructeur dÃ©cors (ainsi que de nombreux petits jobs ayant durÃ© moins d'un an); Langues; Langues pratiquÃ©esÂ : franÃ§ais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : Ã  la retraite ; secrÃ©taire de rÃ©daction dans un grand journal, baccalaurÃ©at ; nÃ©e Ã  Montrouge; PÃ¨reÂ : conseiller Ã  travail Ã  pole emploi, nÃ© Ã  Boulogne Billancourt, baccalaurÃ©at prÃ©sentÃ© 3 ou 4 fois, puis obtenu par Ã©quivalence. ; "@fr, "EnquÃªtÃ© 4: Etat civil; EnquÃªtÃ© 4; PrÃ©nom et NOM fictifsÂ : ; SexeÂ : fÃ©minin / masculin; Lieu de naissance : Paris 14e; Lieu de la petite enfance : Paris 4e et 3e ; Date de naissance : 06.12.1976; Age au moment de lâenregistrementÂ : 34 ans; Situation familialeÂ : ; MobilitÃ© gÃ©ographiqueÂ : Rue la BruyÃ¨re 75009 (1976-1978), Rambuteau 75004 (1978-1987/88) & 75003 (chgt de cÃ´tÃ© de rue 1988-1998), a quittÃ© Paris de 1998 Ã  2005Â ; Rue Rambuteau 75003 (2006-2010), rue Gossec 75012 (2011); Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© 2Â : ami; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; RÃ©seau par lequel contactÃ©Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ecole St Merri 75003. Secondaire : CollÃ¨ge FranÃ§ois Couperin 75004, Institut St Pierre Fourier 75012Â ; Cours Charlemagne 75014 ; Travail; ActivitÃ© actuelleÂ : Superviseur Centre d'appels ; ActivitÃ©s passÃ©esÂ : Animateur Club Enfant, RÃ©ceptionniste ; Langues; Langues pratiquÃ©esÂ :; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : retraitÃ©e, Paris, bac + 1/2 (?) ; PÃ¨reÂ : Conseiller PÃ´le Emploi, Paris, bac. ; "@fr, "EnquÃªtÃ© 5: Etat civil; PrÃ©nom et NOM fictifsÂ : Ãtienne SALVEGAS; SexeÂ : masculin; Lieu de naissance : Paris 12e; Lieu de la petite enfance : Paris 3e Le Marais; Date de naissance :  16.06.1978; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : mariÃ© 1 enfant, 12 ans de vie commune ; MobilitÃ© gÃ©ographiqueÂ : 3e (1978-1999) 19e (1999-2002) 9e (2002-2011) ; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© 2 : ami ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; PrÃ©nom et NOM rÃ©elsÂ : ; ScolaritÃ©; Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [ ] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ecole Vertus, rue des Vertus 3Ã¨me. Secondaire CollÃ¨ge Montgolfier / LycÃ©e Turgot 3Ã¨me. SupÃ©rieur Droit Univ. Paris PanthÃ©on Sorbonne ; Travail; ActivitÃ© actuelleÂ : Ã©vÃ¨nementiel; ActivitÃ©s passÃ©esÂ : stagiaire journaliste sportif / relations publiques, accueil, agent de  sÃ©curitÃ©, remplacement gardien, rÃ©ceptionniste hÃ´tellerie.; Langues; Langues pratiquÃ©esÂ :; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : nÃ© Ã  Paris, lycÃ©e ; PÃ¨reÂ : nÃ© Ã  Oloron (64), lycÃ©e ; "@fr ;
          dc11:language <http://lexvo.org/id/iso639-3/fra> ;
          dc11:publisher <http://viaf.org/viaf/142432638> ;
          dc11:type "primary_text"^^olac:linguistic-type, "narrative"^^olac:discourse-type, "report"^^olac:discourse-type, "unintelligible_speech"^^olac:discourse-type ;
          dc:license <http://creativecommons.org/licenses/by-nc-sa/3.0/> ;
          dc11:subject "lexicography"^^olac:linguistic-field, "phonetics"^^olac:linguistic-field, "anthropological_linguistics"^^olac:linguistic-field, "general_linguistics"^^olac:linguistic-field, <http://lexvo.org/id/iso639-3/fra>, "text_and_corpus_linguistics"^^olac:linguistic-field, "phonology"^^olac:linguistic-field, "semantics"^^olac:linguistic-field, "sociolinguistics"^^olac:linguistic-field, "syntax"^^olac:linguistic-field, "typology"^^olac:linguistic-field, "discourse_analysis"^^olac:linguistic-field, "historical_linguistics"^^olac:linguistic-field, "language_documentation"^^olac:linguistic-field, "mathematical_linguistics"^^olac:linguistic-field ;
          dc11:title "CFPP2000 [03-01] Ozgur_Kilic_H_32_alii_3e Entretien de Ozgur KILIÇ 2 (annotations)"@fr ;
          dc:accessRights "Freely available for non-commercial use" ;
          dc:isPartOf <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE>, <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_CFPP2000> ;
          dc11:source "CFPP2000"@fr ;
          dc11:coverage "Quartier concerné : 3e"@fr ;
          olac:depositor <http://viaf.org/viaf/93752300> ;
          dc11:contributor <http://viaf.org/viaf/93752300>, "Branca-Rosoff, Sonia", "Kiliç Ozgur", "Chevrier Michel", "du-Breuil-de-Pont-en-Auge Benoît", "du-Breuil-de-Pont-en-Auge Augustin", "Salvegas Etienne", "Tanguy Noalig" ;
          olac:interviewer "Branca-Rosoff, Sonia" ;
          olac:responder "Kiliç Ozgur", "Chevrier Michel", "du-Breuil-de-Pont-en-Auge Benoît", "du-Breuil-de-Pont-en-Auge Augustin", "Salvegas Etienne" ;
          olac:transcriber "Tanguy Noalig" ;
          dc:available "2013-11-04"^^dc:W3CDTF ;
          dc:created "2010-11-17"^^dc:W3CDTF ;
          dc:issued "2013-11-04T22:20:07+01:00"^^dc:W3CDTF ;
          dc:spatial [
            a edm:Place ;
            geo:lat "99.83975"^^xsd:float ;
            geo:long "25.3542"^^xsd:float ;
            skos:note "FR"^^dc:ISO3166, "France, Paris, Université Sorbonne Nouvelle Paris 3, site Censier"@fr, "Other place very important"@fr ;
            owl:sameAs <http://vocab.getty.edu/tgn/1234567890>
          ] .

        <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xml>
          a edm:WebResource ;
          dc11:format "application/xml"^^dc:IMT ;
          dc:created "2010-11-17"^^dc:W3CDTF ;
          dc:issued "2013-11-04T22:20:07+01:00"^^dc:W3CDTF .

        <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xhtml>
          a edm:WebResource ;
          dc11:format "application/xhtml+xml"^^dc:IMT ;
          dc:created "2010-11-17"^^dc:W3CDTF ;
          dc:issued "2013-11-04T22:20:07+01:00"^^dc:W3CDTF .
EOT
    ];

    const TEST_INPUT_ID = "crdo-ESLO1_ENT_019";
    const TEST_GRAPH_URI = "http://corpusdelaparole.huma-num.fr/corpus/res/crdo-ESLO1_ENT_019";
    const TEST_CHO_URI = "http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019";


    private $inputGraphes = [];
    private $resGraph;
    private $merger;

    function __construct(string $name = null) {
        parent::__construct($name);
    }

    public function setUp() {
        parent::setUp();

        foreach(CocoonSoundRdfMergerTest::TEST_INPUT_DOCS as $key => $inputDoc) {
            $this->inputGraphes[$key] = new Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND", $inputDoc);
        }

        $this->merger = new CocoonSoundRdfMerger();
        $this->resGraph = $this->merger->mergeGraph($this->inputGraphes['BASE'], $this->inputGraphes['SOURCE']);
    }

    public function tearDown() {
        parent::tearDown();
    }

    /**
     * Just test that the construct and setup are ok
     *
     * @return void
     */
    public function testInit()
    {
        $this->assertTrue(true);
        echo $this->resGraph->serialise('turtle');
    }

    /**
     * test the result graph.
     *
     * @return void
     */
    public function testResGraph() {
        $this->assertNotNull($this->resGraph, "Res graph must not be null");
        $this->assertEquals("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND", $this->resGraph->getUri(), "graph uri must be equals to http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND");
    }

    /**
     * test the merged Aggregation
     *
     * @return void
     */
    public function testAggregationResource() {
        $aggregationResList = $this->resGraph->allOfType('http://www.openarchives.org/ore/terms/Aggregation');
        $this->assertCount(1, $aggregationResList, "Only one Aggregation node");
        $aggregationRes = $aggregationResList[0];

        $this->assertEquals("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND",$aggregationRes->getUri(),"Aggregation node uri must be http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND");

        $singleValuesRes = [
            'http://www.europeana.eu/schemas/edm/aggregatedCHO' => ['size'=>1, 'val' => $this->resGraph->resource("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND")],
            'http://www.europeana.eu/schemas/edm/provider' => ['size'=>1, 'val' => new Literal("Corpus de la Parole", "fr", null)],
            'http://www.europeana.eu/schemas/edm/dataProvider' => ['size'=>1, 'val' => $this->resGraph->resource("http://viaf.org/viaf/142432638")],
            'http://www.europeana.eu/schemas/edm/isShownAt' => ['size'=>1, 'val' => $this->resGraph->resource("http://corpusdelaparole.huma-num.fr/corpus-app#/detail/crdo-CFPP2000_35_SOUND")],
            'http://www.europeana.eu/schemas/edm/isShownBy' => ['size'=>1, 'val' => $this->resGraph->resource("http://cocoon.huma-num.fr/data/archi/masters/372593.wav")],
            'http://www.europeana.eu/schemas/edm/rights' => ['size'=>1, 'val' => $this->resGraph->resource("http://creativecommons.org/licenses/by-nc-sa/4.0/")],
            'http://www.europeana.eu/schemas/edm/hasView' => ['size' => 4, 'val' => [
                $this->resGraph->resource("http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav"),
                $this->resGraph->resource("http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3"),
                $this->resGraph->resource("http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xml"),
                $this->resGraph->resource("http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xhtml")
            ]]
        ];

        foreach($singleValuesRes as $property => $resVal) {
            $resList = $aggregationRes->all($this->inputGraphes['BASE']->resource($property));
            $this->assertCount($resVal['size'], $resList, "$property list size $resVal[size]");
            if($resVal['size'] == 1) {
                $this->assertEquals($resVal['val'], $resList[0], "$property size one not equals");
            }
            else {
                $this->assertEquals($resVal['val'], $resList, "$property size more than one not equals");
            }
        }

    }

    /**
     * test web resources
     *
     * @return void
     */
    public function testWebResources() {
        $webResList = $this->resGraph->allOfType('http://www.europeana.eu/schemas/edm/WebResource');

        $this->assertCount(5, $webResList, "Must have 5 webResource");
    }


    /**
    * Test one to one mapping spatial info
    *
    * @return void
    */
    public function testProvidedCHOSpatial() {
        $providedCHO = $this->resGraph->get('edm:ProvidedCHO', '^rdf:type');

        $this->assertNotNull($providedCHO);

        $spatials = $providedCHO->all($this->resGraph->resource('http://purl.org/dc/terms/spatial'));

        $this->assertCount(1, $spatials, "Must have only one spatial node");
        $spatial = $spatials[0];
        $this->assertTrue($spatial->isBNode(),"spatial node must be blank");
        $this->assertEquals("edm:Place", $spatial->type(), "spatial node type must be edm:Place");

        $lats = $spatial->all('geo:lat');
        $this->assertCount(1, $lats, "One latitude");
        $this->assertInstanceOf("EasyRdf\Literal", $lats[0], "Latitude must be a litteral");
        $this->assertEquals(Literal::create("48.83975", null, 'xsd:float'), $lats[0], "lat must be '48.83975'^^xsd:float");

        $longs = $spatial->all('geo:long');
        $this->assertCount(1, $longs, "One longitude");
        $this->assertInstanceOf("EasyRdf\Literal", $longs[0], "Longitude must be a litteral");
        $this->assertEquals(Literal::create("2.3542", null, 'xsd:float'), $longs[0], "long must be '2.3542'^^xsd:float");

        $notes = $spatial->all('skos:note');
        $this->assertCount(3, $notes, "3 notes");
        $this->assertContainsOnlyInstancesOf("EasyRdf\Literal", $notes, "Notes mus be only literals");
        $this->assertEquals(Literal::create("FR", null, "dc:ISO3166"), $notes[0], "notes contains 'FR'^^dc:ISO3166");
        $this->assertEquals(Literal::create("France, Paris, Université Sorbonne Nouvelle Paris 3, site Censier", "fr", null), $notes[1], "notes contains 'France, Centre, Loiret, Orléans'@fr");
        $this->assertEquals(Literal::create("Other place very important", "fr", null), $notes[2], "notes contains 'Other place very important'@fr");

        $sameas = $spatial->all('owl:sameAs');
        $this->assertCount(1, $sameas, "1 same as");
        $this->assertContainsOnlyInstancesOf("EasyRdf\Resource", $sameas, "Notes mus be only resources");
        $this->assertEquals('http://vocab.getty.edu/tgn/7008038', $sameas[0]->getUri(), "uri must be http://vocab.getty.edu/tgn/7008038");
    }


    /**
     * Test one to one mapping
     *
     * @return void
     */
    public function testProvidedCHOSingleproperty() {

        $properties = [
            "http://purl.org/dc/elements/1.1/title",
            "http://purl.org/dc/terms/extent",
            "http://purl.org/dc/terms/available",
            "http://purl.org/dc/terms/created",
            "http://purl.org/dc/terms/issued"
        ];

        $providedCHO = $this->resGraph->get('edm:ProvidedCHO', '^rdf:type');
        $baseNode = $this->inputGraphes['BASE']->get('edm:ProvidedCHO', '^rdf:type');

        $this->assertNotNull($providedCHO);
        $this->assertNotNull($baseNode);

        foreach ($properties as $prop) {
            $outputValuesStr = [];
            foreach($providedCHO->all($this->resGraph->resource($prop)) as $outputValue) {
                array_push($outputValuesStr, strval($outputValue));
            }
            $this->assertNotEmpty($outputValuesStr, "we must found some values to test $prop");
            foreach ($baseNode->all($this->inputGraphes['BASE']->resource($prop)) as $value) {
                $this->assertContains(strval($value), $outputValuesStr, "$prop not found in output graph");
            }
        }

    }


}
