<?php

use Mockery as m;

use EasyRdf\Literal;

class DateStatsControllerTest extends TestCase
{
    private $sparqlClient;

    public function setUp() {

        parent::setup();

        // create a mock of the post repository interface and inject it into the
        // IoC container
        $this->sparqlClient = m::mock('CorpusParole\Libraries\Sparql\SparqlClient');
        $this->app->instance('CorpusParole\Libraries\Sparql\SparqlClient', $this->sparqlClient);
    }

    public function tearDown() {
        m::close();
        parent::tearDown();
    }


    public function testIndexQuery() {

        $query =  preg_replace('/\s+/', ' ', "SELECT (?d as ?date) (COUNT(?d) AS ?count)
            WHERE {
                ?_ a <http://www.europeana.eu/schemas/edm/ProvidedCHO>.
                ?_ <http://purl.org/dc/terms/created> ?d
            }
            GROUP BY ?d
            ORDER BY ?d");


        $this->sparqlClient
            ->shouldReceive('query')
            ->with($query)
            ->once()
            ->andReturn(new \ArrayIterator([]));
        $this->get('/api/v1/stats/datestats/');
        $this->seeJsonEquals(["datestats" => []]);
    }

    public function testIndexMultiple() {

         $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('1975', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
                 (object)['date'=>new Literal('1965', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(2)],
                 (object)['date'=>new Literal('1955', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(3)],
             ]));
         $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
         $this->seeJsonEquals(["datestats" => [
             "1955" => 3,
             "1965" => 2,
             "1975" => 1,
         ]]);
    }

    public function testIndexSimple() {

         $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('1955', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
                 (object)['date'=>new Literal('1965', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
                 (object)['date'=>new Literal('1975', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
             ]));
         $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
         $this->seeJsonEquals(["datestats" => [
             "1955" => 1,
             "1965" => 1,
             "1975" => 1,
         ]]);
    }

    public function testIndexPeriod() {

        $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('start=1955; end=1965', null, "http://purl.org/dc/terms/Period"), 'count' => Literal::create(11)],
             ]));
        $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
        $this->seeJsonEquals(["datestats" => [
            "1955" => 1,
            "1956" => 1,
            "1957" => 1,
            "1958" => 1,
            "1959" => 1,
            "1960" => 1,
            "1961" => 1,
            "1962" => 1,
            "1963" => 1,
            "1964" => 1,
            "1965" => 1,
        ]]);
    }

    public function testIndexPeriodRemainMore() {

        $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('start=1955; end=1965', null, "http://purl.org/dc/terms/Period"), 'count' => Literal::create(15)],
             ]));
        $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
        $this->seeJsonEquals(["datestats" => [
            "1955" => 2,
            "1956" => 2,
            "1957" => 2,
            "1958" => 2,
            "1959" => 1,
            "1960" => 1,
            "1961" => 1,
            "1962" => 1,
            "1963" => 1,
            "1964" => 1,
            "1965" => 1,
        ]]);
    }

    public function testIndexPeriodRemainLess() {

        $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('start=1955; end=1965', null, "http://purl.org/dc/terms/Period"), 'count' => Literal::create(10)],
             ]));
        $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
        $this->seeJsonEquals(["datestats" => [
            "1955" => 1,
            "1956" => 1,
            "1957" => 1,
            "1958" => 1,
            "1959" => 1,
            "1960" => 1,
            "1961" => 1,
            "1962" => 1,
            "1963" => 1,
            "1964" => 1,
        ]]);
    }

    public function testIndexMix() {

        $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('start=1955; end=1965', null, "http://purl.org/dc/terms/Period"), 'count' => Literal::create(11)],
                 (object)['date'=>new Literal('1960', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(2)],
             ]));
        $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
        $this->seeJsonEquals(["datestats" => [
            "1955" => 1,
            "1956" => 1,
            "1957" => 1,
            "1958" => 1,
            "1959" => 1,
            "1960" => 3,
            "1961" => 1,
            "1962" => 1,
            "1963" => 1,
            "1964" => 1,
            "1965" => 1,
        ]]);
    }

    public function testIndexBadDate() {

         $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('1955', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
                 (object)['date'=>new Literal('HELLO', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
                 (object)['date'=>new Literal('1975', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
             ]));
         $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
         $this->seeJsonEquals(["datestats" => [
             "1955" => 1,
             "1975" => 1,
         ]]);
    }

    public function testIndexBadPeriod() {

        $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('start=1955; end=FOO', null, "http://purl.org/dc/terms/Period"), 'count' => Literal::create(11)],
             ]));
        $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
        $this->seeJsonEquals(["datestats" => [
        ]]);
    }

    public function testIndexBadPeriodMissing() {

        $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('start=1955', null, "http://purl.org/dc/terms/Period"), 'count' => Literal::create(11)],
             ]));
        $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
        $this->seeJsonEquals(["datestats" => [
        ]]);
    }

    public function testIndexFullPeriod() {

        $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('start=1955; end=1965; scheme=v3; name=v4;', null, "http://purl.org/dc/terms/Period"), 'count' => Literal::create(11)],
             ]));
        $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
        $this->seeJsonEquals(["datestats" => [
            "1955" => 1,
            "1956" => 1,
            "1957" => 1,
            "1958" => 1,
            "1959" => 1,
            "1960" => 1,
            "1961" => 1,
            "1962" => 1,
            "1963" => 1,
            "1964" => 1,
            "1965" => 1,
        ]]);
    }

    public function testIndexMultipleFormat() {

         $this->sparqlClient
             ->shouldReceive('query')
             ->once()
             ->andReturn(new \ArrayIterator([
                 (object)['date'=>new Literal('1975-02-05', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(1)],
                 (object)['date'=>new Literal('1965-03', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(2)],
                 (object)['date'=>new Literal('1955-02-12T08:30:00+00:00', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(3)],
                 (object)['date'=>new Literal('1950-08-18T08:30:00Z', null, "http://purl.org/dc/terms/W3CDTF"), 'count' => Literal::create(4)],
             ]));
         $this->get('/api/v1/stats/datestats/')->assertTrue($this->response->isOk(), $this->response->content());
         $this->seeJsonEquals(["datestats" => [
             "1950" => 4,
             "1955" => 3,
             "1965" => 2,
             "1975" => 1,
         ]]);
    }



}
