#!/usr/bin/env bash -l

set -e
## option --prod/--dev

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

VAGRANT_STARTED=false

green="\x1B[0;32m" # '\e[1;32m' is too bright for white bg.
blue="\x1B[1;34m"
endColor="\x1B[0m"

function echoblue() {
    echo -e "${blue}$1${endColor}"
}

function install() {
    pushd "$DIR"

    echoblue "---> preparing bo client"
    pushd ../server/bo_client
    /usr/local/bin/npm install
    ./node_modules/.bin/bower install
    popd
    echoblue "---> preparing bo client done"

    echoblue "---> preparing back"
    pushd ../server/src
    php composer.phar install
    /usr/local/bin/npm install
    ./node_modules/.bin/bower install
    popd
    echoblue "---> preparing back done"

    echoblue "---> preparing app-client"
    pushd ../cms/app-client
    /usr/local/bin/npm install
    ./node_modules/.bin/bower install
    popd
    echoblue "---> preparing app-client done"

    echoblue "---> preparing module"
    pushd ../cms
    npm install
    popd
    echoblue "---> preparing module done"


    echoblue "---> checking vagrant"
    if vagrant status | grep -q -v "running"; then
        echoblue "---> starting vagrant"
        # if which virtualenvwrapper.sh > /dev/null 2>&1 ; then
        #     echoblue "---> activating ansible"
        #     source `which virtualenvwrapper.sh`
        #     workon ansible
        # fi
        vagrant up
        VAGRANT_STARTED=true
        # if type -t deactivate > /dev/null 2>&1 ; then
        #     deactivate
        # fi
    fi

    echoblue "---> done"

    popd > /dev/null

    return 0
}

function usage() {
    cat <<EOF
Usage: $0 [-I] [-p|-d] [-h]
    -I : do not run install
    -h : print this message
    -p : build for production
    -d : build for development
EOF
}

environment=""
do_install=true


while getopts "Ihpd" opt; do
  case $opt in
    I) do_install=false ;;
    h) usage; exit 0 ;;
    p) [[ -n "$environment" ]] && { usage >&2; exit 1; } || environment='production' ;;
    d) [[ -n "$environment" ]] && { usage >&2; exit 1; } || environment='development' ;;
    \?) usage >&2; exit 1 ;;
  esac
done
shift $((OPTIND-1))

[[ -z "$environment" ]] && environment='production'

case $environment in
    development)
        build_option='--dev'
        build_option_back='--development'
        ;;
    *)
        build_option='--prod'
        build_option_back='--production'
        ;;
esac

echo "environment: $environment"
echo "do_install: $do_install"
[[ "$do_install" == true ]] && echoblue "DO INSTALL" && install;

pushd "$DIR"

echoblue "---> cleaning build folder"
rm -fr root

echoblue "---> creating build folder"
mkdir -p root/var/www/corpusdelaparole/corpus-back
mkdir -p root/var/www/corpusdelaparole/drupal/sites/all/modules

echoblue "---> building back"
pushd ../server/src
version=$(sed -n "s/[[:space:]]*\'version\'[[:space:]]*=>[[:space:]]*\'\([\.0-9]*\)\'/\1/p" config/version.php)
version=${version:-0.0.0}
npm install
./node_modules/.bin/bower install
./node_modules/.bin/gulp copy-build ${build_option_back}
popd
echoblue "---> building back done"

echoblue "---> building app-client"
pushd ../cms/app-client
npm install
./node_modules/.bin/bower install
./node_modules/.bin/ember build ${build_option}
popd
echoblue "---> building app-client done"

echoblue "---> building module"
pushd ../cms
npm install
./node_modules/.bin/gulp copy-build ${build_option} --version="$version"
popd

echoblue "---> building package"
vagrant ssh -c "/vagrant/build_rpm.sh"
echoblue "---> building package done"

if [ "$VAGRANT_STARTED" = true ]; then
    echoblue "---> Stopping vagrant"
    vagrant halt
    echoblue "---> Stopping vagrant done"
fi

popd > /dev/null

echoblue "---> done"

