<?php

use CorpusParole\Libraries\Sparql\SparqlClient;
use CorpusParole\Libraries\CorpusParoleException;
use CorpusParole\Models\Document;

use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Middleware;

use Mockery as m;

class SparqlClientTest extends TestCase {

    const TEST_DOC_ADD = <<<EOT
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "drama"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> .
EOT;

    const TEST_DOC_DELETE = <<<EOT
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> .
EOT;


    private $container;
    private $sparqlClients;
    private $transactionUrl;

    private $responsesArray;

    function __construct(string $name = null) {
        parent::__construct($name);
    }

    private function getSparqlClient($responses, &$container, $sparqlClient) {
        $mock = new MockHandler($responses);
        $handler = HandlerStack::create($mock);
        $history = Middleware::history($container);
        $handler->push($history);
        $httpClient = new Client(['handler' => $handler, 'http_errors' => false]);

        return new SparqlClient($httpClient, $sparqlClient);
    }

    public function setUp() {
        parent::setUp();
        $this->sesameRepository = config('corpusparole.sesame_repository');
        $this->transactionUrl = config('corpusparole.sesame_query_url').'/transactions/64a5937f-c112-d014-a044-f0123b93';

        $this->addGraph = new EasyRdf\Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", SparqlClientTest::TEST_DOC_ADD);
        $this->deleteGraph = new EasyRdf\Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", SparqlClientTest::TEST_DOC_DELETE);

    }

    public function tearDown() {
        m::close();
        parent::tearDown();
    }

    public function testCreateTransaction() {
        $responses = [
            new Response(201, ['Location' => "$this->transactionUrl"]),
        ];
        $container = [];
        $sparqlClientMock = m::mock('EasyRdf\Sparql\Client');

        $documentRepository = $this->getSparqlClient($responses, $container, $sparqlClientMock);

        $documentRepository->startTransaction();

        $this->assertEquals($this->transactionUrl, $documentRepository->getCurrentTransactionUrl(), 'Must have correct transaction url');
    }

    public function testCreateTransactionHistory() {
        $responses = [
            new Response(201, ['Location' => "$this->transactionUrl"]),
        ];
        $container = [];
        $sparqlClientMock = m::mock('EasyRdf\Sparql\Client');

        $documentRepository = $this->getSparqlClient($responses, $container, $sparqlClientMock);

        $documentRepository->startTransaction();

        $this->assertCount(1, $container, 'One request');
        $req = $container[0]['request'];
        $this->assertEquals("http:{$this->sesameRepository}/transactions?isolation-level=http%3A%2F%2Fwww.openrdf.org%2Fschema%2Fsesame%23SNAPSHOT_READ", (string)$req->getUri(), "url must be ok");
        $this->assertEquals('POST', $container[0]['request']->getMethod(), "methos is POST");
    }

    public function testRollbackTransaction() {
        $responses = [
            new Response(201, ['Location' => "$this->transactionUrl"]),
            new Response(204)
        ];
        $container = [];
        $sparqlClientMock = m::mock('EasyRdf\Sparql\Client');

        $documentRepository = $this->getSparqlClient($responses, $container, $sparqlClientMock);

        $documentRepository->startTransaction();
        $documentRepository->rollback();

        $this->assertCount(2, $container, '2 requests');

        $this->assertNull($documentRepository->getCurrentTransactionUrl(), "Current Transaction url must be null");

        $req = $container[1]['request'];

        $this->assertEquals($this->transactionUrl, (string)$req->getUri(), "uri must be the transaction url");
        $this->assertEquals('DELETE', $req->getMethod(), "Method must be DELETE");

    }

    public function testCommitTransaction() {
        $responses = [
            new Response(201, ['Location' => "$this->transactionUrl"]),
            new Response(200)
        ];
        $container = [];
        $sparqlClientMock = m::mock('EasyRdf\Sparql\Client');

        $documentRepository = $this->getSparqlClient($responses, $container, $sparqlClientMock);

        $documentRepository->startTransaction();
        $documentRepository->commit();


        $this->assertCount(2, $container, '2 requests');

        $this->assertNull($documentRepository->getCurrentTransactionUrl(), "Current Transaction url must be null");

        $req = $container[1]['request'];

        $this->assertEquals($this->transactionUrl."?action=COMMIT", (string)$req->getUri(), "uri must be the transaction url");
        $this->assertEquals('PUT', $req->getMethod(), "Method must be PUT");

    }

    /**
     * @expectedException CorpusParole\Libraries\CorpusParoleException
     */
    public function testCommitTransactionFail() {
        $responses = [
            new Response(201, ['Location' => "$this->transactionUrl"]),
            new Response(404, [], "Not found")
        ];
        $container = [];
        $sparqlClientMock = m::mock('EasyRdf\Sparql\Client');

        $documentRepository = $this->getSparqlClient($responses, $container, $sparqlClientMock);

        $documentRepository->startTransaction();
        $documentRepository->commit();

    }


    public function testAdd() {

        $responses = [
            new Response(201, ['Location' => "$this->transactionUrl"]),
            new Response(204)
        ];
        $container = [];
        $sparqlClientMock = m::mock('EasyRdf\Sparql\Client');

        $documentRepository = $this->getSparqlClient($responses, $container, $sparqlClientMock);

        $documentRepository->startTransaction();
        $documentRepository->add($this->addGraph);

        $this->assertCount(2, $container, '2 requests');

        $this->assertNotNull($documentRepository->getCurrentTransactionUrl(), "Current Transaction url must be not null");

        $req = $container[1]['request'];

        $this->assertEquals($this->transactionUrl."?action=UPDATE", (string)$req->getUri(), "uri must be the transaction url");
        $this->assertEquals('PUT', $req->getMethod(), "Method must be PUT");
        $this->assertEquals(['application/sparql-update; charset=utf-8'], $req->getHeader('Content-type'), "content type must be form urlencoded");

        $body = (string)$req->getBody();

        $this->assertContains('INSERT DATA {', $body, 'update parameter must contain INSERT');
        $this->assertContains('GRAPH <'.$this->addGraph->getUri().'> {', $body, 'update parameter must contain GRAPH id');
        $this->assertContains('<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type>', $body, 'update parameter must contain dialogue');
        $this->assertContains('<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "drama"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> .', $body, 'update parameter must contain drama');
    }

    public function testDelete() {
        $responses = [
            new Response(201, ['Location' => "$this->transactionUrl"]),
            new Response(204)
        ];
        $container = [];
        $sparqlClientMock = m::mock('EasyRdf\Sparql\Client');

        $documentRepository = $this->getSparqlClient($responses, $container, $sparqlClientMock);

        $documentRepository->startTransaction();
        $documentRepository->delete($this->deleteGraph);

        $this->assertCount(2, $container, '2 requests');

        $this->assertNotNull($documentRepository->getCurrentTransactionUrl(), "Current Transaction url must be not null");

        $req = $container[1]['request'];

        $this->assertEquals($this->transactionUrl."?action=UPDATE", (string)$req->getUri(), "uri must be the transaction url");
        $this->assertEquals('PUT', $req->getMethod(), "Method must be PUT");
        $this->assertEquals(['application/sparql-update; charset=utf-8'], $req->getHeader('Content-type'), "content type must be form urlencoded");

        $body = (string)$req->getBody();

        $this->assertContains('DELETE DATA {', $body, 'update parameter must contain DELETE');
        $this->assertContains('GRAPH <'.$this->addGraph->getUri().'> {', $body, 'update parameter must contain GRAPH id');
        $this->assertContains('<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type>', $body, 'update parameter must contain GRAPH id');

    }

}
