<?php
namespace CorpusParole\Models;

use Config;
use CorpusParole\Libraries\Utils;
use CorpusParole\Libraries\RdfModel\RdfModelResource;
use JsonSerializable;
use Log;
use EasyRdf\Literal;
use EasyRdf\Resource;
use EasyRdf\Graph;
use EasyRdf\Isomorphic;


/**
 * Model class for Document. Inherit from EasyRd\Resource
 * SELECT DISTINCT ?g WHERE {GRAPH ?g {?s ?p ?o}}
 */
class Document extends RdfModelResource implements JsonSerializable {

    public function __construct($uri, $graph = null) {
        parent::__construct($uri, $graph);
    }

    private $id = null;

    public function getId() {
        if(is_null($this->id)) {
            $this->id = substr($this->uri, strlen(Config::get('corpusparole.cocoon_doc_id_base_uri')));
        }
        return $this->id;
    }

    public function getTitle() {
        try {
            return $this->getLiteral('<http://purl.org/dc/elements/1.1/title>');
        } catch(\Exception $e) {
            return null;
        }
    }

    public function getPublishers() {
        try {
            return $this->allLiterals('dc11:publisher');
        } catch(\Exception $e) {
           return [];
       }
    }

    public function getMediaArray() {

        //TODO: add media type
        $res = [];
        $formats = [];
        try {
            $formats = $this->allResources("dc:isFormatOf");
        } catch(\Exception $e) {
            // do nothing
        }
        foreach ($formats as $f) {
            $uri = $f->getUri();
            $mimetype = Utils::get_mimetype($uri);
            array_push($res, ["url" => $uri, "format" => $mimetype]);
        }

        $format = null;
        try {
            $format = $this->getLiteral('dc11:format');
        } catch(\Exception $e) {
            // do nothing
        }
        array_push($res, ["url" => $this->getUri(), "format" => $format]);
        return $res;
    }

    public function getTypes() {
        return $this->all('dc11:type');
    }

    public function getDiscourseTypes() {
        return array_values(array_filter($this->getTypes(), function($v) {
            return $v instanceof Literal && $v->getDatatypeUri() === Config::get('constants.OLAC_DISCOURSE_TYPE')['uri'];
        }));
    }

    public function getOtherTypes() {
        $res = array_values(array_filter($this->getTypes(), function($v) {
            return $v instanceof Resource || $v->getDatatypeUri() !== Config::get('constants.OLAC_DISCOURSE_TYPE')['uri'];
        }));
        return $res;
    }

    public function updateDiscourseTypes(array $discoursesTypes) {

        $this->startDelta();

        foreach($this->getDiscourseTypes() as $discourseType) {
            $this->delete('dc11:type', $discourseType);
            $this->currentDelta->getDeletedGraph()->add($this, 'dc11:type', new Literal($discourseType, null, Config::get('constants.OLAC_DISCOURSE_TYPE')['uri']));
        }
        // re-add them

        foreach($discoursesTypes as $dType) {
            $this->add('dc11:type', new Literal($dType, null, Config::get('constants.OLAC_DISCOURSE_TYPE')['uri']));
            $this->currentDelta->getAddedGraph()->add($this, 'dc11:type', new Literal($dType, null, Config::get('constants.OLAC_DISCOURSE_TYPE')['uri']));
        }
    }

    public function isIsomorphic($doc) {
        return Isomorphic::isomorphic($this->graph, $doc->graph);
    }

    /*
     * Clone document.
     * clone also the innerDocumenent
     */
    public function __clone() {

        $this->graph = new Graph($this->graph->getUri(), $this->graph->toRdfPhp());
    }

    public function jsonSerialize() {
        if(!$this->graph) {
            return [
                'id' => $this->getId(),
            ];
        } else {
            $mediaArray = array_map(
                function($m) {
                    $f = Utils::process_literal_or_string($m['format']);
                    return ['url' => $m['url'], 'format' => $f];},
                $this->getMediaArray()
            );

            $publishers = array_map(
                function($v) { return Utils::process_literal_or_string($v); },
                $this->getPublishers()
            );

            return [
                'id' => $this->getId(),
                'uri' => $this->getUri(),
                'title' => $this->getTitle()->getValue(),
                'publishers' => $publishers,
                'mediaArray'=> $mediaArray
            ];
        }
    }

}
