#!/usr/bin/env bash -l

set -e
## option --prod/--dev

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

VAGRANT_STARTED=false

green="\x1B[0;32m" # '\e[1;32m' is too bright for white bg.
blue="\x1B[1;34m"
endColor="\x1B[0m"

function echoblue() {
    echo -e "${blue}$1${endColor}"
}

function install() {
    pushd "$DIR" > /dev/null

    echoblue "---> preparing bo client"
    pushd ../server/bo_client > /dev/null
    /usr/local/bin/npm install
    ./node_modules/.bin/bower install
    popd > /dev/null
    echoblue "---> preparing bo client done"

    echoblue "---> preparing back"
    pushd ../server/src > /dev/null
    php composer.phar install
    /usr/local/bin/npm install
    ./node_modules/.bin/bower install
    popd > /dev/null
    echoblue "---> preparing back done"

    echoblue "---> preparing app-client"
    pushd ../cms/app-client > /dev/null
    /usr/local/bin/npm install
    ./node_modules/.bin/bower install
    popd > /dev/null
    echoblue "---> preparing app-client done"

    echoblue "---> preparing module"
    pushd ../cms > /dev/null
    npm install
    popd > /dev/null
    echoblue "---> preparing module done"


    echoblue "---> checking vagrant"
    if vagrant status | grep -q -v "running"; then
        echoblue "---> starting vagrant"
        # if which virtualenvwrapper.sh > /dev/null 2>&1 ; then
        #     echoblue "---> activating ansible"
        #     source `which virtualenvwrapper.sh`
        #     workon ansible
        # fi
        vagrant up
        VAGRANT_STARTED=true
        # if type -t deactivate > /dev/null 2>&1 ; then
        #     deactivate
        # fi
    fi

    echoblue "---> done"

    popd > /dev/null

    return 0
}

function usage() {
    cat <<EOF
Usage: $0 [-I] [-p|-d] [-h]
    -I : do not run install
    -h : print this message
    -p : build for production
    -d : build for development
EOF
}

environment=""
do_install=true


while getopts "Ihpd" opt; do
  case $opt in
    I) do_install=false ;;
    h) usage; exit 0 ;;
    p) [[ -n "$environment" ]] && { usage >&2; exit 1; } || environment='production' ;;
    d) [[ -n "$environment" ]] && { usage >&2; exit 1; } || environment='development' ;;
    \?) usage >&2; exit 1 ;;
  esac
done
shift $((OPTIND-1))

[[ -z "$environment" ]] && environment='production'

case $environment in
    development)
        build_option='--dev'
        build_option_back='--development'
        ;;
    *)
        build_option='--prod'
        build_option_back='--production'
        ;;
esac

echo "environment: $environment"
echo "do_install: $do_install"
[[ "$do_install" == true ]] && echoblue "DO INSTALL" && install;

pushd "$DIR" > /dev/null

echoblue "---> cleaning build folder"
rm -fr root

echoblue "---> creating build folder"
mkdir -p root/var/www/corpusdelaparole/corpus-back
mkdir -p root/var/www/corpusdelaparole/drupal/sites/all/modules

echoblue "---> building back"
pushd ../server/src > /dev/null
version=$(sed -n "s/[[:space:]]*\'version\'[[:space:]]*=>[[:space:]]*\'\([\.0-9]*\)\'/\1/p" config/version.php)
version=${version:-0.0.0}
npm install
./node_modules/.bin/bower install
./node_modules/.bin/gulp copy-build ${build_option_back}
popd > /dev/null
echoblue "---> building back done"

echoblue "---> building app-client"
pushd ../cms/app-client > /dev/null
npm install
./node_modules/.bin/bower install
./node_modules/.bin/ember build ${build_option}
popd > /dev/null
echoblue "---> building app-client done"

echoblue "---> building module"
pushd ../cms > /dev/null
npm install
./node_modules/.bin/gulp copy-build ${build_option} --version="$version"
popd > /dev/null

echoblue "---> building package"
vagrant ssh -c "/vagrant/build_rpm.sh"
echoblue "---> building package done"

if [ "$VAGRANT_STARTED" = true ]; then
    echoblue "---> Stopping vagrant"
    vagrant halt
    echoblue "---> Stopping vagrant done"
fi

popd > /dev/null

echoblue "--> archiving dist"

pushd "$DIR/dist" > /dev/null

rm -f corpusdelaparole-back_*_*.tar.gz
ARCHIVE_NAME="corpusdelaparole-back_$(date +%Y-%m-%d)_${version}"

mkdir "$ARCHIVE_NAME"

cp bootstrap-puppet.sh corpusdelaparole-$version-*.noarch.rpm installDrupal.sh puppet-corpusdelaparole-$version-*.noarch.rpm "$ARCHIVE_NAME"

tar zcvf "${ARCHIVE_NAME}.tar.gz" "$ARCHIVE_NAME"

rm -fr "$ARCHIVE_NAME"

popd > /dev/null

echoblue "---> done"

