<?php

use CorpusParole\Libraries\Sparql\SparqlQueryAnalyser;

class SparqlQueryAnalyserTest extends TestCase
{
    const QUERIES = [
        "BASE" => "base.sparql",
        "ASK" => "ask.sparql",
        "GRAPH" => "graph.sparql",
        "LIMIT_OFFSET" => "limit_offset.sparql",
        "PREFIXES" => "prefixes.sparql",
        "SELECT" => "select.sparql",
        "UNKNOWN" => "unknown.sparql",
    ];


    public function getTestQuery($key) {
        return file_get_contents(__DIR__.'/files/SparqlQueryAnalyserTest/'.self::QUERIES[$key]);
    }
    /**
     * A basic test jsut test object creation.
     *
     * @return void
     */
    public function testCreation() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("BASE"));
        $this->assertNotNull($analyser);
    }

    public function testQuerySelect() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("SELECT"));
        $this->assertEquals(SparqlQueryAnalyser::SELECT_QUERY, $analyser->getQueryType());
    }

    public function testQueryGraph() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("GRAPH"));
        $this->assertEquals(SparqlQueryAnalyser::GRAPH_QUERY, $analyser->getQueryType());
    }

    public function testQueryAsk() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("ASK"));
        $this->assertEquals(SparqlQueryAnalyser::ASK_QUERY, $analyser->getQueryType());
    }

    public function testQueryUnkown() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("UNKNOWN"));
        $this->assertEquals(SparqlQueryAnalyser::UNKNOWN_QUERY, $analyser->getQueryType());
    }

    public function testLimitOffset() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("LIMIT_OFFSET"));
        $this->assertEquals(20, $analyser->getOffset());
        $this->assertEquals(10, $analyser->getLimit());
    }

    public function testPrefixes() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("PREFIXES"));
        $this->assertCount(5, $analyser->getPrefixes());
        $this->assertEquals([
            "" => "http://www.google.com/",
            "rdf" => "http://www.w3.org/1999/02/22-rdf-syntax-ns#",
            "rdfs" => "http://www.w3.org/2000/01/rdf-schema#",
            "foaf" => "http://xmlns.com/foaf/0.1/",
            "dbpedia-owl" => "http://dbpedia.org/ontology/"
        ], $analyser->getPrefixes());
    }

    public function testRawPrefixes() {
        $analyser = new SparqlQueryAnalyser($this->getTestQuery("PREFIXES"));
        $this->assertCount(5, $analyser->getRawPrefixes());
        $this->assertEquals([
            "BASE <http://www.google.com/>",
            "PREFIX rdf:<http://www.w3.org/1999/02/22-rdf-syntax-ns#>",
            "PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>",
            "prefix foaf: <http://xmlns.com/foaf/0.1/>",
            "PREFIX dbpedia-owl: <http://dbpedia.org/ontology/>"
        ], $analyser->getRawPrefixes());
    }

}
