<?php

namespace CorpusParole\Http\Controllers\Api;

use Illuminate\Http\Request;
//use Illuminate\Http\Response;

use CorpusParole\Http\Requests;
use CorpusParole\Http\Controllers\Controller;
use CorpusParole\Repositories\DocumentRepository;
use CorpusParole\Services\TranscriptManager;
use CorpusParole\Libraries\Filters\CorpusFilterManager;

class DocumentController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct(DocumentRepository $documentRepo, TranscriptManager $transcriptManager) {
        $this->documentRepository = $documentRepo;
        $this->transcriptManager = $transcriptManager;
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index(Request $request)
    {
        $perPage = intval($request->input('perpage', config('corpusparole.documents_per_page')));

        $filters = [];

        $languages = CorpusFilterManager::prepareLanguages($request->input('language', []));
        if(!empty($languages)) {
            $filters['language'] = $languages;
        }

        $location = CorpusFilterManager::prepareLocation($request->input('location', ''));
        if(!empty($location)) {
            $filters['location'] = $location;
        }

        $themes = CorpusFilterManager::prepareTheme($request->input('theme', []));
        if(!empty($themes)) {
            $filters['themes'] = $themes;
        }

        $discourses = CorpusFilterManager::prepareDiscourse($request->input('discourse', []));
        if(!empty($discourses)) {
            $filters['discourses'] = $discourses;
        }

        $sort = $request->input('sort', null);

        $paginator = $this->documentRepository->paginate($filters, $perPage, config('corpusparole.pagination_page_param'), null, $sort);

        $res = $paginator->toArray();
        if(array_key_exists('data', $res)) {
            $documents = $res['data'];
            unset($res['data']);
        } else {
            $documents = [];
        }
        return response()->json([ 'documents' => $documents, 'meta' => $res]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  Request  $request
     * @return Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  string  $id
     * @return Response
     */
    public function show(Request $request, $id)
    {
        $id = urldecode($id);
        $short = filter_var($request->input('short', false), FILTER_VALIDATE_BOOLEAN);
        $doc = $this->documentRepository->get($id, $short);
        if(is_null($doc)) {
            abort(404);
        }
        //$this->documentRepository->resolveLexvo([$doc,]);
        return response()->json(["document" => $doc]);
    }

    /**
     * Display the resource transcript
     *
     * @param string $id
     * @return Response
     */
    public function transcript($id) {
        $id= urldecode($id);

        $doc = $this->documentRepository->get($id);
        if(is_null($doc) || is_null($doc->getTranscript()) ) {
            abort(404);
        }
        $transcript = $doc->getTranscript();
        $transcriptUrl = $transcript->getUrl();
        if(empty($transcriptUrl) || empty($transcript->getConformsTo())) {
            abort(404);
        }
        $converter = $this->transcriptManager->getConverterUrl($transcript->getConformsTo(), $doc, $transcriptUrl);
        return response()->json($converter->convertToJson());

    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified document in storage.
     *
     * @param  Request  $request
     * @param  int  $id
     * @return Response
     */
    public function update(Request $request, $id)
    {
        $id= urldecode($id);

        $data = $request->json();
        $document = $data->get('document');
        $doc = $this->documentRepository->get($id);
        if(is_null($doc)) {
            abort(404);
        }

        //for now, update contributors and subjects only
        $doc->setContributors($document['contributors']);
        $doc->setSubjects($document['subjects']);
        $doc->addGeoInfo()->setRefLocs($document['reflocs']);
        $doc->getGeoInfo()->commit();

        $doc->setModified();

        $this->documentRepository->save($doc);

        return response('', 204);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return Response
     */
    public function destroy($id)
    {
        //
    }
}
