class sysconfig::handle (
    $hdl_host     = hiera('sysconfig::params::hdl_host', $sysconfig::params::hdl_host),
    $hdl_user     = hiera('sysconfig::params::hdl_user', $sysconfig::params::hdl_user),
    $hdl_srv_port = hiera('sysconfig::params::hdl_srv_port', $sysconfig::params::hdl_srv_port),
    $hdl_web_port = hiera('sysconfig::params::hdl_web_port', $sysconfig::params::hdl_web_port),
    $hdl_key_pswd = hiera('sysconfig::params::hdl_key_pswd', $sysconfig::params::hdl_key_pswd),
    $hdl_prefix   = hiera('sysconfig::params::hdl_prefix', $sysconfig::params::hdl_prefix),
    $hdl_db_name  = hiera('sysconfig::params::hdl_db_name', $sysconfig::params::hdl_db_name),
    $db_user      = hiera('sysconfig::params::db_user',      $sysconfig::params::db_user),
    $db_pw        = hiera('sysconfig::params::db_pw',        $sysconfig::params::db_pw),
    $db_host      = hiera('sysconfig::params::db_host',      $sysconfig::params::db_host)
) inherits sysconfig::params {

    # iptables -t nat -A PREROUTING -p tcp --dport 80 -j REDIRECT --to-ports 1080

    notify {'handle': name => "\$hdl_host : ${hdl_host}, \$hdl_srv_port : ${hdl_srv_port}, \$hdl_web_port : ${hdl_web_port}", withpath => true }

    $handle_dirs = [
        "/opt/handle",
        "/opt/handle/sbin",
        "/tmp/handle"
    ]

    group { "$hdl_user":
        ensure => present,
    }->
    user { "handle user":
        name => $hdl_user,
        ensure => present,
        gid => $hdl_user,
        system => true,
        home => '/var/lib/handle',
        managehome => false,
        shell => '/sbin/nologin'
    }->
    file { "/var/lib/handle":
        ensure => directory,
        owner => "$hdl_user",
        group => "$hdl_user"
    }->
    file { $handle_dirs:
        ensure => directory,
    }->
    archive { 'handle-8.1':
        ensure => present,
        url => "http://www.handle.net/hnr-source/hsj-8.1.0.tar.gz",
        checksum => false,
        follow_redirects => true,
        strip_components => 1,
        target => "/opt/handle",
        src_target => '/tmp',
        timeout => 500,
    }->
    file { "/opt/handle/sbin/start_handle":
        owner => "$hdl_user",
        group => "$hdl_user",
        mode => "0750",
        source => "puppet:///modules/sysconfig/handle/start_handle",
    }->
    file { '/opt/handle/lib/mysql-connector-java-5.1.38-bin.jar':
        ensure => file,
        source => "puppet:///modules/sysconfig/handle/mysql-connector-java-5.1.38-bin.jar",
        mode => "0644"
    }->
    file { '/tmp/handle/handle_db.sql':
        ensure => file,
        source => "puppet:///modules/sysconfig/handle/handle_db.sql",
        owner => "$hdl_user",
        group => "$hdl_user",
        mode => "0750"
    }->
    mysql::db { "handle_db":
        dbname => $hdl_db_name,
        user => $db_user,
        password => $db_pw,
        host => $db_host,
        sql => '/tmp/handle/handle_db.sql',
        charset => 'utf8',
        collate => 'utf8_general_ci',
        notify => Exec['rm-tmp-handle-db'],
        require => Service["mariadb"],
    }->
    file { "/tmp/handle/handle_init_script.exp":
        ensure => file,
        content => template("sysconfig/handle/handle_script.exp.erb"),
        owner => "$hdl_user",
        group => "$hdl_user",
        mode => "0750"
    }->
    exec { "init-handle":
        command => "/tmp/handle/handle_init_script.exp",
        cwd => "/opt/handle/bin",
        provider => 'shell',
        creates => "/var/lib/handle/sitebndl.zip",
        user => $hdl_user,
        notify => Exec['rm-tmp-handle'],
        require => [Package['java-install'], Package['expect']]
    }->
    file { "/var/lib/handle/config.dct":
        ensure => file,
        owner => "$hdl_user",
        group => "$hdl_user",
        content => template("sysconfig/handle/handle_config.dct.erb"),
        replace => true
    }->
    exec { "handle-first-start":
        command => "/usr/bin/echo \"${HANDLE_KEY_PSWD}\" | /opt/handle/bin/hdl-server /var/lib/handle &",
        provider => 'shell',
        creates => "/var/lib/handle/.handle_initialized",
        user => $hdl_user,
        require => File['/etc/sysconfig/handle']
    }->
    exec { "handle-first-stop":
        command => "/usr/bin/sleep 10 && rm -f /var/lib/handle/delete_this_to_stop_server && touch /var/lib/handle/.handle_initialized",
        provider => 'shell',
        creates => "/var/lib/handle/.handle_initialized",
        user => $hdl_user,
        require => File['/etc/sysconfig/handle']
    }->
    file_line { "handle-detached-prefix":
        path => "/var/lib/handle/config.dct",
        match => "\s*\"300:0.NA/$hdl_prefix\"",
        line => "      \"300:$hdl_prefix/ADMIN\"",
        multiple => true,
    }

    # file { "/tmp/handle/handle_answers.txt":
    #     ensure => file,
    #     content => template("sysconfig/handle/handle_answers.tmp.erb"),
    # }#->
    # exec { "init-handle":
    #     command => "/opt/handle/bin/hdl-setup-server /var/lib/handle < /tmp/handle/handle_answers.txt",
    #     cwd => "/opt/handle/bin",
    #     provider => 'shell',
    #     creates => "/var/lib/handle/sitebndl.zip",
    #     user => $hdl_user,
    #     notify => Exec['rm-tmp-handle'],
    #     require => Package['java-install']
    # }->
    # file { "/var/lib/handle/config.dct":
    #     ensure => file,
    #     owner => "$hdl_user",
    #     group => "$hdl_user",
    #     content => template("sysconfig/handle/handle_config.dct.erb"),
    #     replace => true
    # }->
    # exec { "open-handle-web-port":
    #     command => "firewall-cmd --permanent --add-port=8000/tcp",
    #     path => [ '/bin', '/usr/bin', '/usr/local/bin' ]
    # }->
    # exec { "open-handle-srv-port":
    #     command => "firewall-cmd --permanent --add-port=2641/tcp",
    #     path => [ '/bin', '/usr/bin', '/usr/local/bin' ]
    # }->
    # exec { "open-handle-port-reload":
    #     command => "firewall-cmd --reload",
    #     path => [ '/bin', '/usr/bin', '/usr/local/bin' ]
    # }->
    # exec { "handle-first-start":
    #     command => "/usr/bin/echo \"${HANDLE_KEY_PSWD}\" | /opt/handle/bin/hdl-server /var/lib/handle &",
    #     provider => 'shell',
    #     creates => "/var/lib/handle/.handle_initialized",
    #     user => $hdl_user,
    #     require => File['/etc/sysconfig/handle']
    # }->
    # exec { "handle-first-stop":
    #     command => "/usr/bin/sleep 10 && rm -f /var/lib/handle/delete_this_to_stop_server && touch /var/lib/handle/.handle_initialized",
    #     provider => 'shell',
    #     creates => "/var/lib/handle/.handle_initialized",
    #     user => $hdl_user,
    #     require => File['/etc/sysconfig/handle']
    # }#->
    # file_line { "handle-detached-prefix":
    #     path => "/var/lib/handle/config.dct",
    #     match => "\s*\"300:0.NA/$hdl_prefix\"",
    #     line => "      \"300:$hdl_prefix/ADMIN\"",
    #     multiple => true,
    # }


    file { "/etc/sysconfig/handle":
        ensure => file,
        owner => "root",
        group => "$hdl_user",
        content => template("sysconfig/handle/handle_sysconfig.erb"),
        replace => true,
        mode => "0640"
    }->
    file { "/etc/systemd/system/handle.service":
        ensure => file,
        owner => "root",
        group => "root",
        source => "puppet:///modules/sysconfig/handle/handle.service",
        require => File['/opt/handle/sbin/start_handle'],
        notify => Exec['systemctl-daemon-reload']
    }

    exec { "handle-enable-on-boot":
        command => "/usr/bin/systemctl enable handle",
        subscribe => [ Exec['systemctl-daemon-reload'], File['/etc/systemd/system/handle.service'] ],
        unless => "/usr/bin/systemctl is-enabled handle.service"
    }

    # exec { "rm-tmp-handle":
    #     command => "/usr/bin/rm /tmp/handle/handle_answers.txt",
    #     refreshonly => true
    # }

    exec { "rm-tmp-handle":
        command => "/usr/bin/rm /tmp/handle/handle_init_script.exp",
        refreshonly => true
    }

    exec { "rm-tmp-handle-db":
        command => "/usr/bin/rm /tmp/handle/handle_db.sql",
        refreshonly => true
    }

    exec { "open-handle-web-port":
        command => "firewall-cmd --permanent --add-port=8000/tcp",
        path => [ '/bin', '/usr/bin', '/usr/local/bin' ]
    }->
    exec { "open-handle-srv-port":
        command => "firewall-cmd --permanent --add-port=2641/tcp",
        path => [ '/bin', '/usr/bin', '/usr/local/bin' ]
    }->
    exec { "open-handle-port-reload":
        command => "firewall-cmd --reload",
        path => [ '/bin', '/usr/bin', '/usr/local/bin' ]
    }

}
