<?php

/**
 * Returns the predefined administrative areas.
 *
 * @param $country_code
 *   The country code for which administrative areas should be returned.
 *
 * @return
 *   An array of predefined administrative areas for the given country code, or
 *   NULL if not found.
 */
function addressfield_get_administrative_areas($country_code) {
  // Maintain a static cache to avoid passing the administrative areas through
  // t() more than once per request.
  $administrative_areas = &drupal_static(__FUNCTION__, array());
  if (empty($administrative_areas)) {
    // Get the default administrative areas.
    $administrative_areas = _addressfield_get_administrative_areas_defaults();

    // Allow other modules to alter the administrative areas.
    drupal_alter('addressfield_administrative_areas', $administrative_areas);
  }

  return isset($administrative_areas[$country_code]) ? $administrative_areas[$country_code] : NULL;
}

/**
 * Provides the default administrative areas.
 */
function _addressfield_get_administrative_areas_defaults() {
  // To avoid needless pollution of the strings list we only pass to t()
  // those administrative areas that are in English (or a latin transcription),
  // and belong to a country that either has multiple official languages (CA)
  // or uses a non-latin script (AE, CN, JP, KR, UA, RU, etc).
  // No translation is expected in other cases.
  $administrative_areas = array();
  $administrative_areas['AE'] = array(
    'AZ' => t('Abu Dhabi'),
    'FU' => t('Fujairah'),
    'UQ' => t('Umm al-Quwain'),
    'SH' => t('Sharjah'),
    'DU' => t('Dubai'),
    'RK' => t('Ras al-Khaimah'),
    'AJ' => t('Ajmān'),
  );
  $administrative_areas['AR'] = array(
    'B' => 'Buenos Aires',
    'K' => 'Catamarca',
    'H' => 'Chaco',
    'U' => 'Chubut',
    'C' => 'Ciudad de Buenos Aires',
    'X' => 'Córdoba',
    'W' => 'Corrientes',
    'E' => 'Entre Ríos',
    'P' => 'Formosa',
    'Y' => 'Jujuy',
    'L' => 'La Pampa',
    'F' => 'La Rioja',
    'M' => 'Mendoza',
    'N' => 'Misiones',
    'Q' => 'Neuquén',
    'R' => 'Río Negro',
    'A' => 'Salta',
    'J' => 'San Juan',
    'D' => 'San Luis',
    'Z' => 'Santa Cruz',
    'S' => 'Santa Fe',
    'G' => 'Santiago del Estero',
    'V' => 'Tierra del Fuego',
    'T' => 'Tucumán',
  );
  $administrative_areas['AU'] = array(
    'ACT' => 'Australian Capital Territory',
    'NSW' => 'New South Wales',
    'NT' => 'Northern Territory',
    'QLD' => 'Queensland',
    'SA' => 'South Australia',
    'TAS' => 'Tasmania',
    'VIC' => 'Victoria',
    'WA' => 'Western Australia',
  );
  $administrative_areas['BR'] = array(
    'AC' => 'Acre',
    'AL' => 'Alagoas',
    'AM' => 'Amazonas',
    'AP' => 'Amapá',
    'BA' => 'Bahia',
    'CE' => 'Ceará',
    'DF' => 'Distrito Federal',
    'ES' => 'Espírito Santo',
    'GO' => 'Goiás',
    'MA' => 'Maranhão',
    'MG' => 'Minas Gerais',
    'MS' => 'Mato Grosso do Sul',
    'MT' => 'Mato Grosso',
    'PA' => 'Pará',
    'PB' => 'Paraíba',
    'PE' => 'Pernambuco',
    'PI' => 'Piauí',
    'PR' => 'Paraná',
    'RJ' => 'Rio de Janeiro',
    'RN' => 'Rio Grande do Norte',
    'RO' => 'Rondônia',
    'RR' => 'Roraima',
    'RS' => 'Rio Grande do Sul',
    'SC' => 'Santa Catarina',
    'SE' => 'Sergipe',
    'SP' => 'São Paulo',
    'TO' => 'Tocantins',
  );
  $administrative_areas['CA'] = array(
    'AB' => t('Alberta'),
    'BC' => t('British Columbia'),
    'MB' => t('Manitoba'),
    'NB' => t('New Brunswick'),
    'NL' => t('Newfoundland and Labrador'),
    'NT' => t('Northwest Territories'),
    'NS' => t('Nova Scotia'),
    'NU' => t('Nunavut'),
    'ON' => t('Ontario'),
    'PE' => t('Prince Edward Island'),
    'QC' => t('Quebec'),
    'SK' => t('Saskatchewan'),
    'YT' => t('Yukon Territory'),
  );
  $administrative_areas['CL'] = array(
    'AI' => 'Aysén del General Carlos Ibáñez del Campo',
    'AN' => 'Antofagasta',
    'AR' => 'Araucanía',
    'AP' => 'Arica y Parinacota',
    'AT' => 'Atacama',
    'BI' => 'Biobío',
    'CO' => 'Coquimbo',
    'LI' => 'Libertador General Bernardo O\'Higgins',
    'LL' => 'Los Lagos',
    'LR' => 'Los Ríos',
    'MA' => 'Magallanes y de la Antártica Chilena',
    'ML' => 'Maule',
    'RM' => 'Metropolitana de Santiago',
    'TA' => 'Tarapacá',
    'VS' => 'Valparaíso',
  );
  $administrative_areas['CN'] = array(
    '34' => t('Anhui Sheng'),
    '92' => t('Macau'),
    '11' => t('Beijing Shi'),
    '50' => t('Chongqing Shi'),
    '35' => t('Fujian Sheng'),
    '62' => t('Gansu Sheng'),
    '44' => t('Guangdong Sheng'),
    '45' => t('Guangxi Zhuangzuzizhiqu'),
    '52' => t('Guizhou Sheng'),
    '46' => t('Hainan Sheng'),
    '13' => t('Hebei Sheng'),
    '41' => t('Henan Sheng'),
    '23' => t('Heilongjiang Sheng'),
    '42' => t('Hubei Sheng'),
    '43' => t('Hunan Sheng'),
    '22' => t('Jilin Sheng'),
    '32' => t('Jiangsu Sheng'),
    '36' => t('Jiangxi Sheng'),
    '21' => t('Liaoning Sheng'),
    '15' => t('Neimenggu Zizhiqu'),
    '64' => t('Ningxia Huizuzizhiqu'),
    '63' => t('Qinghai Sheng'),
    '37' => t('Shandong Sheng'),
    '14' => t('Shanxi Sheng'),
    '61' => t('Shaanxi Sheng'),
    '31' => t('Shanghai Shi'),
    '51' => t('Sichuan Sheng'),
    '71' => t('Taiwan'),
    '12' => t('Tianjin Shi'),
    '54' => t('Xizang Zizhiqu'),
    '91' => t('Hong Kong'),
    '65' => t('Xinjiang Weiwuerzizhiqu'),
    '53' => t('Yunnan Sheng'),
    '33' => t('Zhejiang Sheng'),
  );
  $administrative_areas['CO'] = array(
    'AMA' => 'Amazonas',
    'ANT' => 'Antioquia',
    'ARA' => 'Arauca',
    'ATL' => 'Atlántico',
    'BOL' => 'Bolívar',
    'BOY' => 'Boyacá',
    'CAL' => 'Caldas',
    'CAQ' => 'Caquetá',
    'CAS' => 'Casanare',
    'CAU' => 'Cauca',
    'CES' => 'Cesar',
    'COR' => 'Córdoba',
    'CUN' => 'Cundinamarca',
    'CHO' => 'Chocó',
    'GUA' => 'Guainía',
    'GUV' => 'Guaviare',
    'HUI' => 'Huila',
    'LAG' => 'La Guajira',
    'MAG' => 'Magdalena',
    'MET' => 'Meta',
    'NAR' => 'Nariño',
    'NSA' => 'Norte de Santander',
    'PUT' => 'Putumayo',
    'QUI' => 'Quindío',
    'RIS' => 'Risaralda',
    'SAP' => 'San Andrés, Providencia y Santa Catalina',
    'SAN' => 'Santander',
    'SUC' => 'Sucre',
    'TOL' => 'Tolima',
    'VAC' => 'Valle del Cauca',
    'VAU' => 'Vaupés',
    'VID' => 'Vichada',
  );
  $administrative_areas['EE'] = array(
    '37' => 'Harjumaa',
    '39' => 'Hiiumaa',
    '44' => 'Ida-Virumaa',
    '49' => 'Jõgevamaa',
    '51' => 'Järvamaa',
    '57' => 'Läänemaa',
    '59' => 'Lääne-Virumaa',
    '65' => 'Põlvamaa',
    '67' => 'Pärnumaa',
    '70' => 'Raplamaa',
    '74' => 'Saaremaa',
    '78' => 'Tartumaa',
    '82' => 'Valgamaa',
    '84' => 'Viljandimaa',
    '86' => 'Võrumaa',
  );
  $administrative_areas['EG'] = array(
    'ALX' => t('Alexandria'),
    'ASN' => t('Aswan'),
    'AST' => t('Asyut'),
    'BH' => t('Beheira'),
    'BNS' => t('Beni Suef'),
    'C' => t('Cairo'),
    'DK' => t('Dakahlia'),
    'DT' => t('Damietta'),
    'FYM' => t('Faiyum'),
    'GH' => t('Gharbia'),
    'GZ' => t('Giza'),
    'IS' => t('Ismailia'),
    'KFS' => t('Kafr el-Sheikh'),
    'MT' => t('Matruh'),
    'MN' => t('Minya'),
    'MNF' => t('Monufia'),
    'WAD' => t('New Valley'),
    'SIN' => t('North Sinai'),
    'PTS' => t('Port Said'),
    'KB' => t('Qalyubia'),
    'KN' => t('Qena'),
    'BA' => t('Red Sea'),
    'SHR' => t('Sharqia'),
    'SHG' => t('Sohag'),
    'JS' => t('South Sinai'),
    'SUZ' => t('Suez'),
    'LX' => t('Luxor'),
  );
  $administrative_areas['ES'] = array(
    'C' => "A Coruña",
    'VI' => 'Alava',
    'AB' => 'Albacete',
    'A' => 'Alicante',
    'AL' => "Almería",
    'O' => 'Asturias',
    'AV' => "Ávila",
    'BA' => 'Badajoz',
    'PM' => 'Baleares',
    'B' => 'Barcelona',
    'BU' => 'Burgos',
    'CC' => "Cáceres",
    'CA' => "Cádiz",
    'S' => 'Cantabria',
    'CS' => "Castellón",
    'CE' => 'Ceuta',
    'CR' => 'Ciudad Real',
    'CO' => "Córdoba",
    'CU' => 'Cuenca',
    'GI' => 'Girona',
    'GR' => 'Granada',
    'GU' => 'Guadalajara',
    'SS' => "Guipúzcoa",
    'H' => 'Huelva',
    'HU' => 'Huesca',
    'J' => "Jaén",
    'LO' => 'La Rioja',
    'GC' => 'Las Palmas',
    'LE' => "León",
    'L' => "Lleida",
    'LU' => 'Lugo',
    'M' => 'Madrid',
    'MA' => "Málaga",
    'ML' => 'Melilla',
    'MU' => 'Murcia',
    'NA' => 'Navarra',
    'OR' => 'Ourense',
    'P' => 'Palencia',
    'PO' => 'Pontevedra',
    'SA' => 'Salamanca',
    'TF' => 'Santa Cruz de Tenerife',
    'SG' => 'Segovia',
    'SE' => 'Sevilla',
    'SO' => 'Soria',
    'T' => 'Tarragona',
    'TE' => 'Teruel',
    'TO' => 'Toledo',
    'V' => 'Valencia',
    'VA' => 'Valladolid',
    'BI' => 'Vizcaya',
    'ZA' => 'Zamora',
    'Z' => 'Zaragoza',
  );
  $administrative_areas['HK'] = array(
    // HK subdivisions have no ISO codes assigned.
    'Kowloon' => t('Kowloon'),
    'Hong Kong Island' => t('Hong Kong Island'),
    'New Territories' => t('New Territories'),
  );
  $administrative_areas['ID'] = array(
    'AC' => t('Aceh'),
    'BA' => t('Bali'),
    'BB' => t('Bangka Belitung'),
    'BT' => t('Banten'),
    'BE' => t('Bengkulu'),
    'JK' => t('DKI Jakarta'),
    'YO' => t('D.I. Yogyakarta'),
    'GO' => t('Gorontalo'),
    'JA' => t('Jambi'),
    'JB' => t('Jawa Barat'),
    'JT' => t('Jawa Tengah'),
    'JI' => t('Jawa Timur'),
    'KB' => t('Kalimantan Barat'),
    'KS' => t('Kalimantan Selatan'),
    'KT' => t('Kalimantan Tengah'),
    'KI' => t('Kalimantan Timur'),
    'KR' => t('Kepulauan Riau'),
    'LA' => t('Lampung'),
    'MA' => t('Maluku'),
    'MU' => t('Maluku Utara'),
    'NB' => t('Nusa Tenggara Barat'),
    'NT' => t('Nusa Tenggara Timur'),
    'PA' => t('Papua'),
    'PB' => t('Papua Barat'),
    'RI' => t('Riau'),
    'SR' => t('Sulawesi Barat'),
    'SN' => t('Sulawesi Selatan'),
    'ST' => t('Sulawesi Tengah'),
    'SG' => t('Sulawesi Tenggara'),
    'SA' => t('Sulawesi Utara'),
    'SB' => t('Sumatera Barat'),
    'SS' => t('Sumatera Selatan'),
    'SU' => t('Sumatera Utara'),
  );
  $administrative_areas['IE'] = array(
    'CW' => 'Co Carlow',
    'CN' => 'Co Cavan',
    'CE' => 'Co Clare',
    'CO' => 'Co Cork',
    'DL' => 'Co Donegal',
    'D' => 'Co Dublin',
    'D1' => 'Dublin 1',
    'D2' => 'Dublin 2',
    'D3' => 'Dublin 3',
    'D4' => 'Dublin 4',
    'D5' => 'Dublin 5',
    'D6' => 'Dublin 6',
    'D6W' => 'Dublin 6w',
    'D7' => 'Dublin 7',
    'D8' => 'Dublin 8',
    'D9' => 'Dublin 9',
    'D10' => 'Dublin 10',
    'D11' => 'Dublin 11',
    'D12' => 'Dublin 12',
    'D13' => 'Dublin 13',
    'D14' => 'Dublin 14',
    'D15' => 'Dublin 15',
    'D16' => 'Dublin 16',
    'D17' => 'Dublin 17',
    'D18' => 'Dublin 18',
    'D19' => 'Dublin 19',
    'D20' => 'Dublin 20',
    'D21' => 'Dublin 21',
    'D22' => 'Dublin 22',
    'D23' => 'Dublin 23',
    'D24' => 'Dublin 24',
    'G' => 'Co Galway',
    'KY' => 'Co Kerry',
    'KE' => 'Co Kildare',
    'KK' => 'Co Kilkenny',
    'LS' => 'Co Laois',
    'LM' => 'Co Leitrim',
    'LK' => 'Co Limerick',
    'LD' => 'Co Longford',
    'LH' => 'Co Louth',
    'MO' => 'Co Mayo',
    'MH' => 'Co Meath',
    'MN' => 'Co Monaghan',
    'OY' => 'Co Offaly',
    'RN' => 'Co Roscommon',
    'SO' => 'Co Sligo',
    'TA' => 'Co Tipperary',
    'WD' => 'Co Waterford',
    'WH' => 'Co Westmeath',
    'WX' => 'Co Wexford',
    'WW' => 'Co Wicklow',
  );
  $administrative_areas['IN'] = array(
    'AP' => t('Andhra Pradesh'),
    'AR' => t('Arunachal Pradesh'),
    'AS' => t('Assam'),
    'BR' => t('Bihar'),
    'CT' => t('Chhattisgarh'),
    'GA' => t('Goa'),
    'GJ' => t('Gujarat'),
    'HP' => t('Himachal Pradesh'),
    'HR' => t('Haryana'),
    'JH' => t('Jharkhand'),
    'JK' => t('Jammu & Kashmir'),
    'KA' => t('Karnataka'),
    'KL' => t('Kerala'),
    'MH' => t('Maharashtra'),
    'MN' => t('Manipur'),
    'ML' => t('Meghalaya'),
    'MP' => t('Madhya Pradesh'),
    'MZ' => t('Mizoram'),
    'NL' => t('Nagaland'),
    'OR' => t('Odisha'),
    'PB' => t('Punjab'),
    'RJ' => t('Rajasthan'),
    'SK' => t('Sikkim'),
    'TN' => t('Tamil Nadu'),
    'TG' => t('Telangana'),
    'TR' => t('Tripura'),
    'UP' => t('Uttar Pradesh'),
    'UT' => t('Uttarakhand'),
    'WB' => t('West Bengal'),
    ' ' => t('--'),
    'AN' => t('Andaman & Nicobar'),
    'CH' => t('Chandigarh'),
    'DN' => t('Dadra & Nagar Haveli'),
    'DD' => t('Daman & Diu'),
    'DL' => t('Delhi'),
    'LD' => t('Lakshadweep'),
    'PY' => t('Puducherry'),
  );
  $administrative_areas['IT'] = array(
    'AG' => 'Agrigento',
    'AL' => 'Alessandria',
    'AN' => 'Ancona',
    'AO' => 'Aosta',
    'AP' => 'Ascoli Piceno',
    'AQ' => "L'Aquila",
    'AR' => 'Arezzo',
    'AT' => 'Asti',
    'AV' => 'Avellino',
    'BA' => 'Bari',
    'BG' => 'Bergamo',
    'BI' => 'Biella',
    'BL' => 'Belluno',
    'BN' => 'Benevento',
    'BO' => 'Bologna',
    'BR' => 'Brindisi',
    'BS' => 'Brescia',
    'BT' => 'Barletta-Andria-Trani',
    'BZ' => 'Bolzano/Bozen',
    'CA' => 'Cagliari',
    'CB' => 'Campobasso',
    'CE' => 'Caserta',
    'CH' => 'Chieti',
    'CI' => 'Carbonia-Iglesias',
    'CL' => 'Caltanissetta',
    'CN' => 'Cuneo',
    'CO' => 'Como',
    'CR' => 'Cremona',
    'CS' => 'Cosenza',
    'CT' => 'Catania',
    'CZ' => 'Catanzaro',
    'EN' => 'Enna',
    'FC' => 'Forlì-Cesena',
    'FE' => 'Ferrara',
    'FG' => 'Foggia',
    'FI' => 'Firenze',
    'FM' => 'Fermo',
    'FR' => 'Frosinone',
    'GE' => 'Genova',
    'GO' => 'Gorizia',
    'GR' => 'Grosseto',
    'IM' => 'Imperia',
    'IS' => 'Isernia',
    'KR' => 'Crotone',
    'LC' => 'Lecco',
    'LE' => 'Lecce',
    'LI' => 'Livorno',
    'LO' => 'Lodi',
    'LT' => 'Latina',
    'LU' => 'Lucca',
    'MB' => 'Monza e Brianza',
    'MC' => 'Macerata',
    'ME' => 'Messina',
    'MI' => 'Milano',
    'MN' => 'Mantova',
    'MO' => 'Modena',
    'MS' => 'Massa-Carrara',
    'MT' => 'Matera',
    'NA' => 'Napoli',
    'NO' => 'Novara',
    'NU' => 'Nuoro',
    'OG' => 'Ogliastra',
    'OR' => 'Oristano',
    'OT' => 'Olbia-Tempio',
    'PA' => 'Palermo',
    'PC' => 'Piacenza',
    'PD' => 'Padova',
    'PE' => 'Pescara',
    'PG' => 'Perugia',
    'PI' => 'Pisa',
    'PN' => 'Pordenone',
    'PO' => 'Prato',
    'PR' => 'Parma',
    'PT' => 'Pistoia',
    'PU' => 'Pesaro e Urbino',
    'PV' => 'Pavia',
    'PZ' => 'Potenza',
    'RA' => 'Ravenna',
    'RC' => 'Reggio Calabria',
    'RE' => 'Reggio Emilia',
    'RG' => 'Ragusa',
    'RI' => 'Rieti',
    'RM' => 'Roma',
    'RN' => 'Rimini',
    'RO' => 'Rovigo',
    'SA' => 'Salerno',
    'SI' => 'Siena',
    'SO' => 'Sondrio',
    'SP' => 'La Spezia',
    'SR' => 'Siracusa',
    'SS' => 'Sassari',
    'SV' => 'Savona',
    'TA' => 'Taranto',
    'TE' => 'Teramo',
    'TN' => 'Trento',
    'TO' => 'Torino',
    'TP' => 'Trapani',
    'TR' => 'Terni',
    'TS' => 'Trieste',
    'TV' => 'Treviso',
    'UD' => 'Udine',
    'VA' => 'Varese',
    'VB' => 'Verbano-Cusio-Ossola',
    'VC' => 'Vercelli',
    'VE' => 'Venezia',
    'VI' => 'Vicenza',
    'VR' => 'Verona',
    'VS' => 'Medio Campidano',
    'VT' => 'Viterbo',
    'VV' => 'Vibo Valentia',
  );
  $administrative_areas['JM'] = array(
    '13' => 'Clarendon',
    '09' => 'Hanover',
    '01' => 'Kingston',
    '12' => 'Manchester',
    '04' => 'Portland',
    '02' => 'St. Andrew',
    '06' => 'St. Ann',
    '14' => 'St. Catherine',
    '11' => 'St. Elizabeth',
    '08' => 'St. James',
    '05' => 'St. Mary',
    '03' => 'St. Thomas',
    '07' => 'Trelawny',
    '10' => 'Westmoreland',
  );
  $administrative_areas['JP'] = array(
    '01' => t('Hokkaido'),
    '02' => t('Aomori'),
    '03' => t('Iwate'),
    '04' => t('Miyagi'),
    '05' => t('Akita'),
    '06' => t('Yamagata'),
    '07' => t('Fukushima'),
    '08' => t('Ibaraki'),
    '09' => t('Tochigi'),
    '10' => t('Gunma'),
    '11' => t('Saitama'),
    '12' => t('Chiba'),
    '13' => t('Tokyo'),
    '14' => t('Kanagawa'),
    '15' => t('Niigata'),
    '16' => t('Toyama'),
    '17' => t('Ishikawa'),
    '18' => t('Fukui'),
    '19' => t('Yamanashi'),
    '20' => t('Nagano'),
    '21' => t('Gifu'),
    '22' => t('Shizuoka'),
    '23' => t('Aichi'),
    '24' => t('Mie'),
    '25' => t('Shiga'),
    '26' => t('Kyoto'),
    '27' => t('Osaka'),
    '28' => t('Hyogo'),
    '29' => t('Nara'),
    '30' => t('Wakayama'),
    '31' => t('Tottori'),
    '32' => t('Shimane'),
    '33' => t('Okayama'),
    '34' => t('Hiroshima'),
    '35' => t('Yamaguchi'),
    '36' => t('Tokushima'),
    '37' => t('Kagawa'),
    '38' => t('Ehime'),
    '39' => t('Kochi'),
    '40' => t('Fukuoka'),
    '41' => t('Saga'),
    '42' => t('Nagasaki'),
    '43' => t('Kumamoto'),
    '44' => t('Oita'),
    '45' => t('Miyazaki'),
    '46' => t('Kagoshima'),
    '47' => t('Okinawa'),
  );
  $administrative_areas['KR'] = array(
    '11' => t('Seoul'),
    '26' => t('Busan'),
    '27' => t('Daegu'),
    '30' => t('Daejeon'),
    '29' => t('Gwangju'),
    '28' => t('Incheon'),
    '31' => t('Ulsan'),
    '43' => t('Chungcheongbuk-do'),
    '44' => t('Chungcheongnam-do'),
    '42' => t('Gangwon-do'),
    '41' => t('Gyeonggi-do'),
    '47' => t('Gyeongsangbuk-do'),
    '48' => t('Gyeongsangnam-do'),
    '49' => t('Jeju-do'),
    '45' => t('Jeollabuk-do'),
    '46' => t('Jeollanam-do'),
    '50' => t('Sejong'),
  );
  $administrative_areas['KZ'] = array(
    'AST' => t('Astana'),
    'ALA' => t('Almaty'),
    'ALM' => t('Almaty region'),
    'AKM' => t('Aqmola region'),
    'AKT' => t('Aqtöbe region'),
    'ATY' => t('Atyraū region'),
    'ZAP' => t('Batys Qazaqstan region'),
    'MAN' => t('Mangghystaū region'),
    'YUZ' => t('Ongtüstik Qazaqstan region'),
    'PAV' => t('Pavlodar region'),
    'KAR' => t('Qaraghandy region'),
    'KUS' => t('Qostanay region'),
    'KZY' => t('Qyzylorda region'),
    'VOS' => t('Shyghys Qazaqstan region'),
    'SEV' => t('Soltüstik Qazaqstan region'),
    'ZHA' => t('Zhambyl region'),
  );
  $administrative_areas['MX'] = array(
    'AGU' => 'Aguascalientes',
    'BCN' => 'Baja California',
    'BCS' => 'Baja California Sur',
    'CAM' => 'Campeche',
    'COA' => 'Coahuila',
    'COL' => 'Colima',
    'CHP' => 'Chiapas',
    'CHH' => 'Chihuahua',
    'DIF' => 'Distrito Federal',
    'DUG' => 'Durango',
    'MEX' => 'Estado de México',
    'GUA' => 'Guanajuato',
    'GRO' => 'Guerrero',
    'HID' => 'Hidalgo',
    'JAL' => 'Jalisco',
    'MIC' => 'Michoacán',
    'MOR' => 'Morelos',
    'NAY' => 'Nayarit',
    'NLE' => 'Nuevo León',
    'OAX' => 'Oaxaca',
    'PUE' => 'Puebla',
    'QUE' => 'Queretaro',
    'ROO' => 'Quintana Roo',
    'SLP' => 'San Luis Potosí',
    'SIN' => 'Sinaloa',
    'SON' => 'Sonora',
    'TAB' => 'Tabasco',
    'TAM' => 'Tamaulipas',
    'TLA' => 'Tlaxcala',
    'VER' => 'Veracruz',
    'YUC' => 'Yucatán',
    'ZAC' => 'Zacatecas',
  );
  $administrative_areas['MY'] = array(
    '01' => t('Johor'),
    '02' => t('Kedah'),
    '03' => t('Kelantan'),
    '14' => t('Kuala Lumpur'),
    '15' => t('Labuan'),
    '04' => t('Melaka'),
    '05' => t('Negeri Sembilan'),
    '06' => t('Pahang'),
    '08' => t('Perak'),
    '09' => t('Perlis'),
    '07' => t('Pulau Pinang'),
    '16' => t('Putrajaya'),
    '12' => t('Sabah'),
    '13' => t('Sarawak'),
    '10' => t('Selangor'),
    '11' => t('Terengganu'),
    );
  $administrative_areas['PE'] = array(
    'AMA' => 'Amazonas',
    'ANC' => 'Ancash',
    'APU' => 'Apurimac',
    'ARE' => 'Arequipa',
    'AYA' => 'Ayacucho',
    'CAJ' => 'Cajamarca',
    'CAL' => 'Callao',
    'CUS' => 'Cusco',
    'HUV' => 'Huancavelica',
    'HUC' => 'Huanuco',
    'ICA' => 'Ica',
    'JUN' => 'Junin',
    'LAL' => 'La Libertad',
    'LAM' => 'Lambayeque',
    'LIM' => 'Lima',
    'LOR' => 'Loreto',
    'MDD' => 'Madre de Dios',
    'MOQ' => 'Moquegua',
    'PAS' => 'Pasco',
    'PIU' => 'Piura',
    'PUN' => 'Puno',
    'SAM' => 'San Martin',
    'TAC' => 'Tacna',
    'TUM' => 'Tumbes',
    'UCA' => 'Ucayali',
  );
  $administrative_areas['RU'] = array(
    'MOW' => t('Moskva'),
    'SPE' => t('Sankt-Peterburg'),
    'AD' => t('Adygeya, Respublika'),
    'AL' => t('Altay, Respublika'),
    'BA' => t('Bashkortostan, Respublika'),
    'BU' => t('Buryatiya, Respublika'),
    'CE' => t('Chechenskaya Respublika'),
    'CU' => t('Chuvashskaya Respublika'),
    'DA' => t('Dagestan, Respublika'),
    'IN' => t('Ingushetiya, Respublika'),
    'KB' => t('Kabardino-Balkarskaya Respublika'),
    'KL' => t('Kalmykiya, Respublika'),
    'KC' => t('Karachayevo-Cherkesskaya Respublika'),
    'KR' => t('Kareliya, Respublika'),
    'KK' => t('Khakasiya, Respublika'),
    'KO' => t('Komi, Respublika'),
    'ME' => t('Mariy El, Respublika'),
    'MO' => t('Mordoviya, Respublika'),
    'SA' => t('Sakha, Respublika [Yakutiya]'),
    'SE' => t('Severnaya Osetiya-Alaniya, Respublika'),
    'TA' => t('Tatarstan, Respublika'),
    'TY' => t('Tyva, Respublika [Tuva]'),
    'UD' => t('Udmurtskaya Respublika'),
    'ALT' => t('Altayskiy kray'),
    'KAM' => t('Kamchatskiy kray'),
    'KHA' => t('Khabarovskiy kray'),
    'KDA' => t('Krasnodarskiy kray'),
    'KYA' => t('Krasnoyarskiy kray'),
    'PER' => t('Permskiy kray'),
    'PRI' => t('Primorskiy kray'),
    'STA' => t('Stavropolskiy kray'),
    'ZAB' => t('Zabaykalskiy kray'),
    'AMU' => t('Amurskaya oblast'),
    'ARK' => t('Arkhangelskaya oblast'),
    'AST' => t('Astrakhanskaya oblast'),
    'BEL' => t('Belgorodskaya oblast'),
    'BRY' => t('Bryanskaya oblast'),
    'CHE' => t('Chelyabinskaya oblast'),
    'IRK' => t('Irkutskaya oblast'),
    'IVA' => t('Ivanovskaya oblast'),
    'KGD' => t('Kaliningradskaya oblast'),
    'KLU' => t('Kaluzhskaya oblast'),
    'KEM' => t('Kemerovskaya oblast'),
    'KIR' => t('Kirovskaya oblast'),
    'KOS' => t('Kostromskaya oblast'),
    'KGN' => t('Kurganskaya oblast'),
    'KRS' => t('Kurskaya oblast'),
    'LEN' => t('Leningradskaya oblast'),
    'LIP' => t('Lipetskaya oblast'),
    'MAG' => t('Magadanskaya oblast'),
    'MOS' => t('Moskovskaya oblast'),
    'MUR' => t('Murmanskaya oblast'),
    'NIZ' => t('Nizhegorodskaya oblast'),
    'NGR' => t('Novgorodskaya oblast'),
    'NVS' => t('Novosibirskaya oblast'),
    'OMS' => t('Omskaya oblast'),
    'ORE' => t('Orenburgskaya oblast'),
    'ORL' => t('Orlovskaya oblast'),
    'PNZ' => t('Penzenskaya oblast'),
    'PSK' => t('Pskovskaya oblast'),
    'ROS' => t('Rostovskaya oblast'),
    'RYA' => t('Ryazanskaya oblast'),
    'SAK' => t('Sakhalinskaya oblast'),
    'SAM' => t('Samarskaya oblast'),
    'SAR' => t('Saratovskaya oblast'),
    'SMO' => t('Smolenskaya oblast'),
    'SVE' => t('Sverdlovskaya oblast'),
    'TAM' => t('Tambovskaya oblast'),
    'TOM' => t('Tomskaya oblast'),
    'TUL' => t('Tulskaya oblast'),
    'TVE' => t('Tverskaya oblast'),
    'TYU' => t('Tyumenskaya oblast'),
    'ULY' => t('Ulyanovskaya oblast'),
    'VLA' => t('Vladimirskaya oblast'),
    'VGG' => t('Volgogradskaya oblast'),
    'VLG' => t('Vologodskaya oblast'),
    'VOR' => t('Voronezhskaya oblast'),
    'YAR' => t('Yaroslavskaya oblast'),
    'YEV' => t('Yevreyskaya avtonomnaya oblast'),
    'CHU' => t('Chukotskiy avtonomnyy okrug'),
    'KHM' => t('Khanty-Mansiyskiy avtonomnyy okrug-Yugra'),
    'NEN' => t('Nenetskiy avtonomnyy okrug'),
    'YAN' => t('Yamalo-Nenetskiy avtonomnyy okrug'),
  );
  $administrative_areas['TR'] = array(
    '01' => t('Adana'),
    '02' => t('Adıyaman'),
    '03' => t('Afyon'),
    '04' => t('Ağrı'),
    '68' => t('Aksaray'),
    '05' => t('Amasya'),
    '06' => t('Ankara'),
    '07' => t('Antalya'),
    '75' => t('Ardahan'),
    '08' => t('Artvin'),
    '09' => t('Aydın'),
    '10' => t('Balıkesir'),
    '74' => t('Bartın'),
    '72' => t('Batman'),
    '69' => t('Bayburt'),
    '11' => t('Bilecik'),
    '12' => t('Bingöl'),
    '13' => t('Bitlis'),
    '14' => t('Bolu'),
    '15' => t('Burdur'),
    '16' => t('Bursa'),
    '17' => t('Çanakkale'),
    '18' => t('Çankırı'),
    '19' => t('Çorum'),
    '20' => t('Denizli'),
    '21' => t('Diyarbakır'),
    '81' => t('Düzce'),
    '22' => t('Edirne'),
    '23' => t('Elazığ'),
    '24' => t('Erzincan'),
    '25' => t('Erzurum'),
    '26' => t('Eskişehir'),
    '27' => t('Gaziantep'),
    '28' => t('Giresun'),
    '29' => t('Gümüşhane'),
    '30' => t('Hakkari'),
    '31' => t('Hatay'),
    '76' => t('Iğdır'),
    '32' => t('Isparta'),
    '34' => t('İstanbul'),
    '35' => t('İzmir'),
    '46' => t('Kahramanmaraş'),
    '78' => t('Karabük'),
    '70' => t('Karaman'),
    '36' => t('Kars'),
    '37' => t('Kastamonu'),
    '38' => t('Kayseri'),
    '71' => t('Kırıkkale'),
    '39' => t('Kırklareli'),
    '40' => t('Kırşehir'),
    '79' => t('Kilis'),
    '41' => t('Kocaeli'),
    '42' => t('Konya'),
    '43' => t('Kütahya'),
    '44' => t('Malatya'),
    '45' => t('Manisa'),
    '47' => t('Mardin'),
    '33' => t('Mersin'),
    '48' => t('Muğla'),
    '49' => t('Muş'),
    '50' => t('Nevşehir'),
    '51' => t('Niğde'),
    '52' => t('Ordu'),
    '80' => t('Osmaniye'),
    '53' => t('Rize'),
    '54' => t('Sakarya'),
    '55' => t('Samsun'),
    '56' => t('Siirt'),
    '57' => t('Sinop'),
    '58' => t('Sivas'),
    '63' => t('Şanlıurfa'),
    '73' => t('Şırnak'),
    '59' => t('Tekirdağ'),
    '60' => t('Tokat'),
    '61' => t('Trabzon'),
    '62' => t('Tunceli'),
    '64' => t('Uşak'),
    '65' => t('Van'),
    '77' => t('Yalova'),
    '66' => t('Yozgat'),
    '67' => t('Zonguldak'),
  );
  $administrative_areas['TW'] = array(
    'TXG' => t('Taichung City'),
    'TPE' => t('Taipei City'),
    'TTT' => t('Taitung County'),
    'TNN' => t('Tainan City'),
    'ILA' => t('Yilan County'),
    'HUA' => t('Hualien County'),
    'Kinmen County' => t('Kinmen County'),
    'NAN' => t('Nantou County'),
    'PIF' => t('Pingtung County'),
    'MIA' => t('Miaoli County'),
    'TAO' => t('Taoyuan County'),
    'KHH' => t('Kaohsiung City'),
    'KEE' => t('Keelung City'),
    'Lienchiang County' => t('Lienchiang County'),
    'YUN' => t('Yunlin County'),
    'TPQ' => t('New Taipei City'),
    'HSZ' => t('Hsinchu City'),
    'HSQ' => t('Hsinchu County'),
    'CYI' => t('Chiayi City'),
    'CYQ' => t('Chiayi County'),
    'CHA' => t('Changhua County'),
    'PEN' => t('Penghu County'),
  );
  $administrative_areas['UA'] = array(
    '43' => t('Crimea'),
    '05' => t('Vinnyts\'ka oblast'),
    '07' => t('Volyns\'ka oblast'),
    '12' => t('Dnipropetrovsk Oblast'),
    '14' => t('Donetsk Oblast'),
    '18' => t('Zhytomyrs\'ka oblast'),
    '21' => t('Zakarpats\'ka oblast'),
    '23' => t('Zaporiz\'ka oblast'),
    '26' => t('Ivano-Frankivs\'ka oblast'),
    '30' => t('Kiev Oblast'),
    '35' => t('Kirovohrads\'ka oblast'),
    '09' => t('Luhans\'ka oblast'),
    '46' => t('Lviv Oblast'),
    '48' => t('Mykolaivs\'ka oblast'),
    '51' => t('Odessa Oblast'),
    '53' => t('Poltavs\'ka oblast'),
    '56' => t('Rivnens\'ka oblast'),
    '40' => t('Sevastopol\' city'),
    '59' => t('Sums\'ka oblast'),
    '61' => t('Ternopil\'s\'ka oblast'),
    '63' => t('Kharkiv Oblast'),
    '65' => t('Khersons\'ka oblast'),
    '68' => t('Khmel\'nyts\'ka oblast'),
    '71' => t('Cherkas\'ka oblast'),
    '77' => t('Chernivets\'ka oblast'),
    '74' => t('Chernihivs\'ka oblast'),
  );
  $administrative_areas['US'] = array(
    'AL' => t('Alabama'),
    'AK' => t('Alaska'),
    'AZ' => t('Arizona'),
    'AR' => t('Arkansas'),
    'CA' => t('California'),
    'CO' => t('Colorado'),
    'CT' => t('Connecticut'),
    'DE' => t('Delaware'),
    'DC' => t('District of Columbia'),
    'FL' => t('Florida'),
    'GA' => t('Georgia'),
    'HI' => t('Hawaii'),
    'ID' => t('Idaho'),
    'IL' => t('Illinois'),
    'IN' => t('Indiana'),
    'IA' => t('Iowa'),
    'KS' => t('Kansas'),
    'KY' => t('Kentucky'),
    'LA' => t('Louisiana'),
    'ME' => t('Maine'),
    'MD' => t('Maryland'),
    'MA' => t('Massachusetts'),
    'MI' => t('Michigan'),
    'MN' => t('Minnesota'),
    'MS' => t('Mississippi'),
    'MO' => t('Missouri'),
    'MT' => t('Montana'),
    'NE' => t('Nebraska'),
    'NV' => t('Nevada'),
    'NH' => t('New Hampshire'),
    'NJ' => t('New Jersey'),
    'NM' => t('New Mexico'),
    'NY' => t('New York'),
    'NC' => t('North Carolina'),
    'ND' => t('North Dakota'),
    'OH' => t('Ohio'),
    'OK' => t('Oklahoma'),
    'OR' => t('Oregon'),
    'PA' => t('Pennsylvania'),
    'RI' => t('Rhode Island'),
    'SC' => t('South Carolina'),
    'SD' => t('South Dakota'),
    'TN' => t('Tennessee'),
    'TX' => t('Texas'),
    'UT' => t('Utah'),
    'VT' => t('Vermont'),
    'VA' => t('Virginia'),
    'WA' => t('Washington'),
    'WV' => t('West Virginia'),
    'WI' => t('Wisconsin'),
    'WY' => t('Wyoming'),
    ' ' => t('--'),
    'AA' => t('Armed Forces (Americas)'),
    'AE' => t('Armed Forces (Europe, Canada, Middle East, Africa)'),
    'AP' => t('Armed Forces (Pacific)'),
    'AS' => t('American Samoa'),
    'FM' => t('Federated States of Micronesia'),
    'GU' => t('Guam'),
    'MH' => t('Marshall Islands'),
    'MP' => t('Northern Mariana Islands'),
    'PW' => t('Palau'),
    'PR' => t('Puerto Rico'),
    'VI' => t('Virgin Islands'),
  );
  $administrative_areas['VE'] = array(
    'Z' => 'Amazonas',
    'B' => 'Anzoátegui',
    'C' => 'Apure',
    'D' => 'Aragua',
    'E' => 'Barinas',
    'F' => 'Bolívar',
    'G' => 'Carabobo',
    'H' => 'Cojedes',
    'Y' => 'Delta Amacuro',
    'W' => 'Dependencias Federales',
    'A' => 'Distrito Federal',
    'I' => 'Falcón',
    'J' => 'Guárico',
    'K' => 'Lara',
    'L' => 'Mérida',
    'M' => 'Miranda',
    'N' => 'Monagas',
    'O' => 'Nueva Esparta',
    'P' => 'Portuguesa',
    'R' => 'Sucre',
    'S' => 'Táchira',
    'T' => 'Trujillo',
    'X' => 'Vargas',
    'U' => 'Yaracuy',
    'V' => 'Zulia',
  );

  return $administrative_areas;
}
