<?php

use CorpusParole\Models\Document;
use CorpusParole\Libraries\CocoonUtils;

use EasyRdf\RdfNamespace;

/**
 *
 */
class DocumentTest extends TestCase {

    const TEST_INPUT_DOCS = [
        'TEST' => __DIR__.'/files/DocumentTest/test_doc.ttl',
        'TEST_NO_GEOINFO' => __DIR__.'/files/DocumentTest/test_no_geoinfo.ttl',
    ];

    private $inputGraphes = [];

    public function setUp() {
        parent::setup();

        foreach(self::TEST_INPUT_DOCS as $key => $inputDoc) {
            $this->inputGraphes[$key] =  new EasyRdf\Graph(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", sprintf(file_get_contents($inputDoc), config('corpusparole.corpus_doc_id_base_uri'), config('corpusparole.corpus_id_scheme')));
        }


    }

    public function testConstructor() {
        $this->assertNotNull($this->inputGraphes['TEST'], 'Graph shoud not be null');

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $this->assertEquals(config('corpusparole.corpus_id_scheme').'crdo-CFPP2000_35_SOUNDid',$doc->getId(),'Must have the correct id');
    }

    public function testTitle() {
        $this->assertNotNull($this->inputGraphes['TEST'], 'Graph shoud not be null');

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $this->assertEquals("Entretien de Ozgur Kiliç 2",$doc->getTitle(),'Must have correct title');
        $this->assertInstanceOf(EasyRdf\Literal::class, $doc->getTitle(), "Title must be a literal");
        $this->assertEquals('fr', $doc->getTitle()->getLang(), "Language title must be fr");
    }


    public function testModified() {
        $this->assertNotNull($this->inputGraphes['TEST'], 'Graph shoud not be null');

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);
        $this->assertInstanceOf(EasyRdf\Literal::class, $doc->getModified(), "Modified must be a literal");
        $this->assertEquals("http://purl.org/dc/terms/W3CDTF", $doc->getModified()->getDatatypeURI(), "type must be http://purl.org/dc/terms/W3CDTF");
        $this->assertEquals("2013-10-12T14:35:57+02:00", $doc->getModified(), "modified must be 2013-10-12T14:35:57+02:00");

    }

    public function testSetModified() {
        $currentTime = gmdate(DateTime::ATOM);
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);
        $doc->setModified($currentTime);
        $this->assertInstanceOf(EasyRdf\Literal::class, $doc->getModified(), "Modified must be a literal");
        $this->assertEquals("http://purl.org/dc/terms/W3CDTF", $doc->getModified()->getDatatypeURI(), "type must be http://purl.org/dc/terms/W3CDTF");
        $this->assertEquals(preg_replace('/[\+\-]00(\:?)00$/', 'Z', $currentTime), $doc->getModified()->getValue(), "modified must be $currentTime");
    }

    public function testSetModifiedNull() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);
        $doc->setModified();
        $this->assertInstanceOf(EasyRdf\Literal::class, $doc->getModified(), "Modified must be a literal");
        $this->assertEquals("http://purl.org/dc/terms/W3CDTF", $doc->getModified()->getDatatypeURI(), "type must be http://purl.org/dc/terms/W3CDTF");
        $foundDateTime = \DateTime::createFromFormat(\DateTime::ATOM, $doc->getModified()->getValue());
        $nowUTC = new \DateTime('now', new \DateTimeZone("UTC"));
        $this->assertTrue($nowUTC->getTimestamp() -  $foundDateTime->getTimestamp() < 2, "must have less than 2 seconds diff");
    }

    public function testPublisher() {

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $this->assertCount(1, $doc->getPublishers(), 'Publisher is an array of size 1');
        $this->assertInstanceOf('EasyRdf\Resource', $doc->getPublishers()[0], 'publisher is a resource');
        $this->assertEquals("http://viaf.org/viaf/142432638", $doc->getPublishers()[0]->getUri(),"");
    }

    public function testMediaArray() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $this->assertCount(3, $doc->getMediaArray(), "Media array must be of size 3");
        //print_r($doc->getMediaArray());
        foreach($doc->getMediaArray() as $media) {
            $this->assertInstanceOf('CorpusParole\Models\MediaResource', $media, "media msute of type MediaResource");
            // $this->assertArrayHasKey('format', $media, "media has 'format key'");
            // $this->assertArrayHasKey('url', $media, "media has url");
            // $this->assertCount(5, $media, "media is a 4 element array");
            // $this->assertArrayHasKey('extent', $media, "media has extent");
            // $this->assertArrayHasKey('extent_ms', $media, "media has extent_ms");
            // $this->assertArrayHasKey('master', $media, "media has master");

            //$this->assertEquals($media['url'], $url);

            $this->assertContains(
                $media->getUrl(),
                [ "http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav",
                  "http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3",
                  "http://cocoon.huma-num.fr/data/archi/masters/372593.wav"
                ]
            );
            if($media->getUrl() === "http://cocoon.huma-num.fr/data/archi/masters/372593.wav") {
                $this->assertEquals('audio/x-wav', $media->getFormat());
                $this->assertTrue($media->isMaster() === true, "should be master");
            }
            else {
                $this->assertTrue($media->isMaster() === false, "should not be master");
            }

            $this->assertEquals("PT48M26S", $media->getExtent(), "extent is PT48M26S");
            $this->assertGreaterThan(0, $media->getExtentMs(), "extent_ms must be > 0");
            $this->assertStringStartsWith('audio/', $media->getFormat());
        }
    }

    public function testGetTypes() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $this->assertCount(8, $doc->getTypes(), "types array must be of size 5");

        foreach($doc->getTypes() as $type) {
            $this->assertThat(
                $type,
                $this->logicalXor(
                    $this->isInstanceOf(EasyRdf\Literal::class),
                    $this->isInstanceOf(EasyRdf\Resource::class)
                )
            );
        }
    }

    public function testGetOtherTypes() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $this->assertCount(6, $doc->getOtherTypes(), "types array must be of size 5");

        foreach($doc->getTypes() as $type) {
            $this->assertThat(
                $type,
                $this->logicalXor(
                    $this->isInstanceOf(EasyRdf\Literal::class),
                    $this->isInstanceOf(EasyRdf\Resource::class)
                )
            );
        }
    }

    public function testGetDiscourseTypes() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $this->assertCount(2, $doc->getDiscourseTypes(), "types array must be of size 3");

        $this->assertContainsOnlyInstancesOf("EasyRdf\Resource", $doc->getDiscourseTypes(), "Result contains only Resources");
        $type = $doc->getDiscourseTypes()[0];
        $this->assertEquals("http://ark.bnf.fr/ark:/12148/cb11932135w", $type->getUri(), "discourse type is cb11932135w");
    }

    public function testCloneDocument() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $doc2 = clone $doc;

        $this->assertNotSame($doc, $doc2, "documents must not be the same");
        $this->assertNotSame($doc->getGraph(), $doc2->getGraph(), "documents must not be the same");

        $this->assertTrue(EasyRdf\Isomorphic::isomorphic($doc->getGraph(), $doc2->getGraph()),"graph must be isomorphic");
    }

    public function testIsIsomorphic() {
        $doc1 = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);
        $doc2 = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", new EasyRdf\Graph(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", sprintf(file_get_contents(DocumentTest::TEST_INPUT_DOCS['TEST']) , config('corpusparole.corpus_doc_id_base_uri'), config('corpusparole.corpus_id_scheme'))));

        $this->assertTrue($doc1->isIsomorphic($doc2),"document must be isomorphic");

        $doc2->addLiteral('dc11:type', new EasyRdf\Literal("oratory", null, Config::get('OLAC_DISCOURSE_TYPE')['uri']));

        $this->assertFalse($doc1->isIsomorphic($doc2),"document must not be isomorphic");
    }

    public function testUpdateDiscourseTypes() {

        $newDiscourseTypes = ["http://ark.bnf.fr/ark:/12148/cb11932135w" , "http://ark.bnf.fr/ark:/12148/cb13319048g", "http://ark.bnf.fr/ark:/12148/cb11949715t" , "http://ark.bnf.fr/ark:/12148/cb119783362"];

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);
        $this->assertCount(2, $doc->getDiscourseTypes(), "types array must be of size 2");

        $doc->updateDiscourseTypes($newDiscourseTypes);

        $this->assertCount(4, $doc->getDiscourseTypes(), "types array must be of size 4");

        $discourseTypes = $doc->getDiscourseTypes();
        foreach($newDiscourseTypes as $dt) {
            $this->assertContains($dt, $discourseTypes, "all discourse types must be in result list");
        }

    }

    public function testUpdateDiscourseTypesDelta() {

        $newDiscourseTypes = ["http://ark.bnf.fr/ark:/12148/cb11932135w" , "http://ark.bnf.fr/ark:/12148/cb13319048g", "http://ark.bnf.fr/ark:/12148/cb11949715t" , "http://ark.bnf.fr/ark:/12148/cb119783362"];

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $doc->updateDiscourseTypes($newDiscourseTypes);

        $this->assertTrue($doc->isDirty());

        $this->assertEquals(1, $doc->deltaCount(), "There is one delta");

        $delta = $doc->getDeltaList()[0];

        $this->assertEquals(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $delta->getDeletedGraph()->getUri(), "uri of deleted graph must be ok");
        $this->assertEquals(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $delta->getAddedGraph()->getUri(), "uri of added graph must be ok");

        $this->assertEquals(2, $delta->getDeletedGraph()->countTriples(), "deleted graph must have only 2 triples");
        $this->assertEquals(4, $delta->getAddedGraph()->countTriples(), "deleted graph must have only 4 triples");

        $resQueryDiscourseType = $delta->getAddedGraph()->allResources($doc->getProvidedCHO(), 'dc11:type');
        foreach($resQueryDiscourseType as $dt) {
            $this->assertInstanceOf(EasyRdf\Resource::class, $dt, "This must be a Resource");
            $this->assertRegexp(config('corpusparole.bnf_ark_url_regexp'), $dt->getUri(), "The resource uri must be bnf ark");
        }
        foreach($newDiscourseTypes as $dt) {
            $this->assertContains($dt, $resQueryDiscourseType, "all discourse types must be in result list");
        }

    }

    public function testUpdateTitle() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $oldTitle = $doc->getTitle();

        $doc->setTitle("new title", "en");

        $this->assertEquals("new title", $doc->getTitleValue());
        $this->assertEquals("new title", $doc->getTitle()->getValue());
        $this->assertEquals("en", $doc->getTitle()->getLang());

        $this->assertTrue($doc->isDirty());
        $this->assertEquals(1, $doc->deltaCount(), "There is one delta");

        $delta = $doc->getDeltaList()[0];

        $addedTitles = $delta->getAddedGraph()->allLiterals($doc->getProvidedCHO(), '<http://purl.org/dc/elements/1.1/title>');
        $this->assertCount(1, $addedTitles);

        $removedTitles = $delta->getDeletedGraph()->allLiterals($doc->getProvidedCHO(), '<http://purl.org/dc/elements/1.1/title>');
        $this->assertCount(1, $removedTitles);


    }

    public function testUpdateDiscourseTypesIsomorphic() {

        $newDiscourseTypes = ['oratory','dialogue','narrative'];

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);
        $doc->updateDiscourseTypes($newDiscourseTypes);

        $doc2 = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", new EasyRdf\Graph(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", sprintf(file_get_contents(DocumentTest::TEST_INPUT_DOCS['TEST']), config('corpusparole.corpus_doc_id_base_uri'), config('corpusparole.corpus_id_scheme'))));

        $this->assertFalse($doc->isIsomorphic($doc2),"document must not be isomorphic after adding discourse type");
    }

    public function testGetContributors() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $contributors = $doc->getContributors();

        $this->assertNotEmpty($contributors, "The contributors array should not be empty");
        $this->assertCount(8, $contributors, "The contributors array should have 8 elements");

        foreach ($contributors as $contribDef) {
            $this->assertArrayHasKey('name', $contribDef, "ContribDef must have name key");
            $this->assertArrayHasKey('url', $contribDef, "ContribDef must have url key");
            $this->assertArrayHasKey('role', $contribDef, "ContribDef must have role key");
            $this->assertContains($contribDef['role'], CocoonUtils::OLAC_ROLES, "Role should be in OLAC_ROLES");
        }
    }

    public function testSetContributors() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $contributors = $doc->getContributors();

        $contribList = [[
            "name"=> "Guylaine Brun-Trigaud",
            "url"=> "http://viaf.org/viaf/56666014",
            "role"=> "http://www.language-archives.org/OLAC/1.1/data_inputter"
        ], [
            "name"=> "LDOR",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/depositor"
        ], [
            "name"=> "Thésaurus Occitan",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/depositor"
        ], [
            "name"=> "Équipe de Recherche en Syntaxe et Sémantique",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/editor"
        ], [
            "name"=> "Bases, corpus, langage",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/editor"
        ], [
            "name"=> "Patrick Sauzet",
            "url"=> "http://viaf.org/viaf/51700729",
            "role"=> "http://www.language-archives.org/OLAC/1.1/researcher"
        ], [
            "name"=> "Alazet, Pierre",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/speaker"
        ], [
            "name"=> "Del Duca, Jeanne",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/transcriber"
        ], [
            "name"=> "Jane Austen, 1775-1817",
            "url"=> "http://viaf.org/viaf/102333412",
            "role"=> "http://www.language-archives.org/OLAC/1.1/compiler"
        ]];

        $doc->setContributors($contribList);

        $newContribs = $doc->getContributors();

        $this->assertCount(9, $newContribs);

        $this->assertTrue($doc->isDirty());
        $this->assertEquals(1, $doc->deltaCount(), "There is one delta");

        $delta = $doc->getDeltaList()[0];

        $addedGraph = $delta->getAddedGraph();
        $this->assertEquals(9, $addedGraph->countTriples());

        $removedGraph = $delta->getDeletedGraph();
        $this->assertEquals(count($contributors), $removedGraph->countTriples());

        $foundJaneAusten = false;
        foreach ($newContribs as $contribDef) {
            if(!is_null($contribDef['nameLiteral'])) {
                $lit = $contribDef['nameLiteral'];
                $this->assertNull($lit->getDatatype(), "Data type must be null $lit");
                $this->assertNotNull($lit->getLang(), "lang must not be null $lit");
            }
            if($contribDef['url'] == 'http://viaf.org/viaf/102333412') {
                $this->assertNull($contribDef['name'], 'Name must be null');
                $this->assertNull($contribDef['nameLiteral'], 'Name literal must be null');
                $foundJaneAusten = true;
            }
        }
        $this->assertTrue($foundJaneAusten, "Jane austenn not foud");

    }

    public function testGetSubjects() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $subjects = $doc->getSubjects();

        $this->assertCount(28, $subjects, "Must have 28 subjects");

        foreach ($doc->getSubjects() as $s) {
            $this->assertThat(
                $s,
                $this->logicalXor(
                    $this->isInstanceOf('EasyRdf\Literal'),
                    $this->isInstanceOf('EasyRdf\Resource')
                )
            );
        }
    }

    public function testSetSubjects() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $newSubjects = [
            "http://ark.bnf.fr/ark:/12148/cb13318415c",
            "http://ark.bnf.fr/ark:/12148/cb133188907",
            "http://ark.bnf.fr/ark:/12148/cb11932762f",
            "http://ark.bnf.fr/ark:/12148/cb133183660",
            "http://ark.bnf.fr/ark:/12148/cb122368540",
            "http://ark.bnf.fr/ark:/12148/cb119418302",
            "http://ark.bnf.fr/ark:/12148/cb135540729",
            "http://ark.bnf.fr/ark:/12148/cb133192210",
            "http://ark.bnf.fr/ark:/12148/cb119377452",
            "http://ark.bnf.fr/ark:/12148/cb13320451h",
            "http://ark.bnf.fr/ark:/12148/cb13318422n",
            "http://ark.bnf.fr/ark:/12148/cb11975823c"
        ];

        $doc->setSubjects($newSubjects);

        $this->assertTrue($doc->isDirty(), "The document must have changed");

        $subjects = $doc->getSubjects();

        $this->assertCount(12, $subjects, "Must have 12 subjects");

        foreach ($doc->getSubjects() as $s) {
            $this->assertInstanceOf('EasyRdf\Resource', $s, "Mustbe a Resource");
            $this->assertTrue(strrpos($s->getUri(), "http://ark.bnf.fr/ark:/12148/cb") === 0, "Must start with http://ark.bnf.fr/ark:/12148/cb");
            $this->assertContains($s->getUri(), $newSubjects, "$s must be in new subjects list");
            if(($key = array_search($s->getUri(), $newSubjects)) !== false) {
                unset($newSubjects[$key]);
            }
        }
        $this->assertEmpty($newSubjects, "all subjects must have been removed");
    }

    //TODO: test null transcript + null media array
    public function testJsonSerialize() {
        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $json = $doc->jsonSerialize();

        $this->assertTrue(is_array($json), 'Returned json must be an array');
        $this->assertEquals(
            ["id", "uri", "title", "languages", "modified", "issued", "created", "publishers", "contributors", "subjects", "transcript", "mediaArray", "geoInfo"],
            array_keys($json)
        );
        $this->assertEquals(sprintf('%1$s/crdo-CFPP2000_35_SOUNDid', config('corpusparole.handle_prefix')), $json['id']);
        $this->assertTrue(is_array($json['transcript']));
        $this->assertTrue(is_array($json['mediaArray']));

    }

    public function testAddGeoInfo() {

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST_NO_GEOINFO']);
        //$doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $res = $doc->addGeoInfo();
        $res->commit();

        $this->assertNotNull($res, "A new georesource object must have been created");

        $this->assertInstanceOf('CorpusParole\Models\GeoResource', $res, "the object created must be a geo resource");

        $allSpatials = $doc->getProvidedCHO()->allResources("<http://purl.org/dc/terms/spatial>");

        $this->assertCount(1, $allSpatials, "Must have only one spatial node");

        $spatial = $allSpatials[0];

        $this->assertTrue($spatial->isBNode(), "Spatial node must be a blank node");
        $this->assertEquals($spatial->typeAsResource(), "http://www.europeana.eu/schemas/edm/Place", "type must be a http://www.europeana.eu/schemas/edm/Place");

        $this->assertTrue($doc->isDirty(), "Document must be dirty");
        $this->assertEquals(1, $doc->deltaCount(), "Must have only one delta.");

        $this->assertNotNull($res->getCurrentDelta(), "CurrentDelta is not null");
        $this->assertTrue($res->getCurrentDelta()->getDeletedGraph()->isEmpty(), "deleted graph must be empty");
        $this->assertEmpty($res->getCurrentDelta()->getDeleteWhere(), "Delete where must be empty");
        $this->assertNotNull($res->getCurrentDelta()->getAddedGraph(), "Added graph is not null");
        $this->assertEquals(2, $res->getCurrentDelta()->getAddedGraph()->countTriples(), "Added graph must have 2 triples");

    }


    public function testAddGeoInfoNotEmpty() {

        $doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-CFPP2000_35_SOUND", $this->inputGraphes['TEST']);

        $res = $doc->addGeoInfo();
        $res->commit();

        $this->assertNotNull($res, "A new georesource object must have been created");

        $this->assertInstanceOf('CorpusParole\Models\GeoResource', $res, "the object created must be a geo resource");

        $allSpatials = $doc->getProvidedCHO()->allResources("<http://purl.org/dc/terms/spatial>");

        $this->assertCount(1, $allSpatials, "Must have only one spatial node");

        $spatial = $allSpatials[0];

        $this->assertTrue($spatial->isBNode(), "Spatial node must be a blank node");
        $this->assertEquals($spatial->typeAsResource(), "http://www.europeana.eu/schemas/edm/Place", "type must be a http://www.europeana.eu/schemas/edm/Place");

        $this->assertTrue($doc->isDirty(), "Document must not be dirty");
        $this->assertEquals(1, $doc->deltaCount(), "Must have only one delta.");

        $this->assertNotNull($res->getCurrentDelta(), "CurrentDelta is not null");
        $this->assertTrue($res->getCurrentDelta()->getDeletedGraph()->isEmpty(), "deleted graph must be empty");
        $this->assertCount(1, $res->getCurrentDelta()->getDeleteWhere(), "Delete where must have one element");
        $this->assertNotNull($res->getCurrentDelta()->getAddedGraph(), "Added graph is not null");
        $this->assertEquals(9, $res->getCurrentDelta()->getAddedGraph()->countTriples(), "Added graph must have 7 triples");
    }


}
