<?php

use CorpusParole\Libraries\Mappers\CocoonTextRdfMapper;
use CorpusParole\Libraries\CocoonUtils;

use EasyRdf\Graph;
use EasyRdf\Resource;
use EasyRdf\Literal;

use Illuminate\Foundation\Testing\WithoutMiddleware;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;

class CocoonTextRdfMapperTest extends TestCase
{

    const TEST_INPUT_DOCS = [
        'BASE' => <<<EOT
        @prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .
        @prefix owl: <http://www.w3.org/2002/07/owl#> .
        @prefix xsd: <http://www.w3.org/2001/XMLSchema#> .
        @prefix fn: <http://www.w3.org/2005/xpath-functions#> .
        @prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .
        @prefix sesame: <http://www.openrdf.org/schema/sesame#> .
        @prefix v: <http://rdf.data-vocabulary.org/#> .
        <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-UVE_MOCIKA> a <http://crdo.risc.cnrs.fr/schemas/Resource> ;
            <http://purl.org/dc/elements/1.1/description> "Voilà pourquoi le bernard-l'hermite, aujourd'hui, se cache dans les coquilles vides qu'il trouve, alors que le crabe de cocotier n'a pas honte de se promener tout nu."@fr ;
            <http://purl.org/dc/elements/1.1/format> "text/xml"^^<http://purl.org/dc/terms/IMT> ;
            <http://purl.org/dc/elements/1.1/type> "primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> , <http://purl.org/dc/dcmitype/Text> , "narrative"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> ;
            <http://purl.org/dc/elements/1.1/subject> <http://lexvo.org/id/iso639-3/uve> ;
            <http://purl.org/dc/elements/1.1/language> <http://lexvo.org/id/iso639-3/uve> ;
            <http://www.language-archives.org/OLAC/1.1/depositor> <http://viaf.org/viaf/56614135> ;
            <http://purl.org/dc/elements/1.1/publisher> <http://viaf.org/viaf/154919513> ;
            <http://purl.org/dc/elements/1.1/rights> "Copyright (c) Moyse-Faurie, Claire" ;
            <http://purl.org/dc/elements/1.1/title> "The two hermit crabs and the coconut crab"@en ;
            <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/exist/crdo/moyse-faurie/uve/crdo-UVE_MOCIKA.xml> , <http://cocoon.huma-num.fr/exist/crdo/moyse-faurie/uve/crdo-UVE_MOCIKA.xhtml> ;
            <http://purl.org/dc/terms/accessRights> "Freely available for non-commercial use" ;
            <http://purl.org/dc/terms/license> <http://creativecommons.org/licenses/by-nc-sa/2.5/> ;
            <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE> , <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LACITO> ;
            <http://purl.org/dc/terms/spatial> "NC"^^<http://purl.org/dc/terms/ISO3166> , "New Caledonia, Ohnyat (Ouvéa)" ;
            <http://purl.org/dc/terms/available> "2011-02-05"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/issued> "2011-02-05T23:22:23+01:00"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://www.language-archives.org/OLAC/1.1/speaker> "Idakote, Félicien" ;
            <http://purl.org/dc/terms/requires> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-UVE_MOCIKA_SOUND> ;
            <http://purl.org/dc/terms/alternative> "Les deux bernard-l'hermite et le crabe de cocotier"@fr ;
            <http://www.language-archives.org/OLAC/1.1/researcher> "Moyse-Faurie, Claire" ;
            <http://purl.org/dc/terms/modified> "2002-02-20"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/conformsTo> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-dtd_archive> .
EOT
    ];

    const TEST_INPUT_ID = "crdo-UVE_MOCIKA_SOUND";
    const TEST_GRAPH_URI = "http://corpusdelaparole.huma-num.fr/corpus/res/crdo-UVE_MOCIKA_SOUND";
    const TEST_CHO_URI = "http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-UVE_MOCIKA_SOUND";


    private $inputGraphes = [];
    private $resGraphes = [];
    private $mappers = [];

    function __construct(string $name = null) {
        parent::__construct($name);
    }

    public function setUp() {
        parent::setUp();

        foreach(CocoonTextRdfMapperTest::TEST_INPUT_DOCS as $key => $inputDoc) {
            $this->inputGraphes[$key] = new Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-UVE_MOCIKA", $inputDoc);
            $this->mappers[$key] = new CocoonTextRdfMapper($this->inputGraphes[$key]);
            $this->mappers[$key]->mapGraph();
            $this->resGraphes[$key] = $this->mappers[$key]->getOutputGraphes()["http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-UVE_MOCIKA_SOUND"];
        }
    }

    public function tearDown() {
        parent::tearDown();
    }

    /**
     * Just test that the construct and setup are ok
     *
     * @return void
     */
    public function testInit()
    {
        $this->assertTrue(true);
    }


    /**
     * Test that the returned graph has the same uri that the original.
     *
     * @return void
     */
    public function testUri() {

        //echo $this->resGraphes['BASE']->serialise('turtle');
        //echo var_export($this->resGraphes['BASE']->toRdfPhp());

        $this->assertNotNull($this->resGraphes['BASE']);
        $this->assertEquals(CocoonTextRdfMapperTest::TEST_GRAPH_URI, $this->resGraphes['BASE']->getUri(), "URIS must be translated");
    }

    /**
     * Test that the return graph has one ore:Aggregation resource
     *
     * @return void
     */
    public function testAggregationResource() {

        $resources = $this->resGraphes['BASE']->allOfType('ore:Aggregation');

        $this->assertCount(1, $resources, "Must found only one resources of type ore:Aggregation");
        $this->assertEquals(CocoonTextRdfMapperTest::TEST_GRAPH_URI,$resources[0]->getUri());
    }

    /**
     * Test providedCHO identifier.
     *
     * @return void
     */
    public function testProvidedChoId() {
        $resources = $this->resGraphes['BASE']->allOfType('edm:ProvidedCHO');

        $this->assertCount(1, $resources, "Must found only one resources of type edm:ProvidedCHO");
        $providedCHO = $resources[0];

        $identifier = $providedCHO->getLiteral('<http://purl.org/dc/elements/1.1/identifier>');

        $this->assertInstanceOf('EasyRdf\Literal', $identifier, "identifier value must be a literal");
        $this->assertEquals(config('corpusparole.corpus_id_scheme') . CocoonTextRdfMapperTest::TEST_INPUT_ID, $identifier->getValue(), "Value must be equals to ".config('corpusparole.corpus_id_scheme').CocoonTextRdfMapperTest::TEST_INPUT_ID);

    }

    public function testRightsWebResourceIdentity() {
        $properties = [
            "http://purl.org/dc/elements/1.1/rights",
            'http://purl.org/dc/terms/license',
        ];

        $webResourceList = $this->resGraphes['BASE']->all('edm:WebResource', '^rdf:type');
        $sourceNode = $this->inputGraphes['BASE']->get('http://crdo.risc.cnrs.fr/schemas/Resource', '^rdf:type');

        foreach ($webResourceList as $webResource) {
            foreach ($properties as $prop) {
                $outputValuesStr = [];
                foreach($webResource->all($this->resGraphes['BASE']->resource($prop)) as $outputValue) {
                    array_push($outputValuesStr, strval($outputValue));
                }
                $this->assertNotEmpty($outputValuesStr, "we must found some values to test $prop");
                foreach ($sourceNode->all($this->inputGraphes['BASE']->resource($prop)) as $value) {
                    $this->assertContains(strval($value), $outputValuesStr, "$prop not found in output graph");
                }
            }
        }
    }

    public function testRightsWebResourceTrim() {
        $properties = [
            'http://purl.org/dc/terms/accessRights',
        ];

        $webResourceList = $this->resGraphes['BASE']->all('edm:WebResource', '^rdf:type');
        $sourceNode = $this->inputGraphes['BASE']->get('http://crdo.risc.cnrs.fr/schemas/Resource', '^rdf:type');

        foreach ($webResourceList as $webResource) {
            foreach ($properties as $prop) {
                $outputValuesStr = [];
                foreach($webResource->all($this->resGraphes['BASE']->resource($prop)) as $outputValue) {
                    array_push($outputValuesStr, strval($outputValue));
                }
                $this->assertNotEmpty($outputValuesStr, "we must found some values to test $prop");
                foreach ($sourceNode->all($this->inputGraphes['BASE']->resource($prop)) as $value) {
                    $this->assertContains(trim(strval($value)), $outputValuesStr, "$prop not found in output graph");
                }
            }
        }
    }


    /**
     * Test providedCHO uri.
     *
     * @return void
     */
    public function testProvidedChoURI() {
        $resources = $this->resGraphes['BASE']->allOfType('edm:ProvidedCHO');

        $this->assertCount(1, $resources, "Must found only one resources of type edm:ProvidedCHO");
        $this->assertEquals(CocoonTextRdfMapperTest::TEST_CHO_URI,$resources[0]->getUri());
    }



    /**
     * Test that the returned graph does not have a http://purl.org/dc/dcmitype/Sound type
     *
     * @return void
     */
    public function testType() {
        //"primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> , <http://purl.org/dc/dcmitype/Text> , "narrative"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> ;
        $providedCHO = $this->resGraphes['BASE']->get('edm:ProvidedCHO', '^rdf:type');

        $this->assertNotNull($providedCHO);

        $this->assertcount(3, $providedCHO->all($this->resGraphes['BASE']->resource('http://purl.org/dc/elements/1.1/type')), 'but should find 2 literals');

        $dcTypeList = $providedCHO->all($this->resGraphes['BASE']->resource('http://purl.org/dc/elements/1.1/type'));
        foreach ($dcTypeList as $dcType) {
            $this->assertThat(
                $dcType,
                $this->logicalXor(
                    $this->isInstanceOf(EasyRdf\Literal::class),
                    $this->isInstanceOf(EasyRdf\Resource::class)
                )
            );
        }
    }

     /**
      * Test that the web resources
      *
      * @return void
      */
    public function testWebResources() {

        $resources = $this->resGraphes['BASE']->allOfType('edm:WebResource');

        $this->assertCount(2, $resources, "Must found three webresource");

        $aggregation = $this->resGraphes['BASE']->resource(CocoonTextRdfMapperTest::TEST_GRAPH_URI);

        foreach ($resources as $wres) {
            $mimetypes = $wres->all($this->resGraphes['BASE']->resource('http://purl.org/dc/elements/1.1/format'));
            $this->assertCount(1, $mimetypes, "Must find one mimetype.");
            $mimetype = $mimetypes[0];
            $this->assertInstanceOf("EasyRdf\Literal", $mimetype, "mimetype must be literal");
            $this->assertEquals("dc:IMT",$mimetype->getDatatype());
        }
    }

     /**
      * Test that the web resources
      *
      * @return void
      */
    public function testWebResourcesDate() {

        $resources = $this->resGraphes['BASE']->allOfType('edm:WebResource');

        foreach ($resources as $wres) {
            $this->assertFalse($wres->hasProperty("http://purl.org/dc/terms/available"),"web resource must not have http://purl.org/dc/terms/available");
            $this->assertFalse($wres->hasProperty("http://purl.org/dc/terms/modified"),"web resource must not have http://purl.org/dc/terms/modified");
            $this->assertTrue($wres->hasProperty("http://purl.org/dc/terms/issued"), "Must have http://purl.org/dc/terms/issued");

            $issued = $wres->getLiteral('dc:issued');
            //<http://purl.org/dc/terms/issued> "2014-12-05T15:00:19+01:00"^^<http://purl.org/dc/terms/W3CDTF> ;
            $this->assertInstanceOf('EasyRdf\Literal', $issued, "issued value must be a literal");
            $this->assertEquals('http://purl.org/dc/terms/W3CDTF', $issued->getDatatypeUri(), "issued datatype uri must be a http://purl.org/dc/terms/W3CDTF");
            $this->assertEquals('2011-02-05T23:22:23+01:00', $issued->getValue(), "Value must be 2014-12-05T15:00:19+01:00");

        }

    }


}
