<?php
namespace CorpusParole\Libraries\Sparql;

use EasyRdf\Graph;
use CorpusParole\Libraries\CorpusParoleException;
use Config;
use Log;

/**
 *
 */
class SparqlClient {

    private $currentTransactionUrl = null;

    function __construct($httpClient, $sparqlClient) {
        $this->sparqlClient = $sparqlClient;
        $this->httpClient = $httpClient;
    }

    private function getHttpClient() {
        return $this->httpClient;
    }

    public function getCurrentTransactionUrl() {
        return $this->currentTransactionUrl;
    }

    /**
     * Start transaction.
     * @return boolean true if a transaction was started, false otherwise
     * @throws CorpusParoleException if the  transaction could not be stared
     */
    public function startTransaction() {
        if(!is_null($this->currentTransactionUrl)) {
            // We just continue the current transaction
            return false;
        }
        //Log::debug('http_client base uri: ' . $this->getHttpClient()->getConfig('base_uri'));
        $sesameRepository = config('corpusparole.sesame_repository');
        $resp = $this->getHttpClient()->post("$sesameRepository/transactions");
        //$resp = $this->getHttpClient()->post('transactions');
        //TODO check errors
        if($resp->getStatusCode() != 201) {
            throw new CorpusParoleException("Error when starting transaction : "
                . $resp->getStatusCode() . " - "
                . $resp->getReasonPhrase() . " : " . $resp->getBody(), 1);
        }

        $this->currentTransactionUrl = $resp->getHeader('Location')[0];

        return true;
    }

    /**
     * Rollback transaction.
     * @return boolean true if a transaction was started, false otherwise
     * @throws CorpusParoleException if the  transaction could not be rollbacked
     */
    public function rollback() {
        if(is_null($this->currentTransactionUrl)) {
            // We just continue the current transaction
            return false;
        }
        $resp = $this->getHttpClient()->delete($this->currentTransactionUrl);

        if($resp->getStatusCode() != 204) {
            throw new CorpusParoleException("Could not cancel transaction : "
                . $resp->getStatusCode() . " - "
                . $resp->getReasonPhrase() . " : " . $resp->getBody(), 1);
        }
        $this->currentTransactionUrl = null;

        return true;
    }

    /**
     * Commit transaction.
     * @return boolean true if a transaction was started, false otherwise
     * @throws CorpusParoleException if the  transaction could not be commited
     */
    public function commit() {
        if(is_null($this->currentTransactionUrl)) {
            // We just continue the current transaction
            return false;
        }
        $resp = $this->getHttpClient()->put($this->currentTransactionUrl, ['query' => ['action' => 'COMMIT']]);
        if($resp->getStatusCode() != 200) {
            throw new CorpusParoleException("Could not commit transaction : "
                . $resp->getStatusCode() . " - "
                . $resp->getReasonPhrase() . " : " . $resp->getBody(), 1);
        }
        $this->currentTransactionUrl = null;
        return true;
    }

    protected function updateData($operation, Graph $graph)
    {
        $graphUri = $graph->getUri();
        $query = "$operation DATA {";
        if ($graphUri) {
            $query .= "GRAPH <$graphUri> {";
        }
        $query .= $graph->serialise('ntriples');
        if ($graphUri) {
            $query .= "}";
        }
        $query .= '}';

        $resp = $this->getHttpClient()->put(
            $this->currentTransactionUrl, [
                'query' => ['action' => 'UPDATE'],
                'form_params'=> ['update' => $query],
            ]
        );
        if($resp->getStatusCode() != 204) {
            throw new CorpusParoleException("Could not update in transaction with operation $operation: "
                . $resp->getStatusCode() . " - "
                . $resp->getReasonPhrase() . " : " . $resp->getBody()
                . " : $query" , 1);
        }
    }

    public function add(Graph $graph) {
        $this->updateData('INSERT', $graph);
    }

    public function delete(Graph $graph) {
        $this->updateData('DELETE', $graph);
    }

    /** Make a query to the SPARQL endpoint
     *
     * Just call and return EasyRdf\Sparql\Client::query
     *
     * @param string $query The query string to be executed
     * @return object EasyRdf\Sparql\Result|EasyRdf\Graph Result of the query.
     */
    public function query($query) {
        return $this->sparqlClient->query($query);
    }

}
