<?php

use Illuminate\Support\Facades\Facade;

use CorpusParole\Repositories\RdfDocumentRepository;
use CorpusParole\Libraries\Sparql\SparqlClient;
use CorpusParole\Models\Document;

use Mockery as m;

class DocumentRepositoryTest extends TestCase {

    const TEST_DOC = <<<EOT
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://www.w3.org/2000/01/rdf-schema#Resource> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Document> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/description> "Extrait des enqu\u00EAtes dialectologiques en vue de constituer un atlas linguistique de l'Alsace."@fr .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/format> "audio/x-wav"^^<http://purl.org/dc/terms/IMT> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> <http://purl.org/dc/dcmitype/Sound> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> <http://lexvo.org/id/iso639-3/gsw> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "lexicography"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "phonetics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "anthropological_linguistics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/language> <http://lexvo.org/id/iso639-3/gsw> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> <http://viaf.org/viaf/9122216> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> _:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> _:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/publisher> "Atlas linguistiques, cultures et parlers r\u00E9gionaux de France" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/rights> "Copyright (c) D\u00E9partement de dialectologie alsacienne et mosellane de l'Universit\u00E9 de Strasbourg" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/title> "Atlas Linguistique et ethnographique de l'Alsace - Enquetes sur la conscience linguistique - ALA_608"@fr .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/accessRights> "Freely available for non-commercial use" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/license> <http://creativecommons.org/licenses/by-nc-nd/2.5/> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ALA_CL> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/extent> "PT02H04M16S" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/created> "start=1980; end=1989; name=ann\u00E9es 80"^^<http://purl.org/dc/terms/Period> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/spatial> "France, Alsace"@fr .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/spatial> "FR"^^<http://purl.org/dc/terms/ISO3166> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/ala/ALA_608.mp3> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/ala/masters/ALA_608.wav> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/available> "2004-07-03"^^<http://purl.org/dc/terms/W3CDTF> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/issued> "2014-11-04T16:25:54+01:00"^^<http://purl.org/dc/terms/W3CDTF> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://www.openrdf.org/schema/sesame#directType> <http://xmlns.com/foaf/0.1/Document> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://www.w3.org/2000/01/rdf-schema#Resource> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Agent> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://xmlns.com/foaf/0.1/name> "Bothorel-Witz, Arlette" .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://www.openrdf.org/schema/sesame#directType> <http://xmlns.com/foaf/0.1/Agent> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://www.w3.org/2000/01/rdf-schema#Resource> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Agent> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://xmlns.com/foaf/0.1/name> "Huck, Dominique" .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://www.openrdf.org/schema/sesame#directType> <http://xmlns.com/foaf/0.1/Agent> .
<http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608> <http://xmlns.com/foaf/0.1/primaryTopic> <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> .
EOT;

    const RES_DOC = <<<EOT
    @prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .
    @prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .
    @prefix sesame: <http://www.openrdf.org/schema/sesame#> .
    @prefix owl: <http://www.w3.org/2002/07/owl#> .
    @prefix xsd: <http://www.w3.org/2001/XMLSchema#> .
    @prefix fn: <http://www.w3.org/2005/xpath-functions#> .

    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> a <http://xmlns.com/foaf/0.1/Document> ;
    	<http://purl.org/dc/elements/1.1/description> "Extrait des enquêtes dialectologiques en vue de constituer un atlas linguistique de l'Alsace."@fr ;
    	<http://purl.org/dc/elements/1.1/format> "audio/x-wav"^^<http://purl.org/dc/terms/IMT> ;
    	<http://purl.org/dc/elements/1.1/type> <http://purl.org/dc/dcmitype/Sound> , "primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> , "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> ;
    	<http://purl.org/dc/elements/1.1/subject> <http://lexvo.org/id/iso639-3/gsw> , "lexicography"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> , "phonetics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> , "anthropological_linguistics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> ;
    	<http://purl.org/dc/elements/1.1/language> <http://lexvo.org/id/iso639-3/gsw> ;
    	<http://purl.org/dc/elements/1.1/contributor> <http://viaf.org/viaf/9122216> , _:genid-39d722daa01242edb2da8d2d1e71eae7-ED8A67E3CE4C51EA8EDC4ED1190E161A , _:genid-39d722daa01242edb2da8d2d1e71eae7-FD40C63A1295F7BE6765BB87A70CE839 ;
    	<http://purl.org/dc/elements/1.1/publisher> "Atlas linguistiques, cultures et parlers régionaux de France" ;
    	<http://purl.org/dc/elements/1.1/rights> "Copyright (c) Département de dialectologie alsacienne et mosellane de l'Université de Strasbourg" ;
    	<http://purl.org/dc/elements/1.1/title> "Atlas Linguistique et ethnographique de l'Alsace - Enquetes sur la conscience linguistique - ALA_608"@fr ;
    	<http://purl.org/dc/terms/accessRights> "Freely available for non-commercial use" ;
    	<http://purl.org/dc/terms/license> <http://creativecommons.org/licenses/by-nc-nd/2.5/> ;
    	<http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ALA_CL> , <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE> ;
    	<http://purl.org/dc/terms/extent> "PT02H04M16S" ;
    	<http://purl.org/dc/terms/created> "start=1980; end=1989; name=années 80"^^<http://purl.org/dc/terms/Period> ;
    	<http://purl.org/dc/terms/spatial> "France, Alsace"@fr , "FR"^^<http://purl.org/dc/terms/ISO3166> ;
    	<http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/ala/ALA_608.mp3> , <http://cocoon.huma-num.fr/data/ala/masters/ALA_608.wav> ;
    	<http://purl.org/dc/terms/available> "2004-07-03"^^<http://purl.org/dc/terms/W3CDTF> ;
    	<http://purl.org/dc/terms/issued> "2014-11-04T16:25:54+01:00"^^<http://purl.org/dc/terms/W3CDTF> .

    _:genid-39d722daa01242edb2da8d2d1e71eae7-ED8A67E3CE4C51EA8EDC4ED1190E161A a <http://xmlns.com/foaf/0.1/Agent> ;
    	<http://xmlns.com/foaf/0.1/name> "Bothorel-Witz, Arlette" ;
    	<http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .

    _:genid-39d722daa01242edb2da8d2d1e71eae7-FD40C63A1295F7BE6765BB87A70CE839 a <http://xmlns.com/foaf/0.1/Agent> ;
    	<http://xmlns.com/foaf/0.1/name> "Huck, Dominique" ;
    	<http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .

    <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608> <http://xmlns.com/foaf/0.1/primaryTopic> <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> .
EOT;


    function __construct(string $name = null) {
        parent::__construct($name);
    }

    public function setUp() {
        parent::setUp();
        $this->graph = new EasyRdf\Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", DocumentRepositoryTest::TEST_DOC);
        $this->cocoonDocIdBaseUri = config('corpusparole.cocoon_doc_id_base_uri');
    }

    public function tearDown() {
        parent::tearDown();
    }


    public function testAll() {
        $expectedId = $this->cocoonDocIdBaseUri.'crdo-ALA_608';
        $returnedArray = [
            m::mock('resObj', function($mock) use (&$expectedId) {
                $mock->uri = new EasyRdf\Resource($expectedId);
            }),
        ];

        $sparqlClientMock = m::mock('CorpusParole\Libraries\Sparql\SparqlClient');
        $sparqlClientMock->shouldReceive('query')->andReturn($returnedArray);

        $documentRepository = $this->app->make('CorpusParole\Repositories\RdfDocumentRepository', [$sparqlClientMock]);

        $res = $documentRepository->all();

        $this->assertCount(1, $res, "Should have 1 doc in res");
        $resDoc = $res[0];

        $this->assertInstanceOf(Document::class, $resDoc, "Res doc must be a Document");
        $this->assertEquals('crdo-ALA_608', $resDoc->getId(), "id must be crdo...");
        $this->assertEquals($expectedId, $resDoc->getUri(), 'url must be ...');
        $this->assertNull($resDoc->getGraph(), 'Graph must be null');
    }

    public function testGet() {

        $expectedId = $this->cocoonDocIdBaseUri.'crdo-ALA_608';
        $returnedGraph = new EasyRdf\Graph($expectedId, DocumentRepositoryTest::RES_DOC);

        $sparqlClientMock = m::mock('CorpusParole\Libraries\Sparql\SparqlClient');
        $sparqlClientMock->shouldReceive('query')->with(containsString($expectedId))
                ->andReturn($returnedGraph);
        $documentRepository = $this->app->make('CorpusParole\Repositories\RdfDocumentRepository', [$sparqlClientMock]);

        $res = $documentRepository->get('crdo-ALA_608');

        $this->assertInstanceOf(Document::class, $res, "Result must be of type Document");
        $this->assertEquals('crdo-ALA_608', $res->getId(), 'id should be crdo-ALA_608' );
        $this->assertNotNull($res->getGraph(), "Graph shoul not be null");
        $this->assertTrue(EasyRdf\Isomorphic::isomorphic($res->getGraph(), new EasyRdf\Graph($expectedId, DocumentRepositoryTest::RES_DOC)));
    }

    public function testSave() {

        $sparqlClientMock = m::mock('CorpusParole\Libraries\Sparql\SparqlClient', function($mock) {
            $mock->shouldReceive('startTransaction')->andReturn(true)
                ->shouldReceive('delete')->shouldReceive('add')
                ->shouldReceive('commit')->andReturn(true);
        });
        $documentRepository = $this->app->make('CorpusParole\Repositories\RdfDocumentRepository', [$sparqlClientMock]);


        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);
        $doc->updateDiscourseTypes(['oratory','drama','narrative']);

        $res = $documentRepository->save($doc);

        $this->assertTrue($res, 'Has started a transaction');

    }

    public function testCount() {

        $returnedCount = [
            m::mock('resObj', function($mock) {
                $mock->count = new EasyRdf\Literal(1);
            }),
        ];

        $sparqlClientMock = m::mock('CorpusParole\Libraries\Sparql\SparqlClient');
        $sparqlClientMock->shouldReceive('query')->with(containsString("COUNT"))
                ->andReturn($returnedCount);
        $documentRepository = $this->app->make('CorpusParole\Repositories\RdfDocumentRepository', [$sparqlClientMock]);



    }

}
