<?php

use Mockery as m;

use CorpusParole\Models\Document;


/**
 *
 */
class TranscriptConverterBaseTest extends TestCase {
    const TEST_DOC_BASE = "crdo-UVE_MOCIKA";

    public function setUp() {
        parent::setup();
        $graphContent = sprintf(file_get_contents(__DIR__ . DIRECTORY_SEPARATOR . self::TEST_DOC_BASE.".ttl"), config('corpusparole.corpus_doc_id_base_uri'), config('corpusparole.corpus_id_scheme'));
        $this->graph = new EasyRdf\Graph(config('corpusparole.corpus_doc_id_base_uri')."crdo-UVE_MOCIKA_SOUND", $graphContent);
        $this->doc = new Document(config('corpusparole.corpus_doc_id_base_uri')."crdo-UVE_MOCIKA_SOUND", $this->graph);

        $this->transcriptSource = simplexml_load_file(__DIR__ . DIRECTORY_SEPARATOR . self::TEST_DOC_BASE.".xml");
    }

    public function getMockConverter(...$contructorArgs) {
        return  m::mock("CorpusParole\Libraries\Transcript\TranscriptConverterBase", $contructorArgs)
            ->shouldReceive('parseSource')
            ->andReturn(null)
            ->shouldReceive('buildAnnotations')
            ->andReturn([]);
    }


    public function testConstructor() {
        $converter = $this->getMockConverter($this->doc, $this->transcriptSource)->getMock()->makePartial();
        $json = $converter->convertToJson();
        $this->assertNotnull($json);
    }

    public function testHeaderMeta() {
        $datetime =  (new DateTime())->format(DateTime::ATOM);
        $converter =  $this->getMockConverter($this->doc, $this->transcriptSource, $datetime)->getMock()->makePartial();
        $resJson = $converter->convertTojson();
        $this->assertArrayHasKey('format', $resJson, 'Must have format key');
        $this->assertEquals('http://advene.org/ns/cinelab/', $resJson['format'], 'Format key must be http://advene.org/ns/cinelab/');
        $this->assertArrayHasKey('@context', $resJson, 'Must have @context key');
        $this->assertEquals(
            [
                "dc" =>  "http://purl.org/dc/elements/1.1/",
                "corpus" => "http://corpusdelaparole.huma-num.fr/ns/corpus#"
            ],
            $resJson['@context'],
            "array must contains dc and corpus declaration"
        );
        $this->assertArrayHasKey('meta', $resJson, 'Must have meta declaration');
        $meta = $resJson['meta'];
        $this->assertTrue(is_array($meta), 'meta is an array');

        $this->assertArrayHasKey('dc:creator', $meta, "meta must have dc:creator key");
        $this->assertEquals(config('corpusparole.transcript_default_creator'), $meta['dc:creator'], "dc:creator must be ".config('corpusparole.transcript_default_creator'));
        $this->assertArrayHasKey('dc:contributor', $meta, "meta must have dc:contributor key");
        $this->assertEquals(config('corpusparole.transcript_default_creator'), $meta['dc:contributor'], "dc:contributor must be ".config('corpusparole.transcript_default_creator'));
        $this->assertArrayHasKey('dc:created', $meta, "meta must have dc:created key");
        $this->assertEquals($datetime, $meta["dc:created"], "meta created time mus be $datetime");
        $this->assertArrayHasKey('dc:modified', $meta, "meta must have dc:modified key");
        $this->assertEquals($datetime, $meta["dc:modified"], "meta modified time mus be $datetime");

        $this->assertArrayHasKey('dc:title', $meta, "meta must have dc:title key");
        $this->assertTrue(is_array($meta['dc:title']), 'meta title is an array');
        $this->assertCount(2, $meta['dc:title'], 'meta title array has 2 elements');
        $this->assertArrayHasKey('@language', $meta['dc:title'], "meta title must have @language key");
        $this->assertArrayHasKey('@value', $meta['dc:title'], "meta title must have @value key");
        $this->assertEquals('en',  $meta['dc:title']['@language'], "meta title language must be en");
        $this->assertEquals('The two hermit crabs and the coconut crab',  $meta['dc:title']['@value'], "meta title value must be the Two hermit crabs and the coconut crab");
    }

    public function testHeaderMetaTitleOverrideString() {
        $converter = $this->getMockConverter($this->doc, $this->transcriptSource)->shouldReceive("getSourceTitle")->andReturn("The two hermit crabs and the coconut crab")->getMock()->makePartial();
        $resJson = $converter->convertTojson();

        $meta = $resJson['meta'];
        $this->assertArrayHasKey('dc:title', $meta, "meta must have dc:title key");
        $this->assertTrue(is_string($meta['dc:title']), 'meta title is a string');
        $this->assertEquals('The two hermit crabs and the coconut crab',  $meta['dc:title'], "meta title value must be the Two hermit crabs and the coconut crab");

    }

    public function testHeaderMetaTitleOverrideLanguageMap() {
        $converter = $this->getMockConverter($this->doc, $this->transcriptSource)->shouldReceive("getSourceTitle")->andReturn(['en' => "The two hermit crabs and the coconut crab", "fr" => 'Les deux bernard-l\'hermite et le crabe de cocotier'])->getMock()->makePartial();
        $resJson = $converter->convertTojson();

        $meta = $resJson['meta'];
        $this->assertArrayHasKey('dc:title', $meta, "meta must have dc:title key");
        $this->assertTrue(is_array($meta['dc:title']), 'meta title is an array');
        $this->assertCount(2, $meta['dc:title'], "title must be an array of size 2");
        $this->assertEquals(["@language" => 'en', '@value' => 'The two hermit crabs and the coconut crab'],  $meta['dc:title'][0], "meta title value must be the Two hermit crabs and the coconut crab");
        $this->assertEquals(["@language" => 'fr', '@value' => 'Les deux bernard-l\'hermite et le crabe de cocotier'],  $meta['dc:title'][1], "meta title value must be the Les deux bernard-l\'hermite et le crabe de cocotier");
    }

    public function testMedias() {
        $converter =  $this->getMockConverter($this->doc, $this->transcriptSource)->getMock()->makePartial();
        $resJson = $converter->convertTojson();

        $this->assertArrayHasKey('medias', $resJson, "Must have a medias field");
        $this->assertTrue(is_array($resJson['medias']), 'media is an array');
    }

    public function testMediasContent() {
        $converter =  $this->getMockConverter($this->doc, $this->transcriptSource)->getMock()->makePartial();
        $resJson = $converter->convertTojson();

        $medias = $resJson['medias'];

        $this->assertCount(3, $medias, "Should have 3 media");
        $this->assertCount(3, array_filter($medias, function($m) {
            return array_key_exists('meta', $m) &&
                array_key_exists('url', $m) &&
                array_key_exists('origin', $m) &&
                array_key_exists('unit', $m) &&
                array_key_exists('id', $m) ; }), "all media must have url, id, origin, unit, meta key");

        $this->assertCount(1, array_filter($medias, function($m) { return array_key_exists('corpus:master', $m['meta'])?$m['meta']['corpus:master']:false; }), "should have at least 1 master");

    }

    public function testMediasContentDetail() {
        $converter =  $this->getMockConverter($this->doc, $this->transcriptSource)->getMock()->makePartial();
        $resJson = $converter->convertTojson();

        $medias = $resJson['medias'];
        foreach($medias as $m) {
            $this->assertEquals(0, $m['origin'], "Origin must be 0");
            $this->assertEquals('ms', $m['unit'], "unit must be ms");
            $this->assertStringStartsWith('http', $m['url'], "urls must be string and start with http");
            $this->assertRegExp('/^11280\\.100\\/crdo-UVE_MOCIKA_SOUND_m\d+$/', $m['id'], "id must start with 11280.100\/crdo-UVE_MOCIKA_SOUND_m");
            $this->assertArrayHasKey('meta', $m, "Media def must have a meta");
            $mediaMeta = $m['meta'];
            $this->assertArrayHasKey('dc:duration', $mediaMeta, "media meta must have duration");
            $this->assertEquals(155000, $mediaMeta['dc:duration'], "media meta duration is 1555000 ms");
            $this->assertArrayHasKey('dc:title', $mediaMeta, "Media meta mus have title");
            $this->assertEquals(["@language" => 'en', "@value" => "The two hermit crabs and the coconut crab"], $mediaMeta['dc:title'], "media meta title is ...");
            $this->assertArrayHasKey('dc:format', $mediaMeta, 'media meta has dc:format');
            $this->assertStringStartsWith('audio/', $mediaMeta['dc:format'], "media meta dc:format starts with audio");
        }
    }

    public function testOtherNodes() {
        $converter =  $this->getMockConverter($this->doc, $this->transcriptSource)->getMock()->makePartial();
        $resJson = $converter->convertTojson();

        $this->assertArrayHasKey('resources', $resJson, 'res must have resources');
        $this->assertEquals([], $resJson['resources'], 'Resources must be empty array');
        $this->assertArrayHasKey('lists', $resJson, 'res must have lists');
        $this->assertEquals([], $resJson['lists'], 'Lists must be empty array');
        $this->assertArrayHasKey('annotation-types', $resJson, 'res must have annotation-types');
        $this->assertEquals([], $resJson['annotation-types'], 'Annotation types must be empty array');
        $this->assertArrayHasKey('annotations', $resJson, 'res must have annotations');
        $this->assertEquals([], $resJson['annotations'], 'Annotations must be empty array');
    }

    public function tearDown() {
        m::close();
    }

}
