<?php

use CorpusParole\Libraries\Mappers\CocoonSoundRdfMapper;
use CorpusParole\Libraries\CocoonUtils;

use EasyRdf\Graph;
use EasyRdf\Resource;
use EasyRdf\Literal;

use Illuminate\Foundation\Testing\WithoutMiddleware;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;

class CocoonSoundRdfMapperTest extends TestCase
{

    const TEST_INPUT_DOCS = [
        'BASE' => <<<EOT
        @prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .
        @prefix owl: <http://www.w3.org/2002/07/owl#> .
        @prefix xsd: <http://www.w3.org/2001/XMLSchema#> .
        @prefix fn: <http://www.w3.org/2005/xpath-functions#> .
        @prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .
        @prefix sesame: <http://www.openrdf.org/schema/sesame#> .
        @prefix v: <http://rdf.data-vocabulary.org/#> .
            <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019> a <http://crdo.risc.cnrs.fr/schemas/Resource> ;
            <http://purl.org/dc/elements/1.1/description> "Extrait du corpus d'Orléans, réalisé dans le cadre de l'Enquête SocioLinguistique à Orléans à la fin des années 1960."@fr ;
            <http://purl.org/dc/elements/1.1/type> <http://purl.org/dc/dcmitype/Sound> , "primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> , "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> ;
            <http://purl.org/dc/elements/1.1/subject> <http://lexvo.org/id/iso639-3/fra> , "text_and_corpus_linguistics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> , <http://ark.bnf.fr/ark:/12148/cb135052099> , <http://ark.bnf.fr/ark:/12148/cb11943508j> , <http://ark.bnf.fr/ark:/12148/cb11934124q> , <http://ark.bnf.fr/ark:/12148/cb11931498c> , <http://ark.bnf.fr/ark:/12148/cb11935508t> , <http://ark.bnf.fr/ark:/12148/cb11934132b> , <http://ark.bnf.fr/ark:/12148/cb11939893p> , <http://ark.bnf.fr/ark:/12148/cb11933091b> ;
            <http://purl.org/dc/elements/1.1/language> <http://lexvo.org/id/iso639-3/fra> ;
            <http://www.language-archives.org/OLAC/1.1/depositor> <http://viaf.org/viaf/181558906> ;
            <http://purl.org/dc/elements/1.1/publisher> <http://viaf.org/viaf/181558906> ;
            <http://purl.org/dc/elements/1.1/rights> "Copyright (c) 2012 Université d'Orléans/LLL" ;
            <http://purl.org/dc/elements/1.1/title> "ESLO1: entretien 019"@fr ;
            <http://purl.org/dc/terms/alternative> "ESLO1: entretien 019 alt"@fr ;
            <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/eslo/ESLO1_ENT_019_22km.wav> , <http://cocoon.huma-num.fr/data/eslo/ESLO1_ENT_019.mp3> ;
            <http://purl.org/dc/terms/accessRights> " Freely available for non-commercial use" ;
            <http://purl.org/dc/terms/license> <http://creativecommons.org/licenses/by-nc-sa/3.0/> ;
            <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE> , <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ESLO1>, <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_010> ;
            <http://purl.org/dc/terms/extent> "PT0H47M16S" ;
            <http://purl.org/dc/terms/created> "1969-04-11"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/spatial> "FR"^^<http://purl.org/dc/terms/ISO3166> , <http://vocab.getty.edu/tgn/7008337> , "France, Centre, Loiret, Orléans"@fr ;
            <http://crdo.risc.cnrs.fr/schemas/master> <http://cocoon.huma-num.fr/data/eslo/masters/ESLO1_ENT_019.wav> ;
            <http://purl.org/dc/terms/available> "2014-11-06"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/issued> "2014-12-05T15:00:19+01:00"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/modified> "2014-12-05T15:00:19+01:00"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://www.language-archives.org/OLAC/1.1/speaker> "JK" , "UG393" ;
            <http://purl.org/dc/elements/1.1/source> <http://archivesetmanuscrits.bnf.fr/ead.html?id=FRBNFEAD000095934&c=FRBNFEAD000095934_a19844296> ;
            <http://www.w3.org/2003/01/geo/wgs84_pos#long> "1.90"^^xsd:float ;
            <http://www.w3.org/2003/01/geo/wgs84_pos#lat> "47.90"^^xsd:float ;
            <http://purl.org/dc/terms/tableOfContents> "01 née; famille a toujours habité Orléans03 oui, on y a des amis, relations04 oui, certainementT1 centralisait comptabilité dans une industrieT2 RR passer des écritures, caisse; horaires avant et après loi de 1936T3 aimait travail, rien ne lui déplaisaitT4 n'a jamais pensé à çaT5 actuellement on est forcée quelquefois - assurance sociale - retraite appréciable; autre chose avec enfantL1 concerts, théâtre, réunions, etc...:préfère s'occuperL2 messe, puis a passé la journée avec une amieL3 (quand elle travaillait) au bord de la mer ou voyages organisésE1 n'en sait rienE2 lui semble que ça aide le françaisE3 peut-être chimie, physiqueE4 intelligence, travail, familleE5 dépend de leurs capacitésE6 non : dépend de la vie de l'enfant plus tardP1 oui, à part loyers chersP4 ne comprend pas motifs des étudiants; très calme à OrléansP9 moins qu'avant 1940P10 RR bourgeois = gros rentiers, femmes d'officierss, gros industriels; chaque groupe se recevait mutuellement, formait un clanQ. Orléans une ville renfermée? oui : orléanais assez distant, peut-être moins maintenantP6 il faut avoir la moitié des votesP7 des candidats se retirentP8 droite gaulliste, réservée; gauche avancéeP5 pas grand-chose - attachés à leurs partisP11 oui, maintenant, par l'éducation; famille moins importanteP12 à diminuer : les gens travaillent beaucoup plus et l'argent a changé de mainsOmelette (je casse mes oeufs...)Questionnaire sociolinguistique"@fr ;
            <http://purl.org/dc/terms/isRequiredBy> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019_C> , <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019_A> ;
            <http://www.language-archives.org/OLAC/1.1/editor> <http://viaf.org/viaf/39685504> , <http://viaf.org/viaf/200058210> ;
            <http://purl.org/dc/terms/abstract> " Identifiant du témoin : UG 393 Femme, 72 ans (né en 1897, à Orléans ) ; comptable en retraite Enregistré par Jack Kay, le 11 avril 1969, au domicile du témoin. Questionnaires enregistrés : questionnaire sociolinguistique; questionnaire ouvert(questions 01 ; 03-04 ; T1-T5 ; L1-L3 ; E1-E6 ; P1 ; P4 ; P9-P10 ; P6-P8 ; P5 ; P11-P12 ; omelette). Classe politique du témoin : Centre Remarques sur le témoignage : réponses concises. Intérêt pédagogique : conditions de travail avant la guerre de 1940 ; classes sociales. Acoustique : bonne qualité d'enregistrement."@fr ;
            <http://www.language-archives.org/OLAC/1.1/researcher> "Kay, Jack" ;
            <http://purl.org/dc/elements/1.1/relation> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENTCONT_233> ;
            <http://purl.org/dc/terms/medium> "1 bande magnétique audio"@fr ;
            <http://purl.org/dc/terms/bibliographicCitation> "A.-M. VURPAS (1988), Autrefois dans les monts du Beaujolais, Action graphique.";
            <http://purl.org/dc/elements/1.1/identifier> "Cote producteur: 001_3"@fr ;
            <http://purl.org/dc/elements/1.1/reference> <http://panini.u-paris10.fr> ;
            <http://purl.org/dc/elements/1.1/coverage> "Quartier concerné : 3e"@fr ;
            <http://purl.org/dc/elements/1.1/creator> "Blanc, Michel" , "Biggs, Patricia" .
EOT
        ,
        'NO-FORMAT' => <<<EOT
        @prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .
        @prefix owl: <http://www.w3.org/2002/07/owl#> .
        @prefix xsd: <http://www.w3.org/2001/XMLSchema#> .
        @prefix fn: <http://www.w3.org/2005/xpath-functions#> .
        @prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .
        @prefix sesame: <http://www.openrdf.org/schema/sesame#> .
        @prefix v: <http://rdf.data-vocabulary.org/#> .
            <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019> a <http://crdo.risc.cnrs.fr/schemas/Resource> ;
            <http://purl.org/dc/elements/1.1/description> "Extrait du corpus d'Orléans, réalisé dans le cadre de l'Enquête SocioLinguistique à Orléans à la fin des années 1960."@fr ;
            <http://purl.org/dc/elements/1.1/type> <http://purl.org/dc/dcmitype/Sound> , "primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> , "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> ;
            <http://purl.org/dc/elements/1.1/subject> <http://lexvo.org/id/iso639-3/fra> , "text_and_corpus_linguistics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> , <http://ark.bnf.fr/ark:/12148/cb135052099> , <http://ark.bnf.fr/ark:/12148/cb11943508j> , <http://ark.bnf.fr/ark:/12148/cb11934124q> , <http://ark.bnf.fr/ark:/12148/cb11931498c> , <http://ark.bnf.fr/ark:/12148/cb11935508t> , <http://ark.bnf.fr/ark:/12148/cb11934132b> , <http://ark.bnf.fr/ark:/12148/cb11939893p> , <http://ark.bnf.fr/ark:/12148/cb11933091b> ;
            <http://purl.org/dc/elements/1.1/language> <http://lexvo.org/id/iso639-3/fra> ;
            <http://www.language-archives.org/OLAC/1.1/depositor> <http://viaf.org/viaf/181558906> ;
            <http://purl.org/dc/elements/1.1/publisher> <http://viaf.org/viaf/181558906> ;
            <http://purl.org/dc/elements/1.1/rights> "Copyright (c) 2012 Université d'Orléans/LLL" ;
            <http://purl.org/dc/elements/1.1/title> "ESLO1: entretien 019"@fr ;
            <http://purl.org/dc/terms/alternative> "ESLO1: entretien 019 alt"@fr ;
            <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/eslo/ESLO1_ENT_019_22km.wav> , <http://cocoon.huma-num.fr/data/eslo/ESLO1_ENT_019.mp3> ;
            <http://purl.org/dc/terms/accessRights> " Freely available for non-commercial use" ;
            <http://purl.org/dc/terms/license> <http://creativecommons.org/licenses/by-nc-sa/3.0/> ;
            <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE> , <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ESLO1> ;
            <http://purl.org/dc/terms/extent> "PT0H47M16S" ;
            <http://purl.org/dc/terms/created> "1969-04-11"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/spatial> "FR"^^<http://purl.org/dc/terms/ISO3166> , <http://vocab.getty.edu/tgn/7008337> , "France, Centre, Loiret, Orléans"@fr ;
            <http://crdo.risc.cnrs.fr/schemas/master> <http://cocoon.huma-num.fr/data/eslo/masters/ESLO1_ENT_019.wav> ;
            <http://purl.org/dc/terms/available> "2014-11-06"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/issued> "2014-12-05T15:00:19+01:00"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://purl.org/dc/terms/modified> "2014-12-05T15:00:19+01:00"^^<http://purl.org/dc/terms/W3CDTF> ;
            <http://www.language-archives.org/OLAC/1.1/speaker> "JK" , "UG393" ;
            <http://purl.org/dc/elements/1.1/source> <http://archivesetmanuscrits.bnf.fr/ead.html?id=FRBNFEAD000095934&c=FRBNFEAD000095934_a19844296> ;
            <http://www.w3.org/2003/01/geo/wgs84_pos#long> "1.90"^^xsd:float ;
            <http://www.w3.org/2003/01/geo/wgs84_pos#lat> "47.90"^^xsd:float ;
            <http://purl.org/dc/terms/tableOfContents> "01 née; famille a toujours habité Orléans03 oui, on y a des amis, relations04 oui, certainementT1 centralisait comptabilité dans une industrieT2 RR passer des écritures, caisse; horaires avant et après loi de 1936T3 aimait travail, rien ne lui déplaisaitT4 n'a jamais pensé à çaT5 actuellement on est forcée quelquefois - assurance sociale - retraite appréciable; autre chose avec enfantL1 concerts, théâtre, réunions, etc...:préfère s'occuperL2 messe, puis a passé la journée avec une amieL3 (quand elle travaillait) au bord de la mer ou voyages organisésE1 n'en sait rienE2 lui semble que ça aide le françaisE3 peut-être chimie, physiqueE4 intelligence, travail, familleE5 dépend de leurs capacitésE6 non : dépend de la vie de l'enfant plus tardP1 oui, à part loyers chersP4 ne comprend pas motifs des étudiants; très calme à OrléansP9 moins qu'avant 1940P10 RR bourgeois = gros rentiers, femmes d'officierss, gros industriels; chaque groupe se recevait mutuellement, formait un clanQ. Orléans une ville renfermée? oui : orléanais assez distant, peut-être moins maintenantP6 il faut avoir la moitié des votesP7 des candidats se retirentP8 droite gaulliste, réservée; gauche avancéeP5 pas grand-chose - attachés à leurs partisP11 oui, maintenant, par l'éducation; famille moins importanteP12 à diminuer : les gens travaillent beaucoup plus et l'argent a changé de mainsOmelette (je casse mes oeufs...)Questionnaire sociolinguistique"@fr ;
            <http://purl.org/dc/terms/isRequiredBy> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019_C> , <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019_A> ;
            <http://www.language-archives.org/OLAC/1.1/editor> <http://viaf.org/viaf/39685504> , <http://viaf.org/viaf/200058210> ;
            <http://purl.org/dc/terms/abstract> " Identifiant du témoin : UG 393 Femme, 72 ans (né en 1897, à Orléans ) ; comptable en retraite Enregistré par Jack Kay, le 11 avril 1969, au domicile du témoin. Questionnaires enregistrés : questionnaire sociolinguistique; questionnaire ouvert(questions 01 ; 03-04 ; T1-T5 ; L1-L3 ; E1-E6 ; P1 ; P4 ; P9-P10 ; P6-P8 ; P5 ; P11-P12 ; omelette). Classe politique du témoin : Centre Remarques sur le témoignage : réponses concises. Intérêt pédagogique : conditions de travail avant la guerre de 1940 ; classes sociales. Acoustique : bonne qualité d'enregistrement."@fr ;
            <http://www.language-archives.org/OLAC/1.1/researcher> "Kay, Jack" ;
            <http://purl.org/dc/elements/1.1/relation> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENTCONT_233> ;
            <http://purl.org/dc/terms/medium> "1 bande magnétique audio"@fr ;
            <http://purl.org/dc/terms/bibliographicCitation> "A.-M. VURPAS (1988), Autrefois dans les monts du Beaujolais, Action graphique.";
            <http://purl.org/dc/elements/1.1/identifier> "Cote producteur: 001_3"@fr ;
            <http://purl.org/dc/elements/1.1/reference> <http://panini.u-paris10.fr> ;
            <http://purl.org/dc/elements/1.1/creator> "Blanc, Michel" , "Biggs, Patricia" .

EOT
    ];

    const TEST_INPUT_ID = "crdo-ESLO1_ENT_019";
    const TEST_GRAPH_URI = "http://corpusdelaparole.huma-num.fr/corpus/res/crdo-ESLO1_ENT_019";
    const TEST_CHO_URI = "http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019";


    private $inputGraphes = [];
    private $resGraphes = [];
    private $mappers = [];

    function __construct(string $name = null) {
        parent::__construct($name);
    }

    public function setUp() {
        parent::setUp();

        foreach(CocoonSoundRdfMapperTest::TEST_INPUT_DOCS as $key => $inputDoc) {
            $this->inputGraphes[$key] = new Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_019", $inputDoc);
            $this->mappers[$key] = new CocoonSoundRdfMapper($this->inputGraphes[$key]);
            $this->mappers[$key]->mapGraph();
            $this->resGraphes[$key] = $this->mappers[$key]->getOutputGraphes()[$this->inputGraphes[$key]->getUri()];
        }
    }

    public function tearDown() {
        parent::tearDown();
    }

    /**
     * Just test that the construct and setup are ok
     *
     * @return void
     */
    public function testInit()
    {
        $this->assertTrue(true);
    }

    /**
     * Test that the returned graph has the same uri that the original.
     *
     * @return void
     */
    public function testUri() {

         echo $this->resGraphes['BASE']->serialise('turtle');
         //echo var_export($this->resGraphes['BASE']->toRdfPhp());

         $this->assertNotNull($this->resGraphes['BASE']);
         $this->assertEquals(CocoonSoundRdfMapperTest::TEST_GRAPH_URI, $this->resGraphes['BASE']->getUri(), "URIS must be translated");
     }

    /**
      * Test that the return graph has one ore:Aggregation resource
      *
      * @return void
      */
    public function testAggregationResource() {

         $resources = $this->resGraphes['BASE']->allOfType('ore:Aggregation');

         $this->assertCount(1, $resources, "Must found only one resources of type ore:Aggregation");
         $this->assertEquals(CocoonSoundRdfMapperTest::TEST_GRAPH_URI,$resources[0]);
     }

    /**
      * test that there one (and only one) edm:dateProvider and edm:provider property
      *
      * @return void
      */
    public function testProviders() {

         $providers = $this->resGraphes['BASE']->all(CocoonSoundRdfMapperTest::TEST_GRAPH_URI,'edm:provider');

         $this->assertCount(1,$providers, "one and only one edm provider");
         $this->assertTrue($providers[0] instanceof Literal);
         $this->assertEquals('Corpus de la Parole', $providers[0]->getValue(), 'provider must be Corpus de la Parole');
         $this->assertEquals('fr', $providers[0]->getLang(), 'provider must be in french');

         $dataProviders = $this->resGraphes['BASE']->all(CocoonSoundRdfMapperTest::TEST_GRAPH_URI,'edm:dataProvider');

         $this->assertCount(1,$dataProviders, "one and only one edm data provider");
         //TODO: to change after viaf resolution
         $this->assertTrue($dataProviders[0] instanceof Resource);
         $this->assertEquals('http://viaf.org/viaf/181558906', $dataProviders[0]->getUri(), 'dataProvider uri must be http://viaf.org/viaf/181558906');
    }


    /**
     * test isShownAt and isShownBy
     *
     * @return void
     */
    public function testIsShown() {

        $isShownAts = $this->resGraphes['BASE']->all(CocoonSoundRdfMapperTest::TEST_GRAPH_URI, 'edm:isShownAt');

        $this->assertCount(1, $isShownAts, "One edm:isShownAt");
        $this->assertInstanceOf("EasyRdf\Resource", $isShownAts[0]);
        $this->assertEquals("http://cocoon.huma-num.fr/exist/crdo/meta/crdo-ESLO1_ENT_019", $isShownAts[0]);

        $isShownBys = $this->resGraphes['BASE']->all(CocoonSoundRdfMapperTest::TEST_GRAPH_URI, 'edm:isShownBy');

        $this->assertCount(1, $isShownBys, "One edm:isShownBy");
        $this->assertInstanceOf("EasyRdf\Resource", $isShownBys[0]);
        $this->assertEquals("http://cocoon.huma-num.fr/data/eslo/masters/ESLO1_ENT_019.wav", $isShownBys[0]);

    }

    /**
     * test rights
     *
     * @return void
     */
    public function testRights() {
        $rightsNodes = $this->resGraphes['BASE']->all(CocoonSoundRdfMapperTest::TEST_GRAPH_URI, 'edm:rights');

        $this->assertCount(1, $rightsNodes, "One edm:right");

        $rights = $rightsNodes[0];

        $this->assertTrue($rights instanceof Resource);

        $this->assertEquals("http://creativecommons.org/licenses/by-nc-sa/4.0/", $rights->getUri(), "right must be CC-BY-NC");
    }

    /**
     * Test that the return graph has one ore:Aggregation resource
     *
     * @return void
     */
    public function testProvidedCHO() {

         $resources = $this->resGraphes['BASE']->allOfType('edm:ProvidedCHO');

         $this->assertCount(1, $resources, "Must found only one resources of type edm:aggregatedCHO");
         $this->assertEquals(CocoonSoundRdfMapperTest::TEST_CHO_URI,$resources[0]);
     }

    /**
     * Test one to one mapping
     *
     * @return void
     */
    public function testOneToOneProvidedCHOIdentity() {

        $properties = [
            'http://purl.org/dc/terms/tableOfContents',
            'http://purl.org/dc/elements/1.1/description',
            'http://purl.org/dc/elements/1.1/language',
            'http://purl.org/dc/elements/1.1/publisher',
            'http://purl.org/dc/elements/1.1/rights',
            'http://purl.org/dc/terms/license',
            'http://purl.org/dc/elements/1.1/subject',
            'http://purl.org/dc/elements/1.1/title',
            'http://purl.org/dc/terms/available',
            'http://purl.org/dc/terms/created',
            'http://purl.org/dc/terms/issued',
            'http://purl.org/dc/terms/extent',
            'http://purl.org/dc/elements/1.1/type',
            'http://purl.org/dc/elements/1.1/language',
            'http://purl.org/dc/terms/medium',
            'http://purl.org/dc/elements/1.1/source',
            'http://purl.org/dc/terms/alternative',
            'http://purl.org/dc/terms/modified',
            'http://purl.org/dc/terms/abstract',
            'http://purl.org/dc/terms/bibliographicCitation',
            'http://purl.org/dc/elements/1.1/identifier',
            'http://purl.org/dc/terms/references',
            'http://purl.org/dc/elements/1.1/coverage',
            'http://purl.org/dc/elements/1.1/relation',
            'http://purl.org/dc/elements/1.1/creator',
        ];

        $providedCHO = $this->resGraphes['BASE']->get('edm:ProvidedCHO', '^rdf:type');
        $sourceNode = $this->inputGraphes['BASE']->get('http://crdo.risc.cnrs.fr/schemas/Resource', '^rdf:type');

        $this->assertNotNull($providedCHO);
        $this->assertNotNull($sourceNode);

        foreach ($properties as $prop) {
            $outputValuesStr = [];
            foreach($providedCHO->all($this->resGraphes['BASE']->resource($prop)) as $outputValue) {
                array_push($outputValuesStr, strval($outputValue));
            }
            $this->assertNotEmpty($outputValuesStr, "we must found some values to test $prop");
            foreach ($sourceNode->all($this->inputGraphes['BASE']->resource($prop)) as $value) {
                $this->assertContains(strval($value), $outputValuesStr, "$prop not found in output graph");
            }
        }

    }

    /**
     * Test mapping for isPartOf
     *
     * @return void
     */
    public function testOneToOneProvidedCHOIsPartOf() {
        $providedCHO = $this->resGraphes['BASE']->get('edm:ProvidedCHO', '^rdf:type');
        $sourceNode = $this->inputGraphes['BASE']->get('http://crdo.risc.cnrs.fr/schemas/Resource', '^rdf:type');

        $this->assertNotNull($providedCHO);
        $this->assertNotNull($sourceNode);

        $isGatheredInto = $providedCHO->all($this->resGraphes['BASE']->resource('http://www.europeana.eu/schemas/edm/isGatheredInto'));
        $this->assertCount(2, $isGatheredInto, "Must have two collection node");

        $ispartOf = $providedCHO->all($this->resGraphes['BASE']->resource('http://purl.org/dc/terms/isPartOf'));
        $this->assertCount(1, $ispartOf, "Must have one collection node");

        $outputValuesStr = [];
        foreach($providedCHO->all($this->resGraphes['BASE']->resource('http://www.europeana.eu/schemas/edm/isGatheredInto')) as $outputValue) {
            array_push($outputValuesStr, strval($outputValue));
        }
        $this->assertNotEmpty($outputValuesStr, "we must found some values to test isGatheredInto");

        $this->assertContains("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE", $outputValuesStr, "http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE not found in output graph");
        $this->assertContains("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ESLO1", $outputValuesStr, "http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ESLO1 not found in output graph");

        $this->assertEquals("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_010", strval($providedCHO->get($this->resGraphes['BASE']->resource('http://purl.org/dc/terms/isPartOf'))), "the isPartOf mus equals http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ESLO1_ENT_010");
    }

    /**
     * Test one to one mapping spatial info
     *
     * @return void
     */
    public function testOneToOneProvidedCHOSpatial() {
        $providedCHO = $this->resGraphes['BASE']->get('edm:ProvidedCHO', '^rdf:type');
        $sourceNode = $this->inputGraphes['BASE']->get('http://crdo.risc.cnrs.fr/schemas/Resource', '^rdf:type');

        $this->assertNotNull($providedCHO);
        $this->assertNotNull($sourceNode);

        $spatials = $providedCHO->all($this->resGraphes['BASE']->resource('http://purl.org/dc/terms/spatial'));

        $this->assertCount(1, $spatials, "Must have only one spatial node");
        $spatial = $spatials[0];
        $this->assertTrue($spatial->isBNode(),"spatial node must be blank");
        $this->assertEquals("edm:Place", $spatial->type(), "spatial node type must be edm:Place");

        $lats = $spatial->all('geo:lat');
        $this->assertCount(1, $lats, "One latitude");
        $this->assertInstanceOf("EasyRdf\Literal", $lats[0], "Latitude must be a litteral");
        $this->assertEquals(Literal::create("47.90", null, 'xsd:float'), $lats[0], "lat must be '47.90'^^xsd:float");

        $longs = $spatial->all('geo:long');
        $this->assertCount(1, $longs, "One longitude");
        $this->assertInstanceOf("EasyRdf\Literal", $longs[0], "Longitude must be a litteral");
        $this->assertEquals(Literal::create("1.90", null, 'xsd:float'), $longs[0], "long must be '1.90'^^xsd:float");

        $notes = $spatial->all('skos:note');
        $this->assertCount(2, $notes, "2 notes");
        $this->assertContainsOnlyInstancesOf("EasyRdf\Literal", $notes, "Notes mus be only literals");
        $this->assertEquals(Literal::create("FR", null, "dc:ISO3166"), $notes[0], "notes contains 'FR'^^dc:ISO3166");
        $this->assertEquals(Literal::create("France, Centre, Loiret, Orléans", "fr", null), $notes[1], "notes contains 'France, Centre, Loiret, Orléans'@fr");

        $sameas = $spatial->all('owl:sameAs');
        $this->assertCount(1, $sameas, "1 same as");
        $this->assertContainsOnlyInstancesOf("EasyRdf\Resource", $sameas, "Notes mus be only resources");
        $this->assertEquals('http://vocab.getty.edu/tgn/7008337', $sameas[0]->getUri(), "uri must be http://vocab.getty.edu/tgn/7008337");
    }

    /**
     * Test one to one mapping trim values
     *
     * @return void
     */
    public function testOneToOneProvidedCHOTrim() {

        $properties = [
            'http://purl.org/dc/terms/accessRights',
        ];

        $providedCHO = $this->resGraphes['BASE']->get('edm:ProvidedCHO', '^rdf:type');
        $sourceNode = $this->inputGraphes['BASE']->get('http://crdo.risc.cnrs.fr/schemas/Resource', '^rdf:type');

        $this->assertNotNull($providedCHO);
        $this->assertNotNull($sourceNode);

        foreach ($properties as $prop) {
            $outputValuesStr = [];
            foreach($providedCHO->all($this->resGraphes['BASE']->resource($prop)) as $outputValue) {
                array_push($outputValuesStr, strval($outputValue));
            }
            $this->assertNotEmpty($outputValuesStr, 'we must found some values to test');
            foreach ($sourceNode->all($this->inputGraphes['BASE']->resource($prop)) as $value) {
                $this->assertContains(trim(strval($value)), $outputValuesStr, "$prop not found in output graph");
            }
        }
    }

    /**
     * Test one to one mapping trim values
     *
     * @return void
     */
    public function testOneToOneProvidedCHOReference() {

        $providedCHO = $this->resGraphes['NO-FORMAT']->get('edm:ProvidedCHO', '^rdf:type');
        $sourceNode = $this->inputGraphes['NO-FORMAT']->get('http://crdo.risc.cnrs.fr/schemas/Resource', '^rdf:type');

        $this->assertNotNull($providedCHO);
        $this->assertNotNull($sourceNode);

        $outputValuesStr = [];
        foreach($providedCHO->all($this->resGraphes['NO-FORMAT']->resource('http://purl.org/dc/terms/references')) as $outputValue) {
            array_push($outputValuesStr, strval($outputValue));
        }
        $this->assertNotEmpty($outputValuesStr, 'we must found some values to test for http://purl.org/dc/elements/1.1/reference');
        foreach ($sourceNode->all($this->inputGraphes['NO-FORMAT']->resource('http://purl.org/dc/elements/1.1/reference')) as $value) {
            $this->assertContains(strval($value), $outputValuesStr, "http://purl.org/dc/terms/references not found in output graph");
        }
    }


    /**
     * Test that the web resources
     *
     * @return void
     */
    public function testWebResources() {

        $resources = $this->resGraphes['BASE']->allOfType('edm:WebResource');

        $this->assertCount(3, $resources, "Must found three webresource");

        $aggregation = $this->resGraphes['BASE']->resource(CocoonSoundRdfMapperTest::TEST_GRAPH_URI);

        foreach ($resources as $wres) {
            $mimetypes = $wres->all($this->resGraphes['BASE']->resource('http://purl.org/dc/elements/1.1/format'));
            $this->assertCount(1, $mimetypes, "Must find one mimetype.");
            $mimetype = $mimetypes[0];
            $this->assertInstanceOf("EasyRdf\Literal", $mimetype, "mimetype must be literal");
            $this->assertEquals("dc:IMT",$mimetype->getDatatype());

            $extends = $wres->all($this->resGraphes['BASE']->resource('http://purl.org/dc/terms/extent'));
            $this->assertCount(1, $extends, "Must find one extent.");
            $extent = $extends[0];
            $this->assertInstanceOf("EasyRdf\Literal", $extent, "extent must be literal");
            $this->assertStringStartsWith("PT", $extent->getValue(), "value mus start with PT");

            if($wres->hasProperty('edm:isDerivativeOf')) {
                $derivative = $wres->get('edm:isDerivativeOf');
                $this->assertInstanceOf("EasyRdf\Resource", $derivative, "Derivative must be a resource");
                $this->assertEquals("http://cocoon.huma-num.fr/data/eslo/masters/ESLO1_ENT_019.wav", $derivative->getUri(), "derivative uri must be http://cocoon.huma-num.fr/data/eslo/masters/ESLO1_ENT_019.wav");
                $this->assertTrue($aggregation->hasProperty('edm:hasView', $wres));
            }
            else {
                $this->assertTrue($aggregation->hasProperty('edm:isShownBy', $wres));
            }

        }
    }

    /**
     * Test that the web resources
     *
     * @return void
     */
    public function testWebResourcesDate() {

        $resources = $this->resGraphes['BASE']->allOfType('edm:WebResource');

        foreach ($resources as $wres) {
            $this->assertFalse($wres->hasProperty("http://purl.org/dc/terms/available"),"web resource must not have http://purl.org/dc/terms/available");
            $this->assertFalse($wres->hasProperty("http://purl.org/dc/terms/modified"),"web resource must not have http://purl.org/dc/terms/modified");
            $this->assertTrue($wres->hasProperty("http://purl.org/dc/terms/issued"), "Must have http://purl.org/dc/terms/issued");
            $this->assertTrue($wres->hasProperty("http://purl.org/dc/terms/created"), "Must have http://purl.org/dc/terms/created");

            $issued = $wres->getLiteral('dc:issued');
            //<http://purl.org/dc/terms/issued> "2014-12-05T15:00:19+01:00"^^<http://purl.org/dc/terms/W3CDTF> ;
            $this->assertInstanceOf('EasyRdf\Literal', $issued, "issued value must be a literal");
            $this->assertEquals('http://purl.org/dc/terms/W3CDTF', $issued->getDatatypeUri(), "issued datatype uri must be a http://purl.org/dc/terms/W3CDTF");
            $this->assertEquals('2014-12-05T15:00:19+01:00', $issued->getValue(), "Value must be 2014-12-05T15:00:19+01:00");

            $created = $wres->getLiteral('dc:created');
            //<http://purl.org/dc/terms/created> "1969-04-11"^^<http://purl.org/dc/terms/W3CDTF> ;
            $this->assertInstanceOf('EasyRdf\Literal', $created, "created value must be a literal");
            $this->assertEquals('http://purl.org/dc/terms/W3CDTF', $created->getDatatypeUri(), "created datatype uri must be a http://purl.org/dc/terms/W3CDTF");
            $this->assertEquals('1969-04-11', $created->getValue(), "Value must be 1969-04-11");

        }

    }

    /**
     * Test collection graph creation
     * @return void
     */
    public function testCollectionGraph() {
        $outputs = $this->mappers['BASE']->getOutputGraphes();
        $this->assertCount(3, $outputs, "Must have 3 graphes as output");
        foreach($outputs as $url => $outputGraph) {
            if($outputGraph->getUri() == 'http://corpusdelaparole.huma-num.fr/corpus/res/crdo-ESLO1_ENT_019') {
                continue;
            }
            $this->assertContains($outputGraph->getUri(), ['http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE', 'http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ESLO1']);
            $this->assertEquals($outputGraph->resource($outputGraph->getUri())->type(), 'edm:Collection');
        }
    }

}
