<?php

use CorpusParole\Models\Document;
use CorpusParole\Libraries\CocoonUtils;

/**
 *
 */
class DocumentTest extends TestCase {

    const TEST_DOC = <<<EOT
    @prefix ore: <http://www.openarchives.org/ore/terms/> .
    @prefix edm: <http://www.europeana.eu/schemas/edm/> .
    @prefix dc11: <http://purl.org/dc/elements/1.1/> .
    @prefix olac: <http://www.language-archives.org/OLAC/1.1/> .
    @prefix dc: <http://purl.org/dc/terms/> .
    @prefix geo: <http://www.w3.org/2003/01/geo/wgs84_pos#> .
    @prefix xsd: <http://www.w3.org/2001/XMLSchema#> .
    @prefix skos: <http://www.w3.org/2004/02/skos/core#> .
    @prefix owl: <http://www.w3.org/2002/07/owl#> .

    <http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND>
      a ore:Aggregation ;
      edm:aggregatedCHO <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND> ;
      edm:provider "Corpus de la Parole"@fr ;
      edm:dataProvider <http://viaf.org/viaf/142432638> ;
      edm:isShownAt <http://cocoon.huma-num.fr/exist/crdo/meta/crdo-CFPP2000_35_SOUND> ;
      edm:isShownBy <http://cocoon.huma-num.fr/data/archi/masters/372593.wav> ;
      edm:rights <http://creativecommons.org/licenses/by-nc-sa/4.0/> ;
      edm:hasView <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav>, <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3>, <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xml>,  <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xhtml>.

    <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-CFPP2000_35_SOUND>
      a edm:ProvidedCHO ;
      dc11:description "Enregistrement issu du Corpus de Français Parlé Parisien des années 2000 (CFPP2000)"@fr, "Enregistrement: CaractÃ©ristiques de l'enregistrement; IdentifiantÂ : CFPP2000 [03-01] Ozgur_Kilic_H_32_alii_3e; LangueÂ : franÃ§ais; EditeurÂ : UniversitÃ© Paris III â Syled ; Date : 17.11.2010; Lieu de l'enregistrementÂ : au domicile de lâenquÃªtÃ© Ãzgur KILIÃ ; EnquÃªteurÂ : Sonia BRANCA-ROSOFF; Quartier(s) concernÃ©(s)Â : Paris 3e, et 20e (pour lâÃ¢ge adulte); TranscriptionÂ : Noalig TANGUY; AnonymisationÂ : Noalig TANGUY; DurÃ©eÂ : 0h46â; "@fr, "EnquÃªteur: Etat civil; PrÃ©nom et NOMÂ : Sonia BRANCA-ROSOFF; SexeÂ : fÃ©minin; Lieu de naissanceÂ : New-York (USA); Lieu la petite enfanceÂ : Bretagne, puis rÃ©gion parisienne, puis de 8 Ã  16 ans Nice; Date de naissance : 16.12.1947; Situation familialeÂ : mariÃ©e, deux enfants; MobilitÃ© gÃ©ographiqueÂ : Paris 1967-1971, Aix-en-Provence 1971-1996, Paris 1996-2012; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : Doctorat dâÃtat (sciences du langage) 1987; Travail; ActivitÃ© actuelleÂ : Enseignante-chercheuse (linguistique franÃ§aise), UniversitÃ© de la Sorbonne Nouvelle-Paris III, Paris; ActivitÃ©s passÃ©esÂ : ; Langues; Langue premiÃ¨reÂ : franÃ§ais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : nÃ©e en 1916 Ã  Nice (France), interprÃ¨te de confÃ©rencesÂ ; licence dâanglais; PÃ¨reÂ : nÃ© en 1913 Ã  Mananjary (Madagascar), ingÃ©nieur.; "@fr, "EnquÃªtÃ© 1: Etat civil; PrÃ©nom et NOM fictifsÂ : Ãzgur KILIÃ ; SexeÂ : masculin; Lieu de naissance: Paris 11e, France; Lieu la petite enfanceÂ : Paris 11e, France; Date de naissance : 07.02.1979; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : pacsÃ©, deux enfants; MobilitÃ© gÃ©ographiqueÂ : 3e, 18e, 20e arrondissements; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec autres enquÃªtÃ©sÂ : amis; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire  st Merri 75003Â ; Secondaire : Victor Hugo, Turgot 75003, Jean LurÃ§at 75013 gÃ©nÃ©ral bac ES SupÃ©rieur. Puis St Denis universitÃ© (cinÃ©ma); Travail; ActivitÃ© actuelleÂ : ingÃ©nieur du son pour l'audiovisuel; ActivitÃ©s passÃ©esÂ : restauration, assistant son pour l'audiovisuel ; Langues; Langues pratiquÃ©esÂ : franÃ§ais, anglais, turc; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : psychothÃ©rapeute, nÃ© a Houilles rÃ©gion parisienne France; PÃ¨reÂ : professeur d'universitÃ©, nÃ© a Istanbul en Turquie.  ; "@fr, "EnquÃªtÃ© 2: Etat civil; PrÃ©nom et NOM fictifsÂ : Michel CHEVRIER; SexeÂ : masculin; Lieu de naissance : naissance Ã  Boulogne-Billancourt (92); Lieu la petite enfance : rue du Temple, 3e arrondissement, Paris; Date de naissance : 01.06.1979; Age au moment de lâenregistrementÂ : 31 ans; Situation familialeÂ : concubinage; MobilitÃ© gÃ©ographiqueÂ : 3e, puis 20e (entre 2005 et 2009) puis Romainville (depuis 2009); Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; PrÃ©nom et NOM rÃ©elsÂ : ; RÃ©seau par lequel contactÃ©Â : ; Adresse (si connue)Â : ; TÃ©lÃ©phone (si connu)Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ãcole Ã  aire ouverte rue St Merri 75003Â ; Secondaire (CollÃ¨ge Victor Hugo 75003) puis LycÃ©e AutogÃ©rÃ© de Paris (75015) Bac LittÃ©raire. Deux annÃ©es au CIM (Ã©cole de jazz) dans le 75018, puis 3 ans au conservatoire du 9eme; Travail; ActivitÃ© actuelleÂ : Musicien, Compositeur, Professeur de piano jazz ; ActivitÃ©s passÃ©esÂ : Musicien; Langues; Langues pratiquÃ©esÂ : franÃ§ais, anglais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : ancienne mÃ©decin gÃ©nÃ©raliste pneumologue, puis sociologue, nÃ©e en France; PÃ¨reÂ : enseignant en sciences de l'Ã©ducation Ã  Paris 8 nÃ© en Belgique; "@fr, "EnquÃªtÃ© 3: Etat civil; PrÃ©nom et NOM fictifsÂ : BenoÃ®t DU BREUIL-DE-PONT-EN-AUGE; SexeÂ : masculin; Lieu de naissance : Paris 14e; Lieu la petite enfance :  Paris 3e; Date de naissance : 28.11.1978; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : concubinage avÃ©rÃ©, avec papier de la mairie tamponnÃ© et signÃ©!; MobilitÃ© gÃ©ographiqueÂ : Ã  2 ans vit dans le 9Â°, puis dÃ©mÃ©nage pour le 3Â°, est restÃ© dans la mÃªme rue pendant 20 ans tout en changeant d'immeuble. Ensuite le 19Â° pendant 4 ans, puis Pantin 6 mois puis Les Lilas. ; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec autres enquÃªtÃ©sÂ : voir ScolaritÃ©, PrÃ©cisions; PrÃ©nom et NOM rÃ©elsÂ : ; RÃ©seau par lequel contactÃ©Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [ ] Post-licence [ ] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : CAP; PrÃ©cisionsÂ : Primaire Ã©cole Ã  aire ouverte (St Merri) dans le 4Â° avec Augustin, Ãzgur, Michel. Secondaire : collÃ¨ge Victor-Hugo dans le 3Â° avec Ãzgur ; puis CAP ; Travail; ActivitÃ© actuelleÂ : ; ActivitÃ©s passÃ©esÂ : Ã©bÃ©niste agenceur puis Ã©bÃ©niste restaurateur et enfin constructeur dÃ©cors (ainsi que de nombreux petits jobs ayant durÃ© moins d'un an); Langues; Langues pratiquÃ©esÂ : franÃ§ais; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : Ã  la retraite ; secrÃ©taire de rÃ©daction dans un grand journal, baccalaurÃ©at ; nÃ©e Ã  Montrouge; PÃ¨reÂ : conseiller Ã  travail Ã  pole emploi, nÃ© Ã  Boulogne Billancourt, baccalaurÃ©at prÃ©sentÃ© 3 ou 4 fois, puis obtenu par Ã©quivalence. ; "@fr, "EnquÃªtÃ© 4: Etat civil; EnquÃªtÃ© 4; PrÃ©nom et NOM fictifsÂ : ; SexeÂ : fÃ©minin / masculin; Lieu de naissance : Paris 14e; Lieu de la petite enfance : Paris 4e et 3e ; Date de naissance : 06.12.1976; Age au moment de lâenregistrementÂ : 34 ans; Situation familialeÂ : ; MobilitÃ© gÃ©ographiqueÂ : Rue la BruyÃ¨re 75009 (1976-1978), Rambuteau 75004 (1978-1987/88) & 75003 (chgt de cÃ´tÃ© de rue 1988-1998), a quittÃ© Paris de 1998 Ã  2005Â ; Rue Rambuteau 75003 (2006-2010), rue Gossec 75012 (2011); Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : aucune; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© 2Â : ami; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; RÃ©seau par lequel contactÃ©Â : ; ScolaritÃ©; Niveau : Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [X] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ecole St Merri 75003. Secondaire : CollÃ¨ge FranÃ§ois Couperin 75004, Institut St Pierre Fourier 75012Â ; Cours Charlemagne 75014 ; Travail; ActivitÃ© actuelleÂ : Superviseur Centre d'appels ; ActivitÃ©s passÃ©esÂ : Animateur Club Enfant, RÃ©ceptionniste ; Langues; Langues pratiquÃ©esÂ :; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : retraitÃ©e, Paris, bac + 1/2 (?) ; PÃ¨reÂ : Conseiller PÃ´le Emploi, Paris, bac. ; "@fr, "EnquÃªtÃ© 5: Etat civil; PrÃ©nom et NOM fictifsÂ : Ãtienne SALVEGAS; SexeÂ : masculin; Lieu de naissance : Paris 12e; Lieu de la petite enfance : Paris 3e Le Marais; Date de naissance :  16.06.1978; Age au moment de lâenregistrementÂ : 32 ans; Situation familialeÂ : mariÃ© 1 enfant, 12 ans de vie commune ; MobilitÃ© gÃ©ographiqueÂ : 3e (1978-1999) 19e (1999-2002) 9e (2002-2011) ; Relation (parentÃ©, ami, etc) avec EnquÃªteurÂ : ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© 2 : ami ; Relation (parentÃ©, ami, etc) avec EnquÃªtÃ© NOMÂ : ; PrÃ©nom et NOM rÃ©elsÂ : ; ScolaritÃ©; Primaire [X] Secondaire [X] ; SupÃ©rieurÂ : Licence [X] Post-licence [ ] ; Dernier diplÃ´me, avec annÃ©e dâobtentionÂ : ; PrÃ©cisionsÂ : Primaire : Ecole Vertus, rue des Vertus 3Ã¨me. Secondaire CollÃ¨ge Montgolfier / LycÃ©e Turgot 3Ã¨me. SupÃ©rieur Droit Univ. Paris PanthÃ©on Sorbonne ; Travail; ActivitÃ© actuelleÂ : Ã©vÃ¨nementiel; ActivitÃ©s passÃ©esÂ : stagiaire journaliste sportif / relations publiques, accueil, agent de  sÃ©curitÃ©, remplacement gardien, rÃ©ceptionniste hÃ´tellerie.; Langues; Langues pratiquÃ©esÂ :; Parents (profession, lieu de naissance, scolaritÃ©â¦); MÃ¨reÂ : nÃ© Ã  Paris, lycÃ©e ; PÃ¨reÂ : nÃ© Ã  Oloron (64), lycÃ©e ; "@fr ;
      dc11:language <http://lexvo.org/id/iso639-3/fra> ;
      dc11:publisher <http://viaf.org/viaf/142432638> ;
      dc11:type <http://purl.org/dc/dcmitype/Sound>, "primary_text"^^olac:linguistic-type, "narrative"^^olac:discourse-type, "report"^^olac:discourse-type, "unintelligible_speech"^^olac:discourse-type ;
      dc:license <http://creativecommons.org/licenses/by-nc-sa/3.0/> ;
      dc11:subject "lexicography"^^olac:linguistic-field, "phonetics"^^olac:linguistic-field, "anthropological_linguistics"^^olac:linguistic-field, "general_linguistics"^^olac:linguistic-field, <http://lexvo.org/id/iso639-3/fra>, "text_and_corpus_linguistics"^^olac:linguistic-field, "phonology"^^olac:linguistic-field, "semantics"^^olac:linguistic-field, "sociolinguistics"^^olac:linguistic-field, "syntax"^^olac:linguistic-field, "typology"^^olac:linguistic-field, "discourse_analysis"^^olac:linguistic-field, "historical_linguistics"^^olac:linguistic-field, "language_documentation"^^olac:linguistic-field, "mathematical_linguistics"^^olac:linguistic-field ;
      dc11:title "CFPP2000 [03-01] Ozgur_Kilic_H_32_alii_3e Entretien de Ozgur KILIÇ 2"@fr ;
      dc:accessRights "Freely available for non-commercial use" ;
      dc:extent "PT48M26S" ;
      dc:isPartOf <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE>, <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_CFPP2000> ;
      dc11:source "CFPP2000"@fr ;
      dc11:coverage "Quartier concerné : 3e"@fr ;
      olac:depositor <http://viaf.org/viaf/93752300> ;
      dc11:contributor <http://viaf.org/viaf/93752300>, "Branca-Rosoff, Sonia", "Kiliç Ozgur", "Chevrier Michel", "du-Breuil-de-Pont-en-Auge Benoît", "du-Breuil-de-Pont-en-Auge Augustin", "Salvegas Etienne", "Tanguy Noalig" ;
      olac:interviewer "Branca-Rosoff, Sonia" ;
      olac:responder "Kiliç Ozgur", "Chevrier Michel", "du-Breuil-de-Pont-en-Auge Benoît", "du-Breuil-de-Pont-en-Auge Augustin", "Salvegas Etienne" ;
      olac:transcriber "Tanguy Noalig" ;
      dc:available "2013-10-12"^^dc:W3CDTF ;
      dc:created "2010-11-17"^^dc:W3CDTF ;
      dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF ;
      dc:spatial [
        a edm:Place ;
        geo:lat "48.83975"^^xsd:float ;
        geo:long "2.3542"^^xsd:float ;
        skos:note "FR"^^dc:ISO3166, "France, Paris, Université Sorbonne Nouvelle Paris 3, site Censier"@fr ;
        owl:sameAs <http://vocab.getty.edu/tgn/7008038>
      ] .

    <http://cocoon.huma-num.fr/data/archi/masters/372593.wav>
      a edm:WebResource ;
      dc:extent "PT48M26S" ;
      dc11:format "audio/x-wav"^^dc:IMT ;
      dc:created "2010-11-17"^^dc:W3CDTF ;
      dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF .

    <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav>
      a edm:WebResource ;
      dc:extent "PT48M26S" ;
      dc11:format "audio/x-wav"^^dc:IMT ;
      edm:isDerivativeOf <http://cocoon.huma-num.fr/data/archi/masters/372593.wav> ;
      dc:created "2010-11-17"^^dc:W3CDTF ;
      dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF .

    <http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3>
      a edm:WebResource ;
      dc:extent "PT48M26S" ;
      dc11:format "audio/mpeg"^^dc:IMT ;
      edm:isDerivativeOf <http://cocoon.huma-num.fr/data/archi/masters/372593.wav> ;
      dc:created "2010-11-17"^^dc:W3CDTF ;
      dc:issued "2013-10-12T14:35:57+02:00"^^dc:W3CDTF .

    <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xml>
        a edm:WebResource ;
        dc11:format "application/xml"^^dc:IMT ;
        dc:created "2010-11-17"^^dc:W3CDTF ;
        dc:issued "2013-11-04T22:20:07+01:00"^^dc:W3CDTF .

    <http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xhtml>
        a edm:WebResource ;
        dc11:format "application/xhtml+xml"^^dc:IMT ;
        dc:created "2010-11-17"^^dc:W3CDTF ;
        dc:issued "2013-11-04T22:20:07+01:00"^^dc:W3CDTF .
EOT;


    public function setUp() {

        parent::setup();
        $this->graph = new EasyRdf\Graph("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", DocumentTest::TEST_DOC);

    }

    public function testConstructor() {
        $this->assertNotNull($this->graph, 'Graph shoud not be null');

        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $this->assertEquals('crdo-CFPP2000_35_SOUND',$doc->getId(),'Must have the correct id');
    }

    public function testTitle() {
        $this->assertNotNull($this->graph, 'Graph shoud not be null');

        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $this->assertEquals("CFPP2000 [03-01] Ozgur_Kilic_H_32_alii_3e Entretien de Ozgur KILIÇ 2",$doc->getTitle(),'Must have correct title');
        $this->assertInstanceOf(EasyRdf\Literal::class, $doc->getTitle(), "Title must be a literal");
        $this->assertEquals('fr', $doc->getTitle()->getLang(), "Language title must be fr");
    }

    public function testModified() {
        $this->assertNotNull($this->graph, 'Graph shoud not be null');

        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);
        $this->assertInstanceOf(EasyRdf\Literal::class, $doc->getModified(), "Modified must be a literal");
        $this->assertEquals("http://purl.org/dc/terms/W3CDTF", $doc->getModified()->getDatatypeURI(), "type must be http://purl.org/dc/terms/W3CDTF");
        $this->assertEquals("2013-10-12T14:35:57+02:00", $doc->getModified(), "modified must be 2013-10-12T14:35:57+02:00");

    }


    public function testPublisher() {

        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $this->assertCount(1, $doc->getPublishers(), 'Publisher is an array of size 1');
        $this->assertInstanceOf('EasyRdf\Resource', $doc->getPublishers()[0], 'publisher is a resource');
        $this->assertEquals("http://viaf.org/viaf/142432638", $doc->getPublishers()[0]->getUri(),"");
    }

    public function testMediaArray() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $this->assertCount(5, $doc->getMediaArray(), "Media array must be of size 5");
        foreach($doc->getMediaArray() as $url => $media) {
            $this->assertCount(5, $media, "media is a 5 element array");
            $this->assertArrayHasKey('format', $media, "media has 'format key'");
            $this->assertArrayHasKey('url', $media, "media has url");
            $this->assertArrayHasKey('extent', $media, "media has extent");
            $this->assertArrayHasKey('extent_ms', $media, "media has extent_ms");
            $this->assertArrayHasKey('master', $media, "media has master");

            $this->assertEquals($media['url'], $url);

            $this->assertContains(
                $media['url'],
                [ "http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav",
                  "http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3",
                  "http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xml",
                  "http://cocoon.huma-num.fr/exist/crdo/cfpp2000/fra/Ozgur_Kilic_H_32_alii_3e-2.xhtml",
                  "http://cocoon.huma-num.fr/data/archi/masters/372593.wav"
                ]
            );
            if($url === "http://cocoon.huma-num.fr/data/archi/masters/372593.wav") {
                $this->assertEquals('audio/x-wav', $media['format']);
                $this->assertTrue($media['master'] === true, "should be master");
            }
            else {
                $this->assertTrue($media['master'] === false, "should not be master");
            }

            if( $url === "http://cocoon.huma-num.fr/data/archi/masters/372593.wav" ||
                $url === "http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.mp3" ||
                $url === "http://cocoon.huma-num.fr/data/cfpp2000/Ozgur_Kilic_H_32_alii_3e-2.wav" ) {
                $this->assertEquals("PT48M26S", $media['extent'], "extent is PT48M26S");
                $this->assertGreaterThan(0, $media['extent_ms'], "extent_ms must be > 0");
                $this->assertStringStartsWith('audio/', $media['format']);
            }
            else {
                $this->assertNull($media['extent'], "no media extent");
                $this->assertNull($media['extent_ms'], "extend in ms is null");
                $this->assertStringStartsWith('application/', $media['format']);
            }

        }
    }

    public function testGetTypes() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $this->assertCount(5, $doc->getTypes(), "types array must be of size 5");

        foreach($doc->getTypes() as $type) {
            $this->assertThat(
                $type,
                $this->logicalXor(
                    $this->isInstanceOf(EasyRdf\Literal::class),
                    $this->isInstanceOf(EasyRdf\Resource::class)
                )
            );
        }
    }

    public function testGetOtherTypes() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $this->assertCount(2, $doc->getOtherTypes(), "types array must be of size 3");

        foreach($doc->getTypes() as $type) {
            $this->assertThat(
                $type,
                $this->logicalXor(
                    $this->isInstanceOf(EasyRdf\Literal::class),
                    $this->isInstanceOf(EasyRdf\Resource::class)
                )
            );
        }
    }

    public function testGetDiscourseTypes() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $this->assertCount(3, $doc->getDiscourseTypes(), "types array must be of size 3");

        $this->assertContainsOnlyInstancesOf("EasyRdf\Literal", $doc->getDiscourseTypes(), "Result contains only literals");
        $type = $doc->getDiscourseTypes()[0];
        $this->assertEquals("narrative", $type->getValue(), "discourse type is narrative");
        $this->assertEquals("http://www.language-archives.org/OLAC/1.1/discourse-type", $type->getDatatypeUri(), "discourse type url");
    }

    public function testCloneDocument() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $doc2 = clone $doc;

        $this->assertNotSame($doc, $doc2, "documents must not be the same");
        $this->assertNotSame($doc->getGraph(), $doc2->getGraph(), "documents must not be the same");

        $this->assertTrue(EasyRdf\Isomorphic::isomorphic($doc->getGraph(), $doc2->getGraph()),"graph must be isomorphic");
    }

    public function testIsIsomorphic() {
        $doc1 = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);
        $doc2 = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", new EasyRdf\Graph("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", DocumentTest::TEST_DOC));

        $this->assertTrue($doc1->isIsomorphic($doc2),"document must be isomorphic");

        $doc2->addLiteral('dc11:type', new EasyRdf\Literal("oratory", null, Config::get('OLAC_DISCOURSE_TYPE')['uri']));

        $this->assertFalse($doc1->isIsomorphic($doc2),"document must not be isomorphic");
    }

    public function testUpdateDiscourseTypes() {

        $newDiscourseTypes = ['oratory','dialogue','narrative', 'formulaic', 'ludic'];

        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);
        $this->assertCount(3, $doc->getDiscourseTypes(), "types array must be of size 3");

        $doc->updateDiscourseTypes($newDiscourseTypes);

        $this->assertCount(5, $doc->getDiscourseTypes(), "types array must be of size 5");

        $discourseTypes = $doc->getDiscourseTypes();
        foreach($newDiscourseTypes as $dt) {
            $this->assertContains($dt, $discourseTypes, "all discourse types must be in result list");
        }

    }

    public function testUpdateDiscourseTypesDelta() {

        $newDiscourseTypes = ['oratory','dialogue','narrative', 'formulaic', 'ludic'];

        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $doc->updateDiscourseTypes($newDiscourseTypes);

        $this->assertTrue($doc->isDirty());

        $this->assertEquals(1, $doc->deltaCount(), "There is one delta");

        $delta = $doc->getDeltaList()[0];

        $this->assertEquals("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $delta->getDeletedGraph()->getUri(), "uri of deleted graph must be ok");
        $this->assertEquals("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $delta->getAddedGraph()->getUri(), "uri of added graph must be ok");

        $this->assertEquals(3, $delta->getDeletedGraph()->countTriples(), "deleted graph must have only 3 triples");
        $this->assertEquals(5, $delta->getAddedGraph()->countTriples(), "deleted graph must have only 5 triples");

        $resQueryDiscourseType = $delta->getAddedGraph()->allLiterals($doc->getProvidedCHO(), 'dc11:type');
        foreach($resQueryDiscourseType as $dt) {
            $this->assertInstanceOf(EasyRdf\Literal::class, $dt, "This must be a litteral");
            $this->assertEquals('http://www.language-archives.org/OLAC/1.1/discourse-type', $dt->getDatatypeUri(), "The type of the Litteral must be correct");
        }
        foreach($newDiscourseTypes as $dt) {
            $this->assertContains($dt, $resQueryDiscourseType, "all discourse types must be in result list");
        }

    }

    public function testUpdateTitle() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $oldTitle = $doc->getTitle();

        $doc->setTitle("new title", "en");

        $this->assertEquals("new title", $doc->getTitleValue());
        $this->assertEquals("new title", $doc->getTitle()->getValue());
        $this->assertEquals("en", $doc->getTitle()->getLang());

        $this->assertTrue($doc->isDirty());
        $this->assertEquals(1, $doc->deltaCount(), "There is one delta");

        $delta = $doc->getDeltaList()[0];

        $addedTitles = $delta->getAddedGraph()->allLiterals($doc->getProvidedCHO(), '<http://purl.org/dc/elements/1.1/title>');
        $this->assertCount(1, $addedTitles);

        $removedTitles = $delta->getDeletedGraph()->allLiterals($doc->getProvidedCHO(), '<http://purl.org/dc/elements/1.1/title>');
        $this->assertCount(1, $removedTitles);


    }

    public function testUpdateDiscourseTypesIsomorphic() {

        $newDiscourseTypes = ['oratory','dialogue','narrative'];

        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);
        $doc->updateDiscourseTypes($newDiscourseTypes);

        $doc2 = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", new EasyRdf\Graph("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", DocumentTest::TEST_DOC));

        $this->assertFalse($doc->isIsomorphic($doc2),"document must not be isomorphic after adding discourse type");
    }

    public function testGetContributors() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $contributors = $doc->getContributors();

        $this->assertNotEmpty($contributors, "The contributors array should not be empty");
        $this->assertCount(8, $contributors, "The contributors array should have 8 elements");

        foreach ($contributors as $contribDef) {
            $this->assertArrayHasKey('name', $contribDef, "ContribDef must have name key");
            $this->assertArrayHasKey('url', $contribDef, "ContribDef must have url key");
            $this->assertArrayHasKey('role', $contribDef, "ContribDef must have role key");
            $this->assertContains($contribDef['role'], CocoonUtils::OLAC_ROLES, "Role should be in OLAC_ROLES");
        }
    }

    public function testSetContributors() {
        $doc = new Document("http://corpusdelaparole.huma-num.fr/corpus/res/crdo-CFPP2000_35_SOUND", $this->graph);

        $contributors = $doc->getContributors();

        $contribList = [[
            "name"=> "Guylaine Brun-Trigaud",
            "url"=> "http://viaf.org/viaf/56666014",
            "role"=> "http://www.language-archives.org/OLAC/1.1/data_inputter"
        ], [
            "name"=> "LDOR",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/depositor"
        ], [
            "name"=> "Thésaurus Occitan",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/depositor"
        ], [
            "name"=> "Équipe de Recherche en Syntaxe et Sémantique",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/editor"
        ], [
            "name"=> "Bases, corpus, langage",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/editor"
        ], [
            "name"=> "Patrick Sauzet",
            "url"=> "http://viaf.org/viaf/51700729",
            "role"=> "http://www.language-archives.org/OLAC/1.1/researcher"
        ], [
            "name"=> "Alazet, Pierre",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/speaker"
        ], [
            "name"=> "Del Duca, Jeanne",
            "url"=> null,
            "role"=> "http://www.language-archives.org/OLAC/1.1/transcriber"
        ], [
            "name"=> "Jane Austen, 1775-1817",
            "url"=> "http://viaf.org/viaf/102333412",
            "role"=> "http://www.language-archives.org/OLAC/1.1/compiler"
        ]];

        $doc->setContributors($contribList);

        $newContribs = $doc->getContributors();

        $this->assertCount(9, $newContribs);

        $this->assertTrue($doc->isDirty());
        $this->assertEquals(1, $doc->deltaCount(), "There is one delta");

        $delta = $doc->getDeltaList()[0];

        $addedGraph = $delta->getAddedGraph();
        $this->assertEquals(9, $addedGraph->countTriples());

        $removedGraph = $delta->getDeletedGraph();
        $this->assertEquals(count($contributors), $removedGraph->countTriples());

        $foundJaneAusten = false;
        foreach ($newContribs as $contribDef) {
            if(!is_null($contribDef['nameLiteral'])) {
                $lit = $contribDef['nameLiteral'];
                $this->assertNull($lit->getDatatype(), "Data type must be null $lit");
                $this->assertNotNull($lit->getLang(), "lang must not be null $lit");
            }
            if($contribDef['url'] == 'http://viaf.org/viaf/102333412') {
                $this->assertNull($contribDef['name'], 'Name must be null');
                $this->assertNull($contribDef['nameLiteral'], 'Name literal must be null');
                $foundJaneAusten = true;
            }
        }
        $this->assertTrue($foundJaneAusten, "Jane austenn not foud");

    }

}
