<?php

use CorpusParole\Libraries\Filters\CorpusFilterManager;

class CorpusFilterManagerTest extends TestCase
{
    /**
     * test prepare languages noop
     *
     * @return void
     */
    public function testPrepareLanguagesNoOp()
    {
        $languagesInput = ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/oci'];
        $languagesOutput = CorpusFilterManager::prepareLanguages($languagesInput);
        $this->assertEquals($languagesOutput, $languagesInput);
    }

    /**
     * test prepare languages unknown
     *
     * @return void
     */
    public function testPrepareLanguagesUnkown()
    {
        $languagesInput = ['foobar'];
        $languagesOutput = CorpusFilterManager::prepareLanguages($languagesInput);
        $this->assertEquals($languagesOutput, ['http://lexvo.org/id/iso639-3/foobar']);
    }

    /**
     * test prepare languages addPrefix
     *
     * @return void
     */
    public function testPrepareLanguagesAddPrefix()
    {
        $languagesInput = ['fra', 'gsw', 'bre', 'oci'];
        $languagesOutput = CorpusFilterManager::prepareLanguages($languagesInput);
        $this->assertEquals($languagesOutput, ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/oci']);
    }

    /**
     * test prepare languages mix
     *
     * @return void
     */
    public function testPrepareLanguagesMix()
    {
        $languagesInput = ['fra', 'http://lexvo.org/id/iso639-3/gsw', 'bre', 'http://lexvo.org/id/iso639-3/oci'];
        $languagesOutput = CorpusFilterManager::prepareLanguages($languagesInput);
        $this->assertEquals($languagesOutput, ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/oci']);
    }

    /**
     * test prepare languages recursion simple
     *
     * @return void
     */
    public function testPrepareLanguagesRecusionSimple()
    {
        $languagesInput = ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'corpus-oil'];
        $languagesOutput = CorpusFilterManager::prepareLanguages($languagesInput);
        $this->assertEquals($languagesOutput, ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/bre', 'http://lexvo.org/id/iso639-3/pcd']);
    }

    /**
     * test prepare languages recursion simple
     *
     * @return void
     */
    public function testPrepareLanguagesRecusionComplex()
    {
        $languagesInput = ['fra', 'http://lexvo.org/id/iso639-3/gsw', 'corpus-oil', 'corpus-regionals'];
        $languagesOutput = CorpusFilterManager::prepareLanguages($languagesInput);
        sort($languagesOutput);
        $languagesExpected = ['http://lexvo.org/id/iso639-3/fra', 'http://lexvo.org/id/iso639-3/gsw', 'http://lexvo.org/id/iso639-3/pcd',
             'http://lexvo.org/id/iso639-3/frp', 'http://lexvo.org/id/iso639-3/cos', 'http://lexvo.org/id/iso639-3/rcf',
             'http://lexvo.org/id/iso639-3/gcf'];
        sort($languagesExpected);
        $this->assertEquals($languagesOutput, $languagesExpected);
    }

    /**
     * test prepare location no operation
     *
     * @return void
     */
    public function testPrepareLocationNoOp()
    {
        $locationInput = '3030293';
        $locationOutput = CorpusFilterManager::prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }

    /**
     * test prepare location array
     *
     * @return void
     */
    public function testPrepareLocationArray()
    {
        $locationInput = ['3030293', 'foobar'];
        $locationOutput = CorpusFilterManager::prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }


    /**
     * test prepare location unknown
     *
     * @return void
     */
    public function testPrepareLocationUnknown()
    {
        $locationInput = 'foobar';
        $locationOutput = CorpusFilterManager::prepareLocation($locationInput);
        $this->assertEquals('foobar', $locationOutput);
    }

    /**
     * test prepare location geonames
     *
     * @return void
     */
    public function testPrepareLocationGeonames()
    {
        $locationInput = 'http://sws.geonames.org/3030293';
        $locationOutput = CorpusFilterManager::prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }

    /**
     * test prepare location geonames array
     *
     * @return void
     */
    public function testPrepareLocationGeonamesArray()
    {
        $locationInput = ['http://sws.geonames.org/3030293', 'http://www.geonames.org/3017382'];
        $locationOutput = CorpusFilterManager::prepareLocation($locationInput);
        $this->assertEquals('3030293', $locationOutput);
    }

}
