<?php

use Illuminate\Foundation\Testing\WithoutMiddleware;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;

use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Middleware;
use GuzzleHttp\Exception\RequestException;



class ViafResolverTest extends TestCase
{
    const VIAF_RDF_56666014 = <<<EOT
<?xml version="1.0" encoding="UTF-8"?><rdf:RDF xmlns:schema="http://schema.org/" xmlns:genont="http://www.w3.org/2006/gen/ont#" xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#" xmlns:bgn="http://bibliograph.net/" xmlns:umbel="http://umbel.org/umbel#" xmlns:pto="http://www.productontology.org/id/"><rdf:Description rdf:about="http://viaf.org/viaf/56666014/"><rdf:type rdf:resource="http://xmlns.com/foaf/0.1/Document"/><rdf:type rdf:resource="http://www.w3.org/2006/gen/ont#InformationResource"/><void:inDataset xmlns:void="http://rdfs.org/ns/void#" rdf:resource="http://viaf.org/viaf/data"/><foaf:primaryTopic xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/56666014"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/56666014"><dcterms:identifier xmlns:dcterms="http://purl.org/dc/terms/">56666014</dcterms:identifier><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><schema:birthDate>1961-10-23</schema:birthDate><schema:name xml:lang="fr-FR">Guylaine Brun-Trigaud</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#" xml:lang="en">Warning: skos:prefLabels are not ensured against change!</rdfs:comment><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#" xml:lang="fr-FR">Guylaine Brun-Trigaud</skos:prefLabel><schema:name xml:lang="nl-NL">Guylaine Brun-Trigaud</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#" xml:lang="en">Warning: skos:prefLabels are not ensured against change!</rdfs:comment><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#" xml:lang="nl-NL">Guylaine Brun-Trigaud</skos:prefLabel><schema:name xml:lang="en-US">Guylaine Brun-Trigaud</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#" xml:lang="en">Warning: skos:prefLabels are not ensured against change!</rdfs:comment><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#" xml:lang="en-US">Guylaine Brun-Trigaud</skos:prefLabel><schema:name xml:lang="en">Guylaine Brun-Trigaud</schema:name><schema:name xml:lang="fr-FR">Guylaine Brun-Trigaud</schema:name><schema:alternateName>Guylaine Brun- Trigaud</schema:alternateName><schema:givenName>Guylaine Brun-</schema:givenName><schema:familyName>Trigaud</schema:familyName><schema:alternateName>Trigaud</schema:alternateName><schema:sameAs><rdf:Description rdf:about="http://data.bnf.fr/ark:/12148/cb122483560#foaf:Person"/></schema:sameAs><schema:sameAs><rdf:Description rdf:about="http://id.loc.gov/authorities/names/nr92014179"/></schema:sameAs><schema:sameAs><rdf:Description rdf:about="http://www.idref.fr/031227201/id"/></schema:sameAs><schema:sameAs><rdf:Description rdf:about="http://isni.org/isni/0000000000375823"/></schema:sameAs></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/BNF%7C12248356#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/BNF"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Brun-Trigaud, Guylaine, 1961-....</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Trigaud Guylaine Brun-</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/56666014"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/NTA%7C072991968#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/NTA"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Brun-Trigaud, Guylaine</skos:prefLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/56666014"/><schema:url>http://opc4.kb.nl/PPN?PPN=072991968</schema:url></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/LC%7Cnr+92014179#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/LC"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Brun-Trigaud, Guylaine</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Trigaud, Guylaine Brun-</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/56666014"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/ISNI%7C0000000000375823#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/ISNI"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Brun-Trigaud, Guylaine, 1961-....</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Trigaud, Guylaine Brun-</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/56666014"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/SUDOC%7C031227201#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/SUDOC"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Brun-Trigaud, Guylaine, 1961-....</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Trigaud, Guylaine Brun-</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/56666014"/></rdf:Description></rdf:RDF>
EOT;
    const VIAF_RDF_93752300 = <<<EOT
<?xml version="1.0" encoding="UTF-8"?><rdf:RDF xmlns:schema="http://schema.org/" xmlns:genont="http://www.w3.org/2006/gen/ont#" xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#" xmlns:bgn="http://bibliograph.net/" xmlns:umbel="http://umbel.org/umbel#" xmlns:pto="http://www.productontology.org/id/"><rdf:Description rdf:about="http://viaf.org/viaf/93752300/"><rdf:type rdf:resource="http://xmlns.com/foaf/0.1/Document"/><rdf:type rdf:resource="http://www.w3.org/2006/gen/ont#InformationResource"/><void:inDataset xmlns:void="http://rdfs.org/ns/void#" rdf:resource="http://viaf.org/viaf/data"/><foaf:primaryTopic xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/93752300"><dcterms:identifier xmlns:dcterms="http://purl.org/dc/terms/">93752300</dcterms:identifier><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><rdf:type rdf:resource="http://schema.org/Person"/><schema:name xml:lang="fr-FR">Sonia Branca-Rosoff</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#" xml:lang="en">Warning: skos:prefLabels are not ensured against change!</rdfs:comment><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#" xml:lang="fr-FR">Sonia Branca-Rosoff</skos:prefLabel><schema:name xml:lang="cs-CZ">Sonia Branca-Rosoff</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#" xml:lang="en">Warning: skos:prefLabels are not ensured against change!</rdfs:comment><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#" xml:lang="cs-CZ">Sonia Branca-Rosoff</skos:prefLabel><schema:name xml:lang="en-US">Sonia Branca-Rosoff</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#" xml:lang="en">Warning: skos:prefLabels are not ensured against change!</rdfs:comment><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#" xml:lang="en-US">Sonia Branca-Rosoff</skos:prefLabel><schema:name xml:lang="nl-NL">Sonia Branca-Rosoff</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#" xml:lang="en">Warning: skos:prefLabels are not ensured against change!</rdfs:comment><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#" xml:lang="nl-NL">Sonia Branca-Rosoff</skos:prefLabel><schema:name xml:lang="en">Sonia Branca-Rosoff</schema:name><rdfs:comment xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#">BCP47 tag for 'B2Q' is undetermined. Name is 'Sonia Branca-Rosoff'.</rdfs:comment><schema:alternateName>Sonia Branca-Rosoff</schema:alternateName><schema:name xml:lang="fr-FR">Sonia Branca-Rosoff</schema:name><schema:name xml:lang="pl-PL">Sonia Branca-Rosoff</schema:name><schema:alternateName>Sonia Branca</schema:alternateName><schema:givenName>Sonia</schema:givenName><schema:familyName>Branca</schema:familyName><schema:alternateName>Branca</schema:alternateName><schema:alternateName>Sonia Creusot</schema:alternateName><schema:givenName>Sonia</schema:givenName><schema:familyName>Creusot</schema:familyName><schema:alternateName>Sonia Rosoff</schema:alternateName><schema:givenName>Sonia</schema:givenName><schema:familyName>Rosoff</schema:familyName><schema:alternateName>Sonia Branca- Rosoff</schema:alternateName><schema:givenName>Sonia Branca-</schema:givenName><schema:familyName>Rosoff</schema:familyName><schema:alternateName>Sonia Branca- Rosoff</schema:alternateName><schema:givenName>Sonia Branca-</schema:givenName><schema:familyName>Rosoff</schema:familyName><schema:alternateName>Rosoff</schema:alternateName><schema:sameAs><rdf:Description rdf:about="http://www.idref.fr/030435358/id"/></schema:sameAs><schema:sameAs><rdf:Description rdf:about="http://data.bnf.fr/ark:/12148/cb121850323#foaf:Person"/></schema:sameAs><schema:sameAs><rdf:Description rdf:about="http://isni.org/isni/0000000066460502"/></schema:sameAs><schema:sameAs><rdf:Description rdf:about="http://id.loc.gov/authorities/names/n81045017"/></schema:sameAs><rdf:value rdf:parseType="Literal"><ns2:source xmlns="http://viaf.org/viaf/terms#" xmlns:foaf="http://xmlns.com/foaf/0.1/" xmlns:owl="http://www.w3.org/2002/07/owl#" xmlns:void="http://rdfs.org/ns/void#" xmlns:ns2="http://viaf.org/viaf/terms#" nsid="0000099925">B2Q|0000099925</ns2:source></rdf:value></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/BNF%7C12185032#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/BNF"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff Sonia</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/NKC%7Cjo2012712459#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/NKC"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Creusot, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff, Sonia</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/LC%7Cn++81045017#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/LC"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff, Sonia Branca-</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/NTA%7C146010671#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/NTA"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/><schema:url>http://opc4.kb.nl/PPN?PPN=146010671</schema:url></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/ISNI%7C0000000066460502#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/ISNI"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Creusot, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff, Sonia Branca-</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/B2Q%7C0000099925#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/B2Q"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff, Sonia Branca-</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/SUDOC%7C030435358#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/SUDOC"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca, Sonia</skos:altLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff, Sonia</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description><rdf:Description rdf:about="http://viaf.org/viaf/sourceID/NUKAT%7Cn+2012152967#skos:Concept"><rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/><skos:inScheme xmlns:skos="http://www.w3.org/2004/02/skos/core#" rdf:resource="http://viaf.org/authorityScheme/NUKAT"/><skos:prefLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Branca-Rosoff, Sonia.</skos:prefLabel><skos:altLabel xmlns:skos="http://www.w3.org/2004/02/skos/core#">Rosoff, Sonia Branca-.</skos:altLabel><foaf:focus xmlns:foaf="http://xmlns.com/foaf/0.1/" rdf:resource="http://viaf.org/viaf/93752300"/></rdf:Description></rdf:RDF>
EOT;

//"<?"" (for syntax highliting)

    function __construct(string $name = null) {
        parent::__construct($name);
    }

    public function setUp() {
        parent::setUp();
        $this->container = [];
        $history = Middleware::history($this->container);
        $mock = new MockHandler([
            new Response(200, [], self::VIAF_RDF_56666014),
            new Response(200, [], self::VIAF_RDF_93752300),
        ]);
        $handler = HandlerStack::create($mock);
        $handler->push($history);

        $this->client = new Client(['handler' => $handler]);

        $mock404 = new MockHandler([
            new Response(404),
        ]);
        $this->client404 = new Client(['handler' => HandlerStack::create($mock404)]);

        $mock401 = new MockHandler([
            new Response(401, [], 'Unauthorized'),
            new Response(500),
        ]);
        $this->client401 = new Client(['handler' => HandlerStack::create($mock401)]);

        $mock500 = new MockHandler([
            new Response(500, [], 'Internal Server Error'),
        ]);
        $this->client500 = new Client(['handler' => HandlerStack::create($mock500)]);

    }

    /**
     * Jsut test the setup
     *
     * @return void
     */
    public function testSetUp() {
        $this->assertTrue(true);
    }

    /**
     * test getName
     */
    public function testGetName() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client]);
        $name = $resolver->getName('56666014');

        $this->assertEquals('Guylaine Brun-Trigaud', $name, "Name must be Guylaine Brun-Trigaud");

        $this->assertCount(1, $this->container);

        $this->assertEquals("http://viaf.org/viaf/56666014/", (string)$this->container[0]['request']->getUri());

    }

    /**
     * test getName
     */
    public function testGetName93752300() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client]);
        $name = $resolver->getName('56666014'); //first to consume responses

        $name = $resolver->getName('93752300');

        $this->assertEquals('Sonia Branca-Rosoff', $name, "Name must be Sonia Branca-Rosoff");

        $this->assertCount(2, $this->container);

        $this->assertEquals("http://viaf.org/viaf/93752300/", (string)$this->container[1]['request']->getUri());

    }

    /**
     * test unknown id
     */
    public function testUnkownName404() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client404]);

        $name = $resolver->getName('12345');

        $this->assertNull($name);
    }

    /**
     * test unknown id
     */
    public function testUnkownName() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client]);

        $name = $resolver->getName('12345');

        $this->assertNull($name);
    }

    /**
     * Test exception 401
     * @expectedException        CorpusParole\Services\ViafResolverException
     * @expectedExceptionMessage Client error: `GET http://viaf.org/viaf/12345/` resulted in a `401 Unauthorized` response:
     * @expectedExceptionCode 401
     */
    public function test401Error() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client401]);

        $name = $resolver->getName('12345');
    }


    /**
     * Test exception 500
     * @expectedException        CorpusParole\Services\ViafResolverException
     * @expectedExceptionMessage Server error: `GET http://viaf.org/viaf/12345/` resulted in a `500 Internal Server Error` response:
     * @expectedExceptionCode 500
     */
    public function test500Error() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client500]);

        $name = $resolver->getName('12345');
    }

    /**
     * Test exception malformed
     * @expectedException        CorpusParole\Services\ViafResolverException
     * @expectedExceptionMessage ViafId not in correct format
     * @expectedExceptionCode 400
     */
    public function testMalformedError() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client]);

        $name = $resolver->getName('abcd');
    }

    /**
     * test getnames
     */
    public function testGetNames() {
        $resolver = $this->app->make('CorpusParole\Services\ViafResolver', [$this->client]);
        $names = $resolver->getNames(['56666014', '93752300']);

        $this->assertCount(2, $names);
        $this->assertArrayHasKey('56666014', $names);
        $this->assertArrayHasKey('93752300', $names);

        $this->assertEquals('Sonia Branca-Rosoff', $names['93752300']);
        $this->assertEquals('Guylaine Brun-Trigaud', $names['56666014']);

    }

}
