<?php

use Illuminate\Foundation\Testing\WithoutMiddleware;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;

use GuzzleHttp\Client;
use GuzzleHttp\Middleware;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Exception\RequestException;

use CorpusParole\Libraries\Handle\HandleClient;

class HandleClientTest extends TestCase
{
    const CERT_PEM = <<<EOT
-----BEGIN PRIVATE KEY-----
MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDxemvFVLJTkXl65rTnzSPLtJaP
NacGD/SYxnhE1l98F6lj/JQaEH094GWGsCWf41XkZOZYMAZEGIDqdmr6vZjhW7zNjGXCA83PMb+r
p/BOHuq+O5OQaHy9kI2kbNgfqbm+IzP9vrxTXmsVh4N/mCtfygq/uw8S0z0T5G1O9VoJ3mfBQ6dK
fjqUlHtcp2JPt8gDJQ0a8npQysFOeuX+cVV/KW+M1o53xzIOlufudD0dHV9AK9o/4NHQpCgG0Izl
6vcpNqRvotzEov03ID586gUw744ViLPqtOvxM54ac6X2ZKjjwhzGPFL6/A/BqPKQg2xExMbRxquN
E50MZFSL7AVNAgMBAAECggEBAIgWXAIbl1wtUe8HcdBo9dFpdqhCrlxmRYyL3V++CJXgp9/nined
71cdzVm4j5up/M/4ZTAx8HYT8SMxlVy12SC4GJEmnHpYtnYcLU7KGE2OcvC+loz9tb8HZivSA4Lk
nf8fy78m7O92+JZrhpoz4c3cCW7ANuB/F57MTyX+c7229zEsiALhURc3Vib6rWUfq3AJBWOEYKoa
SLLQozi8UJoFODXwC9LDMaxgioNSaU+aj0W0haR/31HRvMQt+Zy1aYYAeuWHT940D6Wgp38ZLMqd
uj22EY64WhcVfv8lQbIb6jOoZqMIj9erGfWlbGlVxoMhUTGQi0sKmyjFBvClwgECgYEA/Am+vCOf
WbHKvGd8G8zWk1VRHpJg4huxqn/i18OuYWdojXgfC/HDR4JKIbl/XP+r0HSudTg4vrusZYGrMGZR
rD82tPDTj24cAI+mfsSNInm6VE4v3eMlEkAcPvqo+qgV5FXU1qFMW5c02vTqOyC/Swjr4YRkP/H7
XhI8T+C6zkECgYEA9UYuSH4ZOehYEDtSXYK4R5NjhjM38a/bulJ8rUBZ8vtA1UACAkoGYXbPukzE
GkEnihuFFDtW7KEY35q2ATj5zV1n1Eh3kfJ7TURl0hNva63jf/uNIXbS0JaFBh8/0N0tkJYrwCOH
aKjAcaMs6OEMuqiLeJX/lpGdCLy+bdikjA0CgYEAjb0NPtDptI9/7CRmb1vq8EBytjartXzNlY0G
DUFGuBBng+ehzh3QLE7hUcX4ElzWSwzTb6iGgP0aHpulojpog+2o3mjxESBjLw2aHIB8vn6mgvLV
iHtNRc44FFLy5MPSS+7ZGzrqrjeCygAiGc54xcHmpZTGrA9xxZ2LvYAsjgECgYBHpKtKQ4MM1iA7
BYjmhZNcXOH6RiW2M/8Dqqp+VMixX9CtOZgcoRDCEVlV+qxhWty4y5cQtNCMJ/ZDAkbdjo3w/bKh
YFWEzQZVKIBxX3nxQwar0IeF2dZ0VTMf+arDUIfJrwOyiO45lOk5iEdK8smsECVFINNaYMg00CTz
ow9VVQKBgFDCVC97xsu5lOl6nd9mSu2GL+6hzpWPIZDD2SwMs6Vkkj33UhqyqziCOFKmRTF5oON+
ew2EGVPukWZsYXNH0KLwdJ7q0QKEONGPMxVNURkHR9cJ8ScaZkiZ6Zh0MT3S3kTak+A5pzo3eS4H
LhgZ72DZ0ABBEGoMfaVh8fJHZKcm
-----END PRIVATE KEY-----
-----BEGIN CERTIFICATE-----
MIIDVTCCAj2gAwIBAgIJAMa/cIGB8QEVMA0GCSqGSIb3DQEBCwUAMCUxIzAhBgNV
BAMUGjMwMDoxMTI4MC4xMDAvQ09SUFVTX0FETUlOMB4XDTE2MDQxODEyMDg1M1oX
DTE2MDUxODEyMDg1M1owJTEjMCEGA1UEAxQaMzAwOjExMjgwLjEwMC9DT1JQVVNf
QURNSU4wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDxemvFVLJTkXl6
5rTnzSPLtJaPNacGD/SYxnhE1l98F6lj/JQaEH094GWGsCWf41XkZOZYMAZEGIDq
dmr6vZjhW7zNjGXCA83PMb+rp/BOHuq+O5OQaHy9kI2kbNgfqbm+IzP9vrxTXmsV
h4N/mCtfygq/uw8S0z0T5G1O9VoJ3mfBQ6dKfjqUlHtcp2JPt8gDJQ0a8npQysFO
euX+cVV/KW+M1o53xzIOlufudD0dHV9AK9o/4NHQpCgG0Izl6vcpNqRvotzEov03
ID586gUw744ViLPqtOvxM54ac6X2ZKjjwhzGPFL6/A/BqPKQg2xExMbRxquNE50M
ZFSL7AVNAgMBAAGjgYcwgYQwHQYDVR0OBBYEFLycwnSTnQGcssXGrj8TGn2rSAgD
MFUGA1UdIwROMEyAFLycwnSTnQGcssXGrj8TGn2rSAgDoSmkJzAlMSMwIQYDVQQD
FBozMDA6MTEyODAuMTAwL0NPUlBVU19BRE1JToIJAMa/cIGB8QEVMAwGA1UdEwQF
MAMBAf8wDQYJKoZIhvcNAQELBQADggEBAGcQGNpACyRBwBQ/iXy7RxLSGdhvbGmh
bikNjBYJVAnYSZjIY2keF1BnhJvaqWHiixoeXC0J9gmN+caZ/WcCyW7xie7fBbn2
FYYJDUj1ik4hD7Cptvd+YwNu2ZjGOKoOqVghGaZSL5cWcfYNYfwv3gQVj05CG6oq
pHHeD5QJw/o/2LYJuFRymPA2AG96mjUSH2gCsqnSxPudfJYVVzjlkWbehIt8ss8l
SdSJqhGvvC5G6x9WhVC3qw56n65+g1FIbhFCXddG/1WGmSfmE8hwlvaOoYgynQGD
BNpIg6V6Mo5EZWnBOuyK26xy4BgXiQCdJPejLGmmzSAUXNAQx47R7VI=
-----END CERTIFICATE-----
EOT;

    const DSA_PEM = <<<EOT
-----BEGIN PRIVATE KEY-----
MIICXAIBADCCAjUGByqGSM44BAEwggIoAoIBAQCPeTXZuarpv6vtiHrPSVG28y7FnjuvNxjo6sSW
Hz79NgbnQ1GpxBgzObgJ58KuHFObp0dbhdARrbi0eYd1SYRpXKwOjxSzNggooi/6JxEKPWKpk0U0
CaD+aWxGWPhL3SCBnDcJoBBXsZWtzQAjPbpUhLYpH51kjviDRIZ3l5zsBLQ0pqwudemYXeI9sCkv
wRGMn/qdgYHnM423krcw17njSVkvaAmYchU5Feo9a4tGU8YzRY+AOzKkwuDycpAlbk4/ijsIOKHE
UOThjBopo33fXqFD3ktm/wSQPtXPFiPhWNSHxgjpfyEc2B3KI8tuOAdl+CLjQr5ITAV2OTlgHNZn
Ah0AuvaWpoV499/e5/pnyXfHhe8ysjO65YDAvNVpXQKCAQAWplxYIEhQcE51AqOXVwQNNNo6NHjB
VNTkpcAtJC7gT5bmHkvQkEq9rI837rHgnzGC0jyQQ8tkL4gAQWDt+coJsyB2p5wypifyRz6Rh5ui
xOdEvSCBVEy1W4AsNo0fqD7UielOD6BojjJCilx4xHjGjQUntxyaOrsLC+EsRGiWOefTznTbEBpl
qiuH9kxoJts+xy9LVZmDS7TtsC98kOmkltOlXVNb6/xF1PYZ9j897buHOSXC8iTgdzEpbaiH7B5H
SPh++1/et1SEMWsiMt7lU92vAhErDR8C2jCXMiT+J67ai51LKSLZuovjntnhA6Y8UoELxoi34u1D
FuHvF9veBB4CHHTW1E//VJ4Ms6LhUNp9nY8IgcxN6upRxLvE+uE=
-----END PRIVATE KEY-----
EOT;

    const RSA_PEM = <<<EOT
-----BEGIN PRIVATE KEY-----
MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDxemvFVLJTkXl65rTnzSPLtJaP
NacGD/SYxnhE1l98F6lj/JQaEH094GWGsCWf41XkZOZYMAZEGIDqdmr6vZjhW7zNjGXCA83PMb+r
p/BOHuq+O5OQaHy9kI2kbNgfqbm+IzP9vrxTXmsVh4N/mCtfygq/uw8S0z0T5G1O9VoJ3mfBQ6dK
fjqUlHtcp2JPt8gDJQ0a8npQysFOeuX+cVV/KW+M1o53xzIOlufudD0dHV9AK9o/4NHQpCgG0Izl
6vcpNqRvotzEov03ID586gUw744ViLPqtOvxM54ac6X2ZKjjwhzGPFL6/A/BqPKQg2xExMbRxquN
E50MZFSL7AVNAgMBAAECggEBAIgWXAIbl1wtUe8HcdBo9dFpdqhCrlxmRYyL3V++CJXgp9/nined
71cdzVm4j5up/M/4ZTAx8HYT8SMxlVy12SC4GJEmnHpYtnYcLU7KGE2OcvC+loz9tb8HZivSA4Lk
nf8fy78m7O92+JZrhpoz4c3cCW7ANuB/F57MTyX+c7229zEsiALhURc3Vib6rWUfq3AJBWOEYKoa
SLLQozi8UJoFODXwC9LDMaxgioNSaU+aj0W0haR/31HRvMQt+Zy1aYYAeuWHT940D6Wgp38ZLMqd
uj22EY64WhcVfv8lQbIb6jOoZqMIj9erGfWlbGlVxoMhUTGQi0sKmyjFBvClwgECgYEA/Am+vCOf
WbHKvGd8G8zWk1VRHpJg4huxqn/i18OuYWdojXgfC/HDR4JKIbl/XP+r0HSudTg4vrusZYGrMGZR
rD82tPDTj24cAI+mfsSNInm6VE4v3eMlEkAcPvqo+qgV5FXU1qFMW5c02vTqOyC/Swjr4YRkP/H7
XhI8T+C6zkECgYEA9UYuSH4ZOehYEDtSXYK4R5NjhjM38a/bulJ8rUBZ8vtA1UACAkoGYXbPukzE
GkEnihuFFDtW7KEY35q2ATj5zV1n1Eh3kfJ7TURl0hNva63jf/uNIXbS0JaFBh8/0N0tkJYrwCOH
aKjAcaMs6OEMuqiLeJX/lpGdCLy+bdikjA0CgYEAjb0NPtDptI9/7CRmb1vq8EBytjartXzNlY0G
DUFGuBBng+ehzh3QLE7hUcX4ElzWSwzTb6iGgP0aHpulojpog+2o3mjxESBjLw2aHIB8vn6mgvLV
iHtNRc44FFLy5MPSS+7ZGzrqrjeCygAiGc54xcHmpZTGrA9xxZ2LvYAsjgECgYBHpKtKQ4MM1iA7
BYjmhZNcXOH6RiW2M/8Dqqp+VMixX9CtOZgcoRDCEVlV+qxhWty4y5cQtNCMJ/ZDAkbdjo3w/bKh
YFWEzQZVKIBxX3nxQwar0IeF2dZ0VTMf+arDUIfJrwOyiO45lOk5iEdK8smsECVFINNaYMg00CTz
ow9VVQKBgFDCVC97xsu5lOl6nd9mSu2GL+6hzpWPIZDD2SwMs6Vkkj33UhqyqziCOFKmRTF5oON+
ew2EGVPukWZsYXNH0KLwdJ7q0QKEONGPMxVNURkHR9cJ8ScaZkiZ6Zh0MT3S3kTak+A5pzo3eS4H
LhgZ72DZ0ABBEGoMfaVh8fJHZKcm
-----END PRIVATE KEY-----

EOT;

    public function setUp() {
        parent::setUp();

        // $this->markTestSkipped('Not ready yet');

        $this->certTmpPath = tempnam("/tmp", "CERT_TEST");
        file_put_contents($this->certTmpPath, HandleClientTest::CERT_PEM);

    }

    public function tearDown() {
        unlink($this->certTmpPath);

        parent::tearDown();
    }

    private function initClient($respArray) {

        $this->history = [];

        $mock = new MockHandler($respArray);
        $handler = HandlerStack::create($mock);

        $history = Middleware::history($this->history);

        $handler->push($history);

        $this->httpClient = new Client(['handler' => $handler]);

    }


    /**
     * Test Pagination with DSA key.
     *
     * @return void
     */
    public function testPaginateDSA()
    {
        $responses = [
            new Response(201,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8',
                           'Vary' => "Accept-Encoding"
                         ],
                         "{\"sessionId\":\"fo79koatgsdyfjkx4c6p3l0c\",\"nonce\":\"10hYzYGE5vf0a1F9eWfk6g==\"}"
            ),
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"sessionId\":\"fo79koatgsdyfjkx4c6p3l0c\",\"nonce\":\"10hYzYGE5vf0a1F9eWfk6g==\",\"authenticated\":true,\"id\":\"300:11280.101/CORPUS_ADMIN_DSA\"}"
            ),
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"responseCode\":1,\"prefix\":\"11280.101\",\"totalCount\":\"2\",\"page\":0,\"pageSize\":15,\"handles\":[\"11280.101/CORPUS_ADMIN\",\"11280.101/CORPUS_ADMIN_DSA\"]}"
            )

        ];
        $this->initClient($responses);

        $handleClient = new HandleClient(HandleClientTest::DSA_PEM, null, "300:11281.100/CORPUS_ADMIN_DSA", "172.16.1.6", 8000, $this->httpClient);

        $pagination = $handleClient->paginateAll('11280.101');

        $this->assertNotNull($pagination);

        $this->assertInstanceOf("Illuminate\Pagination\LengthAwarePaginator", $pagination, "Must be a LengthAwarePaginator");

        $this->assertEquals(2, $pagination->count(), "Must have only one handle");
        $this->assertEquals(2, $pagination->total(), "Total is one");

        $handles = iterator_to_array($pagination);
        $this->assertEquals(['11280.101/CORPUS_ADMIN','11280.101/CORPUS_ADMIN_DSA'], $handles, "Same handles");

        $this->assertCount(3, $this->history, "Must have 3 transactions");

    }


    /**
     * Test Pagination with RSA key.
     *
     * @return void
     */
    public function testPaginateRSA()
    {
        $responses = [
            new Response(201,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8',
                           'Vary' => "Accept-Encoding"
                         ],
                         "{\"sessionId\":\"fo79koatgsdyfjkx4c6p3l0c\",\"nonce\":\"10hYzYGE5vf0a1F9eWfk6g==\"}"
            ),
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"sessionId\":\"fo79koatgsdyfjkx4c6p3l0c\",\"nonce\":\"10hYzYGE5vf0a1F9eWfk6g==\",\"authenticated\":true,\"id\":\"300:11280.101/CORPUS_ADMIN\"}"
            ),
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"responseCode\":1,\"prefix\":\"11280.101\",\"totalCount\":\"2\",\"page\":0,\"pageSize\":15,\"handles\":[\"11280.101/CORPUS_ADMIN\",\"11280.101/CORPUS_ADMIN_DSA\"]}"
            )

        ];
        $this->initClient($responses);

        $handleClient = new HandleClient(HandleClientTest::RSA_PEM, null, "300:11280.101/CORPUS_ADMIN", "172.16.1.6", 8000, $this->httpClient);

        $pagination = $handleClient->paginateAll('11280.101');

        $this->assertNotNull($pagination);

        $this->assertInstanceOf("Illuminate\Pagination\LengthAwarePaginator", $pagination, "Must be a LengthAwarePaginator");

        $this->assertEquals(2, $pagination->count(), "Must have only 2 handle");
        $this->assertEquals(2, $pagination->total(), "Total is 2");

        $handles = iterator_to_array($pagination);
        $this->assertEquals(['11280.101/CORPUS_ADMIN', '11280.101/CORPUS_ADMIN_DSA'], $handles, "Same handles");

        $this->assertCount(3, $this->history, "Must have 3 transactions");

    }

    /**
     * Test Pagination with Client Certificate.
     *
     * @return void
     */
    public function testPaginateCert()
    {

        $responses = [
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"responseCode\":1,\"prefix\":\"11280.101\",\"totalCount\":\"2\",\"page\":0,\"pageSize\":15,\"handles\":[\"11280.101/CORPUS_ADMIN\",\"11280.101/CORPUS_ADMIN_DSA\"]}"
            )
        ];


        $this->initClient($responses);

        $handleClient = new HandleClient($this->certTmpPath, null, "300:11280.101/CORPUS_ADMIN", "172.16.1.6", 8000, $this->httpClient);

        $pagination = $handleClient->paginateAll('11280.101');

        $this->assertNotNull($pagination);

        $this->assertInstanceOf("Illuminate\Pagination\LengthAwarePaginator", $pagination, "Must be a LengthAwarePaginator");

        $this->assertEquals(2, $pagination->count(), "Must have only 2 handle");
        $this->assertEquals(2, $pagination->total(), "Total is 2");
        $this->assertFalse($pagination->hasPages(), "Do not have a page");

        $handles = iterator_to_array($pagination);
        $this->assertEquals(['11280.101/CORPUS_ADMIN', '11280.101/CORPUS_ADMIN_DSA'], $handles, "Same handles");

        $this->assertCount(1, $this->history, "Must have 1 transactions");

    }


    /**
     * Test Pagination with Client Certificate.
     *
     * @return void
     */
    public function testPaginateFirstPageCert()
    {

        $responses = [
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"responseCode\":1,\"prefix\":\"11280.101\",\"totalCount\":\"12\",\"page\":0,\"pageSize\":5,\"handles\":[\"11280.101/TEST1\",\"11280.101/TEST2\",\"11280.101/TEST3\",\"11280.101/TEST4\",\"11280.101/TEST5\"]}"
            )
        ];


        $this->initClient($responses);

        $handleClient = new HandleClient($this->certTmpPath, null, "300:11280.101/CORPUS_ADMIN", "172.16.1.6", 8000, $this->httpClient);

        $pagination = $handleClient->paginateAll('11280.101', 5);

        $this->assertNotNull($pagination);

        $this->assertInstanceOf("Illuminate\Pagination\LengthAwarePaginator", $pagination, "Must be a LengthAwarePaginator");

        $this->assertEquals(5, $pagination->count(), "Must have only 5 handle");
        $this->assertEquals(5, $pagination->perPage(), "Per page is 5");
        $this->assertEquals(12, $pagination->total(), "Total is 12");
        $this->assertEquals(1, $pagination->currentPage(), "Current page is 1");
        $this->assertEquals(3, $pagination->lastPage(), "Last page is 3");
        $this->assertTrue($pagination->hasMorePages(), "Must have more page");
        $this->assertTrue($pagination->hasPages(), "Have at least page");
        $this->assertEquals(1, $pagination->firstItem(), "First item is 1");
        $this->assertEquals(5, $pagination->lastItem(), "First item is 4");

        $handles = iterator_to_array($pagination);
        $this->assertEquals(['11280.101/TEST1', '11280.101/TEST2', '11280.101/TEST3', '11280.101/TEST4', '11280.101/TEST5'], $handles, "Same handles");


    }

    /**
     * Test Pagination with Client Certificate.
     *
     * @return void
     */
    public function testPaginateNextPageCert()
    {

        $responses = [
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"responseCode\":1,\"prefix\":\"11280.101\",\"totalCount\":\"12\",\"page\":1,\"pageSize\":5,\"handles\":[\"11280.101/TEST6\",\"11280.101/TEST7\",\"11280.101/TEST8\",\"11280.101/TEST9\",\"11280.101/TEST10\"]}"
            )
        ];


        $this->initClient($responses);

        $handleClient = new HandleClient($this->certTmpPath, null, "300:11280.101/CORPUS_ADMIN", "172.16.1.6", 8000, $this->httpClient);

        $pagination = $handleClient->paginateAll('11280.101', 5, 'page', 2);

        $this->assertNotNull($pagination);

        $this->assertInstanceOf("Illuminate\Pagination\LengthAwarePaginator", $pagination, "Must be a LengthAwarePaginator");

        $this->assertEquals(5, $pagination->count(), "Must have only 5 handle");
        $this->assertEquals(5, $pagination->perPage(), "Per page is 5");
        $this->assertEquals(12, $pagination->total(), "Total is 12");
        $this->assertEquals(2, $pagination->currentPage(), "Current page is 2");
        $this->assertEquals(3, $pagination->lastPage(), "Last page is 3");
        $this->assertTrue($pagination->hasPages(), "Have at least page");
        $this->assertTrue($pagination->hasMorePages(), "Must have more page");
        $this->assertEquals(6, $pagination->firstItem(), "First item is 1");
        $this->assertEquals(10, $pagination->lastItem(), "First item is 4");

        $handles = iterator_to_array($pagination);
        $this->assertEquals(['11280.101/TEST6', '11280.101/TEST7', '11280.101/TEST8', '11280.101/TEST9', '11280.101/TEST10'], $handles, "Same handles");


    }

    /**
     * Test Pagination with Client Certificate.
     *
     * @return void
     */
    public function testPaginateLastPageCert()
    {

        $responses = [
            new Response(200,
                         [ 'Location' => '/api/sessions/this',
                           'Content-Type' => 'application/json;charset=UTF-8'
                         ],
                         "{\"responseCode\":1,\"prefix\":\"11280.101\",\"totalCount\":\"12\",\"page\":2,\"pageSize\":5,\"handles\":[\"11280.101/TEST11\",\"11280.101/TEST12\"]}"
            )
        ];


        $this->initClient($responses);

        $handleClient = new HandleClient($this->certTmpPath, null, "300:11280.101/CORPUS_ADMIN", "172.16.1.6", 8000, $this->httpClient);

        $pagination = $handleClient->paginateAll('11280.101', 5, 'page', 3);

        $this->assertNotNull($pagination);

        $this->assertInstanceOf("Illuminate\Pagination\LengthAwarePaginator", $pagination, "Must be a LengthAwarePaginator");

        $this->assertEquals(2, $pagination->count(), "Must have only 2 handle");
        $this->assertEquals(5, $pagination->perPage(), "Per page is 5");
        $this->assertEquals(12, $pagination->total(), "Total is 12");
        $this->assertEquals(3, $pagination->currentPage(), "Current page is 3");
        $this->assertEquals(3, $pagination->lastPage(), "Last page is 3");
        $this->assertFalse($pagination->hasMorePages(), "Must have more page");
        $this->assertTrue($pagination->hasPages(), "Have at least page");
        $this->assertEquals(11, $pagination->firstItem(), "First item is 1");
        $this->assertEquals(12, $pagination->lastItem(), "First item is 4");

        $handles = iterator_to_array($pagination);
        $this->assertEquals(['11280.101/TEST11', '11280.101/TEST12'], $handles, "Same handles");


    }


}
