<?php

namespace CorpusParole\Console\Commands;

use Illuminate\Console\Command;

use GuzzleHttp\Client;
use CorpusParole\Libraries\Handle\HandleClient;
use CorpusParole\Libraries\CocoonUtils;
use CorpusParole\Repositories\DocumentRepository;

class ManageHandles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'corpus-parole:manageHandles
                                {--r|replace : Replace value for existing handles}
                                {--k|key= : Private key}
                                {--p|password=: key password}
                                {--P|ask-password : ask for private key password}
                                {--a|admin-id= : the admin index + handle - default to config value corpusparole.handle_admin_id, env var HANDLE_ADMIN_ID}
                                {--H|host= : the handle host format <ip[:port]> - default to corpusparole.handle_host:corpusparole.handle_port (env HANDLE_HOST:HANDLE_PORT)}
                                {--x|prefix= : handle prefix (default to corpusparole.handle_prefix, env HANDLE_PREFIX)}
                                {--s|step-size=100 : number of documents to retrieve from repository at a time before indexing}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Synchronize the handle registry with the content of the rdf repository.';


    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(DocumentRepository $documentRepository, Client $httpClient)
    {
        $this->documentRepository = $documentRepository;
        $this->httpClient = $httpClient;
        parent::__construct();
    }

    private function registerHandle($doc) {
        $baseUrl = str_replace('corpus-back', 'corpus-app', config('app.url'));
        $this->handleClient->createHandleUrlRecord($doc->getId(), "$baseUrl#/doc/".urlencode($doc->getId()), $doc->getTitle());
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $stepSize = $this->option('step-size');

        $passwordKey = $this->option('password');
        if($this->option('ask-password')) {
            $passwordKey = $this->secret('Private key password?');
        }

        $privateKey = $this->option('key');
        if(empty($privateKey)) {
            $privateKey = config('corpusparole.handle_cert_or_pkey');
        }
        if(empty($privateKey)) {
            throw new Exception("No private key found");
        }

        $adminId = $this->option('admin-id');
        if(empty($adminId)) {
            $adminId = config('corpusparole.handle_admin_id');
        }
        if(empty($adminId)) {
            throw new Exception("No admin id found");
        }

        $handleHost = $this->option('host');
        if(!empty($handleHost)) {
            list($handleHost, $handlePort) = array_pad(explode($handleHost, ':'), 2, 8000);
        }
        if(empty($handleHost)) {
            $handleHost = config('corpusparole.handle_host');
        }
        if(empty($handlePort)) {
            $handlePort = config('corpusparole.handle_port', 8000);
        }
        if(empty($handleHost)) {
            throw new Exception("No handle host found");
        }

        $this->handlePrefix = $this->option('prefix');
        if(empty($this->handlePrefix)) {
            $this->handlePrefix = config('corpusparole.handle_prefix');
        }
        if(empty($this->handlePrefix)) {
            throw new Exception("No prefix found");
        }

        // create handle client
        $this->handleClient = new HandleClient($privateKey, $passwordKey, $adminId, $handleHost, $handlePort, $this->httpClient);

        $this->info("Adding documents handles...");

        $total = $this->documentRepository->getCount();

        $docs = $this->documentRepository->paginate(null, $stepSize, config('corpusparole.pagination_page_param'), 1, '_graph');

        $progressBar = $this->output->createProgressBar($docs->total());
        $progressBar->setFormat(' %current%/%max% [%bar%] %percent:3s%% - %message%');

        while(!is_null($docs)) {
            foreach ($docs as $doc){
                $this->registerHandle($doc);
                $progressBar->setMessage($doc->getId());
                $progressBar->advance();
            }
            $docs = ($docs->hasMorePages()? $this->documentRepository->paginate(null, $stepSize, config('corpusparole.pagination_page_param'), $docs->currentPage()+1, '_graph'):null);
        }

        $progressBar->finish();
        $this->info("\nAdding handles completed\n");

        $this->info("Removing extra handles...\n");

        $handles = $this->handleClient->paginateAll($this->handlePrefix, $stepSize, config('corpusparole.pagination_page_param'));

        $progressBar = $this->output->createProgressBar($handles->total());
        $progressBar->setFormat(' %current%/%max% [%bar%] %percent:3s%% - %message%');

        while(!is_null($handles)) {
            foreach($handles as $handle) {
                if(strpos($handle, config('corpusparole.corpus_id_prefix')) === 0 && is_null($this->documentRepository->get($handle))) {
                    $this->handleClient->deleteHandle($handle);
                }
                $progressBar->setMessage($handle);
                $progressBar->advance();
            }
            $handles = $handles->hasMorePages()?$this->handleClient->paginateAll($this->handlePrefix, $stepSize, config('corpusparole.pagination_page_param'), $handles->currentPage()+1):null;
        }

        $this->info("\nRemoving extra handles completed");

    }
}
