<?php
namespace CorpusParole\Models;

use Config;
use CorpusParole\Libraries\Utils;
use CorpusParole\Libraries\CocoonUtils;
use CorpusParole\Libraries\RdfModel\RdfModelResource;
use JsonSerializable;
use Log;
use EasyRdf\Literal;
use EasyRdf\Resource;
use EasyRdf\Graph;


/**
 */
class DocumentBase extends RdfModelResource implements JsonSerializable {

    public function __construct($uri, $graph = null) {
        //print($graph->dump('html'));
        parent::__construct($uri, $graph);
    }

    private $id = null;

    // memoization
    private $providedCHO = null;
    private $title = false;
    private $lang = null;
    private $langResolved = null;
    private $issued = false;
    private $modified = false;
    private $created = false;


    public function getProvidedCHO() {
        if(is_null($this->providedCHO)) {
            $this->providedCHO = $this->get("<http://www.europeana.eu/schemas/edm/aggregatedCHO>");
        }
        return $this->providedCHO;
    }

    protected function clearMemoizationCache() {
        $this->providedCHO = null;
        $this->title = false;
        $this->lang = null;
        $this->langResolved = null;
        $this->issued = false;
        $this->modified = false;
        $this->created = false;
    }

    public function getId() {
        if(is_null($this->id)) {
            $ids = $this->getProvidedCHO()->all('<http://purl.org/dc/elements/1.1/identifier>');
            foreach ($ids as $id) {
                if($id instanceof Literal && strpos($id->getValue(), config('corpusparole.corpus_id_scheme')) === 0) {
                    $this->id = $id->getValue();
                }
            }
            if(is_null($this->id)) {
                $this->id = CocoonUtils::getIdFromCorpusUri($this->uri);
            }
        }
        return $this->id;
    }

    public function getLanguage() {
        if(is_null($this->lang)) {
            try {
                $langs = $this->getProvidedCHO()->all('<http://purl.org/dc/elements/1.1/language>');
                if(count($langs) > 0) {
                    $this->lang = $langs[0];
                }
            } catch(\Exception $e) {
                $this->lang = null;
            }
        }
        return $this->lang;
    }

    public function getLanguageValue() {
        $lang = $this->getLanguage();
        if($lang instanceof Resource) {
            return $lang->getUri();
        } else if($lang instanceof Literal) {
            return $lang->getValue();
        }
        return null;
    }

    public function getLanguageResolved() {
        return $this->langResolved;
    }

    public function setLanguageResolved($languageResolved) {
        $this->langResolved = $languageResolved;
    }


    public function getTitle() {
        if($this->title === false) {
            try {
                $this->title = $this->getProvidedCHO()->getLiteral('<http://purl.org/dc/elements/1.1/title>');
            } catch(\Exception $e) {
                $this->title = null;
            }
        }
        return $this->title;
    }

    public function setTitle($value, $lang="fr") {
        $oldTitle = $this->getTitle();
        if($oldTitle && $oldTitle->getValue() != $value && $oldTitle->getLang() != $lang) {
            $literalTitle = new Literal($value, $lang, null);
            $this->setSimpleProperty($this->getProvidedCHO(), 'http://purl.org/dc/elements/1.1/title', $oldTitle, $literalTitle);
            //clear cache
            $this->title = false;
        }
    }



    public function getTitleValue() {
        $title = $this->getTitle();
        return is_null($title)?null:$title->getValue();
    }

    public function getIssued() {
        if($this->issued === false) {
            try {
                $this->issued = $this->getProvidedCHO()->getLiteral("<http://purl.org/dc/terms/issued>");
            } catch(\Exception $e) {
                $this->issued = null;
            }
        }
        return $this->issued;
    }

    public function getIssuedValue() {
        $issued = $this->getIssued();
        return is_null($issued)?null:$issued->getValue();
    }

    public function getCreated() {
        if($this->created === false) {
            try {
                $this->created = $this->getProvidedCHO()->getLiteral("<http://purl.org/dc/terms/created>");
            } catch(\Exception $e) {
                $this->created = null;
            }
        }
        return $this->created;
    }

    public function getModified() {
        if($this->modified === false) {
            try {
                $this->modified = $this->getProvidedCHO()->getLiteral("<http://purl.org/dc/terms/modified>");
                if(is_null($this->modified)) {
                    $this->modified = $this->getIssued();
                }
            } catch(\Exception $e) {
                $this->modified = null;
            }
        }
        return $this->modified;
    }

    public function setModified($value = null) {
        if(is_null($value)) {
            $value = gmdate(\DateTime::ATOM);
        } elseif ($value instanceof \DateTime) {
            $value = $value->format(\DateTime::ATOM);
        }
        $value = preg_replace('/[\+\-]00(\:?)00$/', 'Z', $value);

        $modified = $this->getModified();
        if($value && (!$modified || $modified->getValue() !== $value) ) {

            $newModified = new Literal($value, null, "http://purl.org/dc/terms/W3CDTF");
            $this->setSimpleProperty($this->getProvidedCHO(), 'http://purl.org/dc/terms/modified', $modified, $newModified);

            $this->modified = null;
        }
    }


    public function getModifiedValue() {
        $modified = $this->getModified();
        return is_null($modified)?null:$modified->getValue();
    }


    public function jsonSerialize() {
        if(!$this->graph) {
            return [
                'id' => $this->getId(),
            ];
        } else {
            $res = [
                'id' => $this->getId(),
                'uri' => $this->getUri(),
                'title' => $this->getTitleValue(),
                'language' => $this->getLanguageValue(),
                'modified' => $this->getModifiedValue(),
                'issued' => $this->getIssuedValue()            ];

            if($this->language_resolved) {
                $res['language_resolved'] = $this->getLanguageResolved();
            }

            return $res;
        }
    }
}
