<?php

use Illuminate\Support\Facades\Facade;
use Illuminate\Support\Facades\Config;

use CorpusParole\Repositories\DocumentRepository;
use CorpusParole\Repositories\RdfDocumentRepository;
use CorpusParole\Models\Document;

use SebastianBergmann\Diff\Differ;

use GuzzleHttp\Client;

class DocumentRepositoryIntegrationTest extends TestCase {

    const REPO_CREATION_TTL = <<<EOT
    @prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#>.
    @prefix rep: <http://www.openrdf.org/config/repository#>.
    @prefix sr: <http://www.openrdf.org/config/repository/sail#>.
    @prefix sail: <http://www.openrdf.org/config/sail#>.
    @prefix ms: <http://www.openrdf.org/config/sail/memory#>.

    [] a rep:Repository ;
        rep:repositoryID "%1\$s" ;
        rdfs:label "%1\$s test repository" ;
        rep:repositoryImpl [
            rep:repositoryType "openrdf:SailRepository" ;
            sr:sailImpl [
                sail:sailType "openrdf:MemoryStore" ;
                ms:persist false ;
                ms:syncDelay 0
            ]
        ].
EOT;

    const TEST_DOC = <<<EOT
    <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608> <http://xmlns.com/foaf/0.1/primaryTopic> <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Document> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/description> "Extrait des enqu\u00EAtes dialectologiques en vue de constituer un atlas linguistique de l'Alsace."@fr .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/format> "audio/x-wav"^^<http://purl.org/dc/terms/IMT> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> <http://purl.org/dc/dcmitype/Sound> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> <http://lexvo.org/id/iso639-3/gsw> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "lexicography"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "phonetics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "anthropological_linguistics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/language> <http://lexvo.org/id/iso639-3/gsw> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> <http://viaf.org/viaf/9122216> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> _:genid2d5831d84541df446694586ac2006f96042dgenid1 .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> _:genid2d5831d84541df446694586ac2006f96042dgenid2 .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/publisher> "Atlas linguistiques, cultures et parlers r\u00E9gionaux de France" .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/rights> "Copyright (c) D\u00E9partement de dialectologie alsacienne et mosellane de l'Universit\u00E9 de Strasbourg" .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/title> "Atlas Linguistique et ethnographique de l'Alsace - Enquetes sur la conscience linguistique - ALA_608"@fr .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/accessRights> "Freely available for non-commercial use" .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/license> <http://creativecommons.org/licenses/by-nc-nd/2.5/> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ALA_CL> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/extent> "PT02H04M16S" .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/created> "start=1980; end=1989; name=ann\u00E9es 80"^^<http://purl.org/dc/terms/Period> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/spatial> "France, Alsace"@fr .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/spatial> "FR"^^<http://purl.org/dc/terms/ISO3166> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/ala/ALA_608.mp3> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/ala/masters/ALA_608.wav> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/available> "2004-07-03"^^<http://purl.org/dc/terms/W3CDTF> .
    <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/issued> "2014-11-04T16:25:54+01:00"^^<http://purl.org/dc/terms/W3CDTF> .
    <http://lexvo.org/id/iso639-3/gsw> <http://www.w3.org/2000/01/rdf-schema#label> "Alsacien"@fr .
    <http://viaf.org/viaf/9122216> <http://www.w3.org/2000/01/rdf-schema#label> "Huck, Dominique" .
    <http://viaf.org/viaf/9122216> <http://xmlns.com/foaf/0.1/status> "depositor"^^<http://www.language-archives.org/OLAC/1.1/role> .
    _:genid2d5831d84541df446694586ac2006f96042dgenid1 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Agent> .
    _:genid2d5831d84541df446694586ac2006f96042dgenid1 <http://xmlns.com/foaf/0.1/name> "Bothorel-Witz, Arlette" .
    _:genid2d5831d84541df446694586ac2006f96042dgenid1 <http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .
    _:genid2d5831d84541df446694586ac2006f96042dgenid2 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Agent> .
    _:genid2d5831d84541df446694586ac2006f96042dgenid2 <http://xmlns.com/foaf/0.1/name> "Huck, Dominique" .
    _:genid2d5831d84541df446694586ac2006f96042dgenid2 <http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .
EOT;

    function __construct(string $name = null) {
        parent::__construct($name);
        $this->graph = new EasyRdf\Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", DocumentRepositoryIntegrationTest::TEST_DOC);
    }

    public function setUp() {
        parent::setUp();

        $this->httpClient = new Client(['base_uri' => config('corpusparole.sesame_base_url')]);
        $this->sesameRepository = config('corpusparole.sesame_repository');
        $this->cocoonDocIdBaseUri = config('corpusparole.cocoon_doc_id_base_uri');

        $this->documentRepository = $this->app->make('CorpusParole\Repositories\DocumentRepository');
        $uniqueid = uniqid('corpusparole', true);
        $repoCreateStmt = sprintf(DocumentRepositoryIntegrationTest::REPO_CREATION_TTL, $this->sesameRepository);
        $this->httpClient->delete("repositories/$this->sesameRepository", ['http_errors' => false]);
        $this->httpClient->post('repositories/SYSTEM/statements', [
            'headers' => ['Content-type' => 'application/x-turtle;charset=UTF-8'],
            'query' => ['context' => "_:$uniqueid"],
            'body' => $repoCreateStmt,
        ]);
        $this->httpClient->put("repositories/$this->sesameRepository/statements", [
            'headers' => ['Content-type' => 'text/turtle;charset=UTF-8'],
            'body' => DocumentRepositoryIntegrationTest::TEST_DOC,
            'query' => ['context' => '<http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608>'],
        ]);
    }

    public function tearDown() {
        $this->httpClient->delete("repositories/$this->sesameRepository");
        parent::tearDown();
    }

    public function testAll() {
        $expectedId = $this->cocoonDocIdBaseUri.'crdo-ALA_608';
        $docList = $this->documentRepository->all();
        $this->assertCount(1, $docList, "Should have one element");

        $resDoc = $docList[0];

        $this->assertInstanceOf(Document::class, $resDoc, "Res doc must be a Document");
        $this->assertEquals('crdo-ALA_608', $resDoc->getId(), "id must be crdo...");
        $this->assertEquals($expectedId, $resDoc->getUri(), 'url must be ...');
        $this->assertNull($resDoc->getGraph(), 'Graph must be null');

    }

    public function testGet() {

        $expectedId = $this->cocoonDocIdBaseUri.'crdo-ALA_608';
        $returnedGraph = new EasyRdf\Graph($expectedId, DocumentRepositoryIntegrationTest::TEST_DOC);

        $res = $this->documentRepository->get('crdo-ALA_608');

        $this->assertInstanceOf(Document::class, $res, "Result must be of type Document");
        $this->assertEquals('crdo-ALA_608', $res->getId(), 'id should be crdo-ALA_608' );
        $this->assertNotNull($res->getGraph(), "Graph shoul not be null");
        $this->assertTrue(EasyRdf\Isomorphic::isomorphic($res->getGraph(),$returnedGraph));
    }

    public function testSave() {

        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);
        $doc->updateDiscourseTypes(['oratory','drama','narrative']);

        $res = $this->documentRepository->save($doc);

        $this->assertTrue($res, 'Has started a transaction');

        $res = $this->documentRepository->get('crdo-ALA_608');

        $discoursesTypes = $res->getDiscourseTypes();

        $this->assertCount(3, $discoursesTypes, "types array must be of size 1");

        $this->assertContainsOnlyInstancesOf("EasyRdf\Literal", $discoursesTypes, "Result contains only literals");

        $newDiscoursesTypes = [];
        foreach($discoursesTypes as $dt) {
            array_push($newDiscoursesTypes, $dt->getValue());
            $this->assertContains($dt->getValue(),['oratory','drama','narrative'],'Value in [oratory,drama,narrative]');
            $this->assertEquals("http://www.language-archives.org/OLAC/1.1/discourse-type", $dt->getDatatypeUri(), "discourse type url");
        }
        sort($newDiscoursesTypes);
        $this->assertEquals(['drama', 'narrative', 'oratory'], $newDiscoursesTypes, "array type must the same");
    }

    public function testCount() {
        $res = $this->documentRepository->getCount();

        $this->assertNotNull($res, "Res should not be null");
        $this->assertSame(1, $res, "should heve only one document");
    }


}
