<?php

use CorpusParole\Models\Document;

/**
 *
 */
class DocumentTest extends TestCase {

    const TEST_DOC = <<<EOT
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://www.w3.org/2000/01/rdf-schema#Resource> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Document> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/description> "Extrait des enqu\u00EAtes dialectologiques en vue de constituer un atlas linguistique de l'Alsace."@fr .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/format> "audio/x-wav"^^<http://purl.org/dc/terms/IMT> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> <http://purl.org/dc/dcmitype/Sound> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "primary_text"^^<http://www.language-archives.org/OLAC/1.1/linguistic-type> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/type> "dialogue"^^<http://www.language-archives.org/OLAC/1.1/discourse-type> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> <http://lexvo.org/id/iso639-3/gsw> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "lexicography"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "phonetics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/subject> "anthropological_linguistics"^^<http://www.language-archives.org/OLAC/1.1/linguistic-field> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/language> <http://lexvo.org/id/iso639-3/gsw> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> <http://viaf.org/viaf/9122216> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> _:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/contributor> _:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/publisher> "Atlas linguistiques, cultures et parlers r\u00E9gionaux de France" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/rights> "Copyright (c) D\u00E9partement de dialectologie alsacienne et mosellane de l'Universit\u00E9 de Strasbourg" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/elements/1.1/title> "Atlas Linguistique et ethnographique de l'Alsace - Enquetes sur la conscience linguistique - ALA_608"@fr .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/accessRights> "Freely available for non-commercial use" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/license> <http://creativecommons.org/licenses/by-nc-nd/2.5/> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_ALA_CL> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isPartOf> <http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-COLLECTION_LANGUESDEFRANCE> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/extent> "PT02H04M16S" .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/created> "start=1980; end=1989; name=ann\u00E9es 80"^^<http://purl.org/dc/terms/Period> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/spatial> "France, Alsace"@fr .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/spatial> "FR"^^<http://purl.org/dc/terms/ISO3166> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/ala/ALA_608.mp3> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/isFormatOf> <http://cocoon.huma-num.fr/data/ala/masters/ALA_608.wav> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/available> "2004-07-03"^^<http://purl.org/dc/terms/W3CDTF> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://purl.org/dc/terms/issued> "2014-11-04T16:25:54+01:00"^^<http://purl.org/dc/terms/W3CDTF> .
<http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> <http://www.openrdf.org/schema/sesame#directType> <http://xmlns.com/foaf/0.1/Document> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://www.w3.org/2000/01/rdf-schema#Resource> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Agent> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://xmlns.com/foaf/0.1/name> "Bothorel-Witz, Arlette" .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid1 <http://www.openrdf.org/schema/sesame#directType> <http://xmlns.com/foaf/0.1/Agent> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://www.w3.org/2000/01/rdf-schema#Resource> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://xmlns.com/foaf/0.1/Agent> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://xmlns.com/foaf/0.1/name> "Huck, Dominique" .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://xmlns.com/foaf/0.1/status> "interviewer"^^<http://www.language-archives.org/OLAC/1.1/role> .
_:genid2d6c3879d9d2484bd98bb0bcb9dd0364fc2dgenid2 <http://www.openrdf.org/schema/sesame#directType> <http://xmlns.com/foaf/0.1/Agent> .
<http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608> <http://xmlns.com/foaf/0.1/primaryTopic> <http://cocoon.huma-num.fr/data/ala/ALA_608_22km.wav> .
EOT;

    private $testGraph;

    public function setUp() {

        parent::setup();
        $this->graph = new \EasyRdf_Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", DocumentTest::TEST_DOC);

    }

    public function testConstructor() {
        $this->assertNotNull($this->graph, 'Graph shoud not be null');

        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $this->assertEquals('crdo-ALA_608',$doc->getId(),'Must have the correct id');
    }

    public function testTitle() {
        $this->assertNotNull($this->graph, 'Graph shoud not be null');

        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $this->assertEquals("Atlas Linguistique et ethnographique de l'Alsace - Enquetes sur la conscience linguistique - ALA_608",$doc->getTitle(),'Must have correct title');
        $this->assertInstanceOf(\EasyRdf_Literal::class, $doc->getTitle(), "Title must be a literal");
        $this->assertEquals('fr', $doc->getTitle()->getLang(), "Language title must be fr");
    }

    public function testPublisher() {

        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $this->assertCount(1, $doc->getPublishers(), 'Publisher is an array of size 1');
        $this->assertEquals(new \EasyRdf_Literal("Atlas linguistiques, cultures et parlers régionaux de France",null,null), $doc->getPublishers()[0], "Must contains correct publisher");

    }

    public function testMediaArray() {
        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $this->assertCount(3, $doc->getMediaArray(), "Media array must be of size 3");
        foreach($doc->getMediaArray() as $media) {
            $this->assertCount(2, $media, "media is a 2 element array");
            $this->assertArrayHasKey('format', $media, "media has 'format key'");
            $this->assertArrayHasKey('url', $media, "media has url");
        }
    }

    public function testGetTypes() {
        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $this->assertCount(3, $doc->getTypes(), "types array must be of size 3");

        foreach($doc->getTypes() as $type) {
            $this->assertThat(
                $type,
                $this->logicalXor(
                    $this->isInstanceOf(\EasyRdf_Literal::class),
                    $this->isInstanceOf(\EasyRdf_Resource::class)
                )
            );
        }
    }

    public function testGetOtherTypes() {
        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $this->assertCount(2, $doc->getOtherTypes(), "types array must be of size 3");

        foreach($doc->getTypes() as $type) {
            $this->assertThat(
                $type,
                $this->logicalXor(
                    $this->isInstanceOf(\EasyRdf_Literal::class),
                    $this->isInstanceOf(\EasyRdf_Resource::class)
                )
            );
        }
    }

    public function testGetDiscourseTypes() {
        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $this->assertCount(1, $doc->getDiscourseTypes(), "types array must be of size 1");

        $this->assertContainsOnlyInstancesOf("EasyRdf_Literal", $doc->getDiscourseTypes(), "Result contains only literals");
        $type = $doc->getDiscourseTypes()[0];
        $this->assertEquals("dialogue", $type, "discourse type is dialogue");
        $this->assertEquals("http://www.language-archives.org/OLAC/1.1/discourse-type", $type->getDatatypeUri(), "discourse type url");
    }

    public function testCloneDocument() {
        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);

        $doc2 = clone $doc;

        $this->assertNotSame($doc, $doc2, "documents must not be the same");
        $this->assertNotSame($doc->getGraph(), $doc2->getGraph(), "documents must not be the same");

        $this->assertTrue(\EasyRdf_Isomorphic::isomorphic($doc->getGraph(), $doc2->getGraph()),"graph must be isomorphic");
    }

    public function testIsIsomorphic() {
        $doc1 = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);
        $doc2 = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", new \EasyRdf_Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", DocumentTest::TEST_DOC));

        $this->assertTrue($doc1->isIsomorphic($doc2),"document must be isomorphic");

        $doc2->addLiteral('dc11:type', new \EasyRdf_Literal("oratory", null, Config::get('OLAC_DISCOURSE_TYPE')['uri']));

        $this->assertFalse($doc1->isIsomorphic($doc2),"document must not be isomorphic");
    }

    public function testUpdateDiscourseTypes() {

        $newDiscourseTypes = ['oratory','dialogue','narrative'];

        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);
        $this->assertCount(1, $doc->getDiscourseTypes(), "types array must be of size 1");

        $doc->updateDiscourseTypes($newDiscourseTypes);

        $this->assertCount(3, $doc->getDiscourseTypes(), "types array must be of size 3");

        $discourseTypes = $doc->getDiscourseTypes();
        foreach($newDiscourseTypes as $dt) {
            $this->assertContains($dt, $discourseTypes, "all discourse types must be in result list");
        }

    }

    public function testUpdateDiscourseTypesIsomorphic() {

        $newDiscourseTypes = ['oratory','dialogue','narrative'];

        $doc = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", $this->graph);
        $doc->updateDiscourseTypes($newDiscourseTypes);

        $doc2 = new Document("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", new \EasyRdf_Graph("http://purl.org/poi/crdo.vjf.cnrs.fr/crdo-ALA_608", DocumentTest::TEST_DOC));

        $this->assertFalse($doc->isIsomorphic($doc2),"document must not be isomorphic after adding discourse type");
    }
}
