import Ember from 'ember';
import AmCharts from 'ammaps';
import ENV from 'app-client/config/environment';
import * as commonConstants from 'corpus-common-addon/utils/constants';
import * as commonUtils from 'corpus-common-addon/utils/utils';

export default Ember.Component.extend({

  constants: Ember.inject.service(),
  colors: Ember.inject.service(),
  filter: Ember.inject.service(),

  map: null,

  continentsMapVar: AmCharts.maps.continentsLow,
  continentsAreas: [],
  countriesMapVar: AmCharts.maps.worldLow,
  countriesAreas: [],
  franceMapVar: AmCharts.maps.france2016Low,
  franceAreas: [],

  dataProvider: {},

  colorScale: Ember.computed('maxCount', 'minCount', function () {
    let maxCount = this.get('maxCount');
    let minCount = this.get('minCount');
    return this.get('colors').getScaleLinear(minCount, maxCount);
  }),
  counts: Ember.computed.mapBy('geostats', 'count'),
  maxCount: Ember.computed.max('counts'),
  minCount: Ember.computed.min('counts'),

  area: null,
  originalZoomLevel: null,
  originalZoomLongitude: null,
  originalZoomLatitude: null,

  color: 'rgba(113,132,141,.2)',

  locationObserver: Ember.observer('filter.location', function () {
    if (this.get('filter').get('location') === null) {
      this.sendAction('setLocation', this.get('constants').GEONAMES['world']);
      this.get('map').selectObject();
    } else {
      this.sendAction('setLocation', commonUtils.getGeonamesCode(this.get('filter').get('location')));
    }
  }),

  geostats: [],
  observerGeostats: Ember.observer('geostats', function () {
    let map = this.get('map');
    if(!map) {
      return;
    }
    let selectedObject = map.selectedObject;
    if (selectedObject.id) {
      if (selectedObject.parentObject.mapVar === this.get('countriesMapVar')) {
        if (selectedObject.id === this.get('constants').GEONAMES['france']) {
          this.setProperties({
            area: 'france',
            originalZoomLevel: null,
            originalZoomLongitude: null,
            originalZoomLatitude: null,
          });
          this.setFranceAreas();
          this.set('dataProvider', {
            'mapVar': this.get('franceMapVar'),
            'getAreasFromMap': true,
            'areas': this.get('franceAreas')
          });
          map['zoomControl'] = {
            'zoomControlEnabled': false
          };
          map['dragMap'] = false;
          map['dataProvider'] = this.get('dataProvider');
          map.validateData();
        }
      } else if (selectedObject.parentObject.mapVar === this.get('continentsMapVar')) {
        this.setProperties({
          area: 'countries',
          originalZoomLevel: map.zoomLevel(),
          originalZoomLongitude: map.zoomLongitude(),
          originalZoomLatitude: map.zoomLatitude()
        });

        this.setCountriesAreas();
        this.set('dataProvider', {
          'mapVar': this.get('countriesMapVar'),
          'getAreasFromMap': true,
          'areas': this.get('countriesAreas'),
          'zoomLevel': map.zoomLevel(),
          'zoomLatitude': map.zoomLatitude(),
          'zoomLongitude': map.zoomLongitude()
        });
        map['dataProvider'] = this.get('dataProvider');
        map['zoomControl'] = {
          'zoomControlEnabled': true
        };
        map['dragMap'] = true;
        map.validateData();
      }
    } else {
      this.setProperties({
        area: 'continents',
        originalZoomLevel: null,
        originalZoomLongitude: null,
        originalZoomLatitude: null,
      });
      this.setContinentsAreas();
      this.set('dataProvider', {
        'mapVar': this.get('continentsMapVar'),
        'areas': this.get('continentsAreas')
      });
      map['zoomControl'] = {
        'zoomControlEnabled': false
      };
      map['dragMap'] = false;
      this.createAmMap();
    }
  }),

  init: function () {
    this._super(...arguments);
    if (ENV.environment === 'development') {
      this.set('color', '#f3f3f3');
    }
    this.get('service'); // gotcha cf: https://github.com/emberjs/ember.js/issues/10821
  },

  setFranceAreas: function () {
    var self = this;
    var france2016LowAreas = [];
    this.get('franceMapVar')['svg']['g']['path'].forEach(function (area) {
      var object = {
        'id': area.id,
        'passZoomValuesToTarget': false,
        'autoZoom': false
      };
      var departments = self.get('geostats').find(geostat => geostat.id === area.id);
      if (typeof departments === 'undefined') {
        object['mouseEnabled'] = false;
        object['color'] = self.get('color');
      } else {
        object['value'] = departments.get('count');
        object['title'] = area.title + "&nbsp;(" + departments.get('count') + ")";
      }
      france2016LowAreas.push(object);
    });
    this.set('franceAreas', france2016LowAreas);
  },

  setCountriesAreas: function () {
    var self = this;
    var worldLowAreas = [];
    this.get('countriesMapVar')['svg']['g']['path'].forEach(function (area) {
      var countries = self.get('geostats').find(geostat => geostat.id === area.id);
      if (typeof countries === 'undefined') {
        worldLowAreas.push({
          'id': area.id,
          'mouseEnabled': false,
          'color': self.get('color')
        });
      } else {
        if (typeof worldLowAreas.find(country => country.id === area.id) === 'undefined') {
          var object = {
            'id': area.id,
            'selectable': true,
            'value': countries.get('count'),
            'title': area.title + "&nbsp;(" + countries.get('count') + ")"
          };
          if (area.id === 'FR') {
            object['autoZoom'] = true;
          }
          worldLowAreas.push(object);
        }
      }
    });
    this.set('countriesAreas', worldLowAreas);
  },

  setContinentsAreas: function () {
    var self = this;
    var continentsLowAreas = [];
    this.get('continentsMapVar')['svg']['g']['path'].forEach(function (area) {
      var object = {
        'id': area.id,
        'passZoomValuesToTarget': true,
        'selectable': true,
        'mouseEnabled': true,
        'autoZoom': true
      };
      var continent = self.get('geostats').find(geostat => geostat.id === area.id);
      if (typeof continent === 'undefined') {
        object['mouseEnabled'] = false;
        object['color'] = self.get('color');
      } else {
        object['value'] = continent.get('count');
        object['title'] = area.title + "&nbsp;(" + continent.get('count') + ")";
      }
      continentsLowAreas.push(object);
    });
    this.set('continentsAreas', continentsLowAreas);
  },

  didInsertElement: function () {
    this.$('#mapdiv').height(Ember.$('.corpus-app-container').height());
    this.createAmMap();
  },

  createAmMap: function () {

    this.setContinentsAreas();
    this.set('dataProvider', {
      'mapVar': this.get('continentsMapVar'),
      'areas': this.get('continentsAreas')
    });
    this.set('map', AmCharts.makeChart('mapdiv', {
      'type': 'map',
      'accessible': true,
      'fontFamily': 'sans-serif',
      'fontSize': '12px',
      'dragMap': false,
      'zoomOnDoubleClick': false,
      'language': 'fr',
      'areasSettings': {
        'autoZoom': false,
        'selectable': true,
        'color': this.get('colors').LINEAR_COLOR_START,
        'colorSolid': this.get('colors').LINEAR_COLOR_END,
        'colorOutline': '#253946',
        'selectedColor': '#0085cb',
        'rollOverColor': '#0085cb',
        'rollOverOutlineColor': '#ffffff'
      },
      'zoomControl': {
        'zoomControlEnabled': false,
      },
      'dataProvider': this.get('dataProvider'),
      listeners: [{
        'event': 'clickMapObject',
        'method': Ember.run.bind(this, 'clickMapObject')
      }, {
        'event': 'homeButtonClicked',
        'method': Ember.run.bind(this, 'homeButtonClicked')
      }]
    }));
  },

  clickMapObject: function (event) {
    const changed = this.get('filter').setFilter('location', commonConstants.GEONAMES_BASE_URL + event.mapObject.id);
    if(!changed) {
      //force filter change
      this.notifyPropertyChange('filter.location');
    }
  },

  homeButtonClicked: function () {
    const map = this.get('map');
    const { area, originalZoomLevel, originalZoomLongitude, originalZoomLatitude } =
      this.getProperties('area', 'originalZoomLevel', 'originalZoomLongitude', 'originalZoomLatitude');

    if(area === 'countries' && (
      (originalZoomLevel && originalZoomLevel !== map.zoomLevel()) ||
      (originalZoomLatitude && originalZoomLatitude !== map.zoomLatitude()) ||
      (originalZoomLongitude && originalZoomLongitude !== map.zoomLongitude())
    )) {
      // in mode countries, the map has been moved or zoomed, just reset the view
      return;
    }
    this.get('filter').removeFilter('location', null);
    this.sendAction('setLocation', this.get('constants').GEONAMES['world']);
  }

});
