import Ember from 'ember';
import d3 from 'd3';
import env from 'app-client/config/environment';



export default Ember.Component.extend({

    classNames: ['discourses-component'],

    constants: Ember.inject.service(),
    filter: Ember.inject.service(),

    didRender: function() {
        var self = this;
        var baseURL = env.rootURL.replace(/\/$/,"")+'/api/v1';

        d3.json(baseURL + "/discourses", function(discourses) {
            var array = Object.keys(discourses).map(function (key) { return discourses[key].count; });
            var oldMin = Math.min(...array), oldMax = Math.max(...array);
            var sum = array.reduce(function(a, b) { return a + b; });
            var average = sum / array.length;
            var newMin = Math.floor((average - oldMin)), newMax = Math.floor((oldMax - average));

            var width = self.$().parent().width();
            var height = self.$().parent().height() - self.$().siblings().height();

            var bubble = d3.layout.pack()
                .sort(function comparator(a, b) { return a.value + b.value; })
                .size([width, height])
                .value(function(d){
                    return Math.floor((((d.value - oldMin) * (newMax - newMin)) / (oldMax - oldMin)) + newMin);
                })
                .padding(25);

            var element = d3.select('#' + self.get('elementId'));

            var bubbles = bubble.nodes(self.createNodes(discourses));

            var caption = element.append("div")
                .attr("class", "caption");
            var caption_nodes = caption.selectAll(".node")
                .data(bubbles);

            caption_nodes.enter().append("div")
                .attr("class", function(d) { return d.children ? " category": " item"; });

            var item = caption.selectAll(".item")
                .style("width", function(d) { return (d.r * 2) + 'px'; })
                .style("height", function(d) { return (d.r * 2) + 'px'; })
                .style("left", function(d) { return d.x - d.r + 'px'; })
                .style("top", function(d) { return d.y - d.r + 'px'; })
                .style("font-size", function(d) {
                    return Math.floor((((d.value - oldMin) * (13 - 11)) / (oldMax - oldMin)) + 11) + 'px';
                })
                .on('click', function(d) {
                    self.get('filter').set('discourse', d.id);
                });
            item.append("span")
                .html(function(d) { return d.name + ' <span class="count">(' + d.value + ')</span>'; })
                .style("display", function(d) { return d.children ? 'none' : 'inline-block'; })
                .style("width", function() { return Ember.$(this).parent().width() > Ember.$(this).width() ? Ember.$(this).parent().width() + 'px' : ''; })
                .style("margin-left", function() { return ( Ember.$(this).width() > Ember.$(this).parent().width() ? - ( Ember.$(this).width() / 2 ) + ( Ember.$(this).parent().width() / 2 ) : 0 ) + 'px'; })
                .style("margin-top", function() { return Ember.$(this).parent().height() / 2 - Ember.$(this).height() / 2 + 'px'; });

            var svg = element.append("svg")
                .style("width", width + "px")
                .style("height", height + "px")
                .attr("class", "bubble");

            var svg_nodes = svg.selectAll(".node")
                .data(bubbles);
            svg_nodes.enter().append("g")
                .attr("class", function(d) { return "node" + (!d.children ? " leaf" : ""); })
                .attr("transform", function(d) { return "translate(" + d.x + "," + d.y + ")"; });

            var leaf = svg.selectAll(".leaf")
                .on('click', function(d) {
                    self.get('filter').set('discourse', d.id);
                });

            leaf.append("circle")
                .attr("r", function(d) { return d.r; })
                .attr("fill", function(d) { return d.fill; })
                .attr("stroke", function(d) { return d.stroke; });
            element.style("height", height + "px");
        });

        this._super(...arguments);
    },

    createNodes: function(json) {
        var self = this;
        var nodes = {};
        var children = {};

        Object.keys(json).forEach(function(key) {
            var discourse = json[key];
            var category_key = self.get('constants').DISCOURSE_CATEGORY_KEYS[key];
            var category_value = self.get('constants').DISCOURSE_CATEGORY_VALUES[category_key];

            children[category_key] = children[category_key] || {};
            children[category_key]['name'] = category_key;
            children[category_key]['children'] = children[category_key]['children'] || [];
            children[category_key]['children'].push({
                'id': key,
                'name': discourse.label,
                'value': discourse.count,
                'fill': category_value.fill,
                'stroke': category_value.stroke
            });
        });

        nodes.children = [];
        Object.keys(children).forEach(function(key) {
            nodes.children.push(children[key]);
        });

        return nodes;
    }

});
