from flask import Flask, request
from flask.ext.restful import fields
from rdflib import Graph, RDF, RDFS, BNode, Literal, URIRef
from uuid import uuid4
from StringIO import StringIO
from app import *
from persistence import *

app.config["PERSISTENCE_METHOD"] = PersistenceToGithub

"""
Namespace: ld.iri-research.org/ontology/categorisation/#
Category URI: ld.iri-research.org/categorisation/#cat_id

Category Class:
label is the rdf label of the category
description is the description of the category
other_properties is a dictionnary containing every other RDF(S) property
"""


class Category(object):
    def __init__(self, label=None, description=None,
                 other_properties=None, graph=None):
        if not(graph):
            cat_id = uuid4().hex
            self.cat_graph = Graph()
            self.this_category = URIRef(app.config["ONTOLOGY_NAMESPACE"]+cat_id)
            self.cat_graph.add((self.this_category, RDF.ID, Literal(cat_id)))

            # parsing label and description

            if label:
                self.cat_graph.add((self.this_category,
                                   RDFS.label,
                                   Literal(label)))
            if description:
                self.cat_graph.add((self.this_category,
                                   RDF.Description,
                                   Literal(description)))

            # parsing other properties
            if other_properties:
                for (predicate, obj) in other_properties:
                    self.cat_graph.add((self.this_category,
                                       app.config["PROPERTY_LIST"]
                                                 [predicate]
                                                 ["rdflib_class"],
                                       Literal(obj)))

        else:
            self.cat_graph = graph
            self.this_category = self.cat_graph \
                                     .subjects(predicate=RDF.ID) \
                                     .next()  # Warning: not foolproof

    @property
    def label(self):
        return_value = self.cat_graph.value(self.this_category, RDFS.label)
        if return_value is None:
            return None
        else:
            return return_value.toPython()

    @property
    def description(self):
        return_value = self.cat_graph.value(self.this_category, RDF.Description)
        if return_value is None:
            return None
        else:
            return return_value.toPython()

    @property
    def cat_id(self):
        return self.cat_graph.value(self.this_category, RDF.ID).toPython()

    @property
    def properties(self):
        # list of properties from the PROPERTY_LIST defined in settings.py
        property_list = []
        for key in app.config["PROPERTY_LIST"]:
            for obj in self.cat_graph \
                           .objects(subject=self.this_category,
                                    predicate=app.config["PROPERTY_LIST"]
                                                        [key]
                                                        ["rdflib_class"]):
                property_list.append((key, obj.toPython()))
        return property_list

    def edit_category(self, new_label=None, new_description=None, new_other_properties=None):
        # Did the label change?
        if new_label is not None and new_label != self.label:
            # Remove the old triple
            self.cat_graph.remove((self.this_category,
                                   RDFS.label,
                                   self.cat_graph.label(self.this_category)))
            # Make a new one
            self.cat_graph.add((self.this_category,
                                RDFS.label,
                                Literal(new_label)))
        # Did the description change?
        if new_description is not None and new_description != self.description:
            # Remove the old triple
            self.cat_graph.remove((self.this_category,
                                  RDF.Description,
                                  self.cat_graph.value(self.this_category,
                                                       RDF.Description)))
            # Make a new one
            self.cat_graph.add((self.this_category,
                               RDF.Description,
                               Literal(new_description)))

        # Did the properties change?
        if new_other_properties is not None or self.properties is not None:
            for key in app.config["PROPERTY_LIST"]:
                self.cat_graph.remove((self.this_category,
                                       app.config["PROPERTY_LIST"]
                                                 [key]
                                                 ["rdflib_class"],
                                       None))
            for (predicate,obj) in new_other_properties:
                self.cat_graph.add((self.this_category,
                                    app.config["PROPERTY_LIST"]
                                              [predicate]
                                              ["rdflib_class"],
                                    Literal(obj)))


"""
CategoryManager class

This class deals with creation and loading of Category objects

Persistence method is set in config files
"""


class CategoryManager(object):
    def load_cat(self, cat_id):
        p = app.config["PERSISTENCE_METHOD"]()
        cat_serial = p.load(name=cat_id)
        loaded_cat_graph = Graph()
        loaded_cat_graph.parse(source=StringIO(cat_serial), format='turtle')
        cat = Category(graph=loaded_cat_graph)
        return cat

    def save_cat(self, cat, message=None):
        p = app.config["PERSISTENCE_METHOD"]()
        p.save(content=cat.cat_graph.serialize(format='turtle'),
               name=cat.cat_id, message=message)

    def delete_cat(self, cat_id, message=None):
        p = app.config["PERSISTENCE_METHOD"]()
        p.delete(name=cat_id, message="Deleting category "+cat_id)

    def list_cat(self):
        p = app.config["PERSISTENCE_METHOD"]()
        cat_serial_list = p.list()
        cat_list=[]
        for cat_serial in cat_serial_list:
            loaded_cat_graph = Graph()
            loaded_cat_graph.parse(source=StringIO(cat_serial), format='turtle')
            c = Category(graph=loaded_cat_graph)
            cat_list.append(c)
        return cat_list
