from flask import Flask, request
from flask.ext.restful import fields
from rdflib import Graph, RDF, RDFS, BNode, Literal, URIRef
from uuid import uuid4
from io import StringIO
from slugify import slugify
import persistence
from catedit import app

logger = app.logger

"""
Namespace: ld.iri-research.org/ontology/categorisation/#
Category URI: ld.iri-research.org/ontology/categorisation/#cat_id

Category Class:
label is the rdf label of the category
description is the description of the category
other_properties is a dictionnary containing every other supported property
as defined in the PROPERTY_LIST dict in settings.py
"""


class Category(object):
    def __init__(self, label=None, description=None,
                 other_properties=None, graph=None):
        if not(graph):
            # cat_id = uuid4().hex - Alternate method of generating ids
            cat_id = "category_id_"+slugify(bytes(label))
            self.cat_graph = Graph()
            self.this_category = URIRef(app.config["ONTOLOGY_NAMESPACE"] +
                                        cat_id)
            self.cat_graph.add((self.this_category, RDF.ID, Literal(cat_id)))

            if label:
                self.cat_graph.add((self.this_category,
                                   RDFS.label,
                                   Literal(label)))
            if description:
                self.cat_graph.add((self.this_category,
                                   RDF.Description,
                                   Literal(description)))

            if other_properties:
                for (predicate, obj) in other_properties:
                    self.cat_graph.add((self.this_category,
                                       app.config["PROPERTY_LIST"]
                                                 [predicate]
                                                 ["rdflib_class"],
                                       app.config["PROPERTY_LIST"]
                                                 [predicate]
                                                 ["object_rdflib_class"](obj)))

        else:
            self.cat_graph = graph
            self.this_category = self.cat_graph \
                                     .subjects(predicate=RDF.ID) \
                                     .next()  # Warning: not foolproof

    @property
    def label(self):
        return_value = self.cat_graph.value(self.this_category, RDFS.label)
        if return_value is None:
            return None
        else:
            return return_value.toPython()

    @property
    def description(self):
        return_value = \
            self.cat_graph.value(self.this_category, RDF.Description)
        if return_value is None:
            return None
        else:
            return return_value.toPython()

    @property
    def cat_id(self):
        return self.cat_graph.value(self.this_category, RDF.ID).toPython()

    @property
    def properties(self):
        property_list = []
        for key in app.config["PROPERTY_LIST"]:
            for obj in self.cat_graph \
                           .objects(subject=self.this_category,
                                    predicate=app.config["PROPERTY_LIST"]
                                                        [key]
                                                        ["rdflib_class"]):
                property_list.append((key, obj.toPython()))
        return property_list

    def edit_category(self, new_label=False,
                      new_description=False,
                      new_other_properties=False):
        """
        Checks if there is a new label and if so apply changes
        """
        if new_label is not (False or None) and \
           new_label != self.label:
            self.cat_graph.remove((self.this_category,
                                   RDFS.label,
                                   self.cat_graph.label(self.this_category)))
            self.cat_graph.add((self.this_category,
                                RDFS.label,
                                Literal(new_label)))

        """
        Checks if there is a new description and if so apply changes
        """
        if new_description is not (False or None) and \
           new_description != self.description:
            self.cat_graph.remove((self.this_category,
                                  RDF.Description,
                                  self.cat_graph.value(self.this_category,
                                                       RDF.Description)))
            self.cat_graph.add((self.this_category,
                               RDF.Description,
                               Literal(new_description)))

        """
        Checks if there is a new property list and if so apply changes
        (can be [] so it deletes everything)
        """
        if new_other_properties is not False or \
            (new_other_properties is None and
             self.properties is not None):
            logger.debug("before suppressing properties: ")
            logger.debug(self.properties)
            logger.debug("will replace with: ")
            logger.debug(new_other_properties)
            for key in app.config["PROPERTY_LIST"]:
                self.cat_graph.remove((self.this_category,
                                       app.config["PROPERTY_LIST"]
                                                 [key]
                                                 ["rdflib_class"],
                                       None))
            logger.debug("now properties are: ")
            logger.debug(self.properties)
            logger.debug("making new properties: ")
            logger.debug(new_other_properties)
            for (predicate, obj) in new_other_properties:
                self.cat_graph.add((self.this_category,
                                    app.config["PROPERTY_LIST"]
                                              [predicate]
                                              ["rdflib_class"],
                                    app.config["PROPERTY_LIST"]
                                              [predicate]
                                              ["object_rdflib_class"](obj)))


"""
CategoryManager class

This class deals with creation and loading of Category objects

Persistence method is set in config files
"""


class CategoryManager(object):
    def load_cat(self, cat_id):
        p = getattr(persistence, app.config["PERSISTENCE_METHOD"])()
        cat_serial = p.load(name=cat_id)
        loaded_cat_graph = Graph()
        loaded_cat_graph.parse(source=StringIO(cat_serial), format='turtle')
        cat = Category(graph=loaded_cat_graph)
        return cat

    def save_cat(self, cat, message=None):
        p = getattr(persistence, app.config["PERSISTENCE_METHOD"])()
        p.save(content=cat.cat_graph.serialize(format='turtle'),
               name=cat.cat_id, message=message)

    def delete_cat(self, deleted_cat_id, message=None):
        cat_list = self.list_cat()
        for cat in cat_list:
            if cat.cat_id != app.config["ONTOLOGY_NAMESPACE"]+deleted_cat_id:
                new_property_list_for_cat = []
                for (predicate, obj) in cat.properties:
                    if not (
                        (app.config["PROPERTY_LIST"]
                                   [predicate]
                                   ["object_type"] == "uriref-category") and
                        (obj == (app.config["ONTOLOGY_NAMESPACE"] +
                                 deleted_cat_id))
                    ):
                        new_property_list_for_cat.append((predicate, obj))
                cat.edit_category(
                    new_other_properties=new_property_list_for_cat
                )
                self.save_cat(
                    cat,
                    message=message+", cleaning up other properties"
                )
        p = getattr(persistence, app.config["PERSISTENCE_METHOD"])()
        p.delete(name=deleted_cat_id, message=message)

    def list_cat(self):
        p = getattr(persistence, app.config["PERSISTENCE_METHOD"])()
        cat_serial_list = p.list()
        logger.debug(cat_serial_list)
        cat_list = []
        for cat_serial in cat_serial_list:
            loaded_cat_graph = Graph()
            loaded_cat_graph.parse(
                source=StringIO(cat_serial),
                format='turtle'
            )
            c = Category(graph=loaded_cat_graph)
            cat_list.append(c)
        return cat_list
