"""
models.py:
contains the "principal" objects that will be manipulated by the application:
* categories
* helper classes to manage category life cycle
"""

from rdflib import Graph, RDF, RDFS, Literal, URIRef
# from uuid import uuid4
from io import StringIO
from slugify import slugify
from catedit import app
import catedit.persistence


LOGGER = app.logger


class Category(object):
    """
        Category Class:

        Init:
        * label is the rdf label of the category, unique and non-empty
        * description is the description of the category, unique and non-empty
        * other_properties is a dictionnary containing every other supported
        property as defined in the PROPERTY_LIST dict in settings.py
        * graph is used if we want to create the Category from a turtle
        rdf graph

        Additional info:
        * cat_id is a generated, hidden id that uniquely identify
        a given category
    """
    def __init__(self, label=None, description=None,
                 other_properties=None, graph=None):
        if not graph:
            # cat_id = uuid4().hex - Alternate method of generating ids
            cat_id = "category_id_"+slugify(label)
            self.cat_graph = Graph()
            self.this_category = URIRef(app.config["CATEGORY_NAMESPACE"] +
                                        cat_id)
            self.cat_graph.add((self.this_category, RDF.ID, Literal(cat_id)))

            if label:
                self.cat_graph.add((self.this_category,
                                   RDFS.label,
                                   Literal(label)))
            if description:
                self.cat_graph.add((self.this_category,
                                   RDF.Description,
                                   Literal(description)))

            if other_properties:
                for (predicate, obj) in other_properties:
                    self.cat_graph.add((self.this_category,
                                       app.config["PROPERTY_LIST"]
                                                 [predicate]
                                                 ["rdflib_class"],
                                       app.config["PROPERTY_LIST"]
                                                 [predicate]
                                                 ["object_rdflib_class"](obj)))

        else:
            self.cat_graph = graph
            # Warning: not foolproof
            self.this_category = next(self.cat_graph
                                          .subjects(predicate=RDF.ID))

    @property
    def label(self):
        """
            Returns category label
        """
        return_value = self.cat_graph.value(self.this_category, RDFS.label)
        if return_value is None:
            return None
        else:
            return return_value.toPython()

    @property
    def description(self):
        """
            Returns category description
        """
        return_value = \
            self.cat_graph.value(self.this_category, RDF.Description)
        if return_value is None:
            return None
        else:
            return return_value.toPython()

    @property
    def cat_id(self):
        """
            Returns category id
        """
        return self.cat_graph.value(self.this_category, RDF.ID).toPython()

    @property
    def properties(self):
        """
            Returns category property list
        """
        property_list = []
        for key in app.config["PROPERTY_LIST"]:
            for obj in self.cat_graph \
                           .objects(subject=self.this_category,
                                    predicate=app.config["PROPERTY_LIST"]
                                                        [key]
                                                        ["rdflib_class"]):
                property_list.append((key, obj.toPython()))
        return property_list

    def edit_category(self, new_label=False,
                      new_description=False,
                      new_other_properties=False):
        """
            Edits category
            * new_label is the new label of the category
            * new_description is the new description of the category
            * new_other_property is the new property list of the category

            Every argument is optional and defaults to False for consistency,
            as giving a property list that is None means we want
            to delete all properties
        """

        if (new_label is not False) and \
           (new_label is not None) and \
           (new_label != self.label):
            self.cat_graph.remove((self.this_category,
                                   RDFS.label,
                                   self.cat_graph.label(self.this_category)))
            self.cat_graph.add((self.this_category,
                                RDFS.label,
                                Literal(new_label)))

        if (new_description is not False) and \
           (new_description is not None) and \
           (new_description != self.description):
            self.cat_graph.remove((self.this_category,
                                  RDF.Description,
                                  self.cat_graph.value(self.this_category,
                                                       RDF.Description)))
            self.cat_graph.add((self.this_category,
                               RDF.Description,
                               Literal(new_description)))

        if new_other_properties is not False or \
            (new_other_properties is None and
             self.properties is not None):
            LOGGER.debug("before suppressing properties: ")
            LOGGER.debug(self.properties)
            LOGGER.debug("will replace with: ")
            LOGGER.debug(new_other_properties)
            for key in app.config["PROPERTY_LIST"]:
                self.cat_graph.remove((self.this_category,
                                       app.config["PROPERTY_LIST"]
                                                 [key]
                                                 ["rdflib_class"],
                                       None))
            LOGGER.debug("now properties are: ")
            LOGGER.debug(self.properties)
            LOGGER.debug("making new properties: ")
            LOGGER.debug(new_other_properties)
            for (predicate, obj) in new_other_properties:
                self.cat_graph.add((self.this_category,
                                    app.config["PROPERTY_LIST"]
                                              [predicate]
                                              ["rdflib_class"],
                                    app.config["PROPERTY_LIST"]
                                              [predicate]
                                              ["object_rdflib_class"](obj)))


class CategoryManager(object):
    """
        CategoryManager class
        This class deals with creation and loading of Category objects
        Persistence method is set in config.py and used to save
        and load categories
    """
    def load_cat(self, cat_id):
        """
            Loads a category from its id
        """
        persistence = getattr(catedit.persistence,
                              app.config["PERSISTENCE_METHOD"])()
        cat_serial = persistence.load(name=cat_id)
        loaded_cat_graph = Graph()
        loaded_cat_graph.parse(source=StringIO(cat_serial), format='turtle')
        cat = Category(graph=loaded_cat_graph)
        return cat

    def save_cat(self, cat, message=None):
        """
            Saves a category, message serves as commit message
            if github is used as a persistence method
        """
        persistence = getattr(catedit.persistence,
                              app.config["PERSISTENCE_METHOD"])()
        persistence.save(content=cat.cat_graph.serialize(format='turtle'),
                         name=cat.cat_id, message=message)

    def delete_cat(self, deleted_cat_id, message=None):
        """
            Deletes a category from its id, message serves
            as commit message if github is used as a persistence method
        """
        cat_list = self.list_cat()
        for cat in cat_list:
            if cat.cat_id != app.config["CATEGORY_NAMESPACE"]+deleted_cat_id:
                new_property_list_for_cat = []
                for (predicate, obj) in cat.properties:
                    if not (
                        (app.config["PROPERTY_LIST"]
                                   [predicate]
                                   ["object_type"] == "uriref-category") and
                        (obj == (app.config["CATEGORY_NAMESPACE"] +
                                 deleted_cat_id))
                    ):
                        new_property_list_for_cat.append((predicate, obj))
                cat.edit_category(
                    new_other_properties=new_property_list_for_cat
                )
                self.save_cat(
                    cat,
                    message=message+", cleaning up other properties"
                )
        persistence = getattr(catedit.persistence,
                              app.config["PERSISTENCE_METHOD"])()
        persistence.delete(name=deleted_cat_id, message=message)

    def list_cat(self):
        """
            Lists all categories available
        """
        persistence = getattr(catedit.persistence,
                              app.config["PERSISTENCE_METHOD"])()
        cat_serial_list = persistence.list()
        # LOGGER.debug(cat_serial_list)
        cat_list = []
        for cat_serial in cat_serial_list:
            loaded_cat_graph = Graph()
            loaded_cat_graph.parse(
                source=StringIO(cat_serial),
                format='turtle'
            )
            cat = Category(graph=loaded_cat_graph)
            cat_list.append(cat)
        return cat_list
