"""
home.py:
The views functions that handle authentication and index pages
"""

from catedit import app, github
from views.utils import check_user_status
from requests import get
from requests.auth import HTTPBasicAuth
from flask import render_template, request, redirect, url_for, \
                  session, Blueprint
from flask.ext.github import GitHubError
from flask_wtf import Form
from wtforms import StringField, PasswordField
from wtforms.validators import DataRequired

module = Blueprint('home', __name__)
logger = app.logger


@module.route('/', methods=['GET'])
@module.route('/index', methods=['GET'])
def index():
    """
        View for index page (for now it's only a readme text so no computing
        is required)
    """
    return render_template("home/index.html")


class LoginForm(Form):
    """
        Custom form class for commiting changes
    """
    user_login = StringField(
        "Nom d'utilisateur Github",
        validators=[DataRequired()]
    )
    user_password = PasswordField(
        "Mot de passe Github",
        validators=[DataRequired()]
    )


@module.route('/catedit-login', methods=["GET", "POST"])
def login():
    """
        Function that manages authentication (Github), login

        Note: If Persistence is set to PersistenceToFile (categories stored
        in local files, used for debugging), creates a mock user named
        "FileEditUser"
    """
    if not session.get("user_logged", False):
        session["modified_categories"] = {
            repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                           ["REPOSITORY_LIST"]
        }
        session["deleted_categories"] = {
            repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                           ["REPOSITORY_LIST"]
        }
        if app.config["PERSISTENCE_CONFIG"]["METHOD"] == "PersistenceToGithub":
            login_form = LoginForm(request.form)
            if request.method == "GET":
                # We'll render the login form
                return render_template(
                    "home/login.html",
                    form=login_form,
                )
            elif request.method == "POST":
                if login_form.validate_on_submit():
                    # We'll try to get the auth token for given username
                    try:
                        auth_response = get(
                            "https://api.github.com/"
                            + "authorizations",
                            auth=HTTPBasicAuth(
                                login_form.user_login.data,
                                login_form.user_password.data
                            )
                        )
                        for auth in auth_response.json():
                            if auth["app"]["client_id"] \
                            == app.config["GITHUB_CLIENT_ID"]:
                                session["user_code"] = auth["token"]
                                session["user_logged"] = True
                    except:
                        logger.debug(
                            "Error requesting authorizations for"
                            + " user. Either the user is new to catedit, or "
                            + "entered a wrong username/password"
                        )
                    logger.debug(str(github.get("rate_limit")["resources"]))
                    logger.debug(
                        "user token found by request: "
                        + str(session.get("user_code", None))
                    )
                    if session.get("user_code", None) == None:
                        # We didn't get it, so we direct the user to the login page
                        # with a link to github oauth system
                        return render_template(
                            "home/login.html",
                            form=login_form
                        )
                    else:
                        # we did get it, so we redirect to callback function
                        # to wrap up user auth
                        return redirect(url_for('home.login_callback'))
                else:
                    # form didn't validate, so we send it back to user
                    return render_template(
                        "home/login.html",
                        form = login_form
                    )
        elif app.config["PERSISTENCE_CONFIG"]["METHOD"] == "PersistenceToFile":
            session["user_logged"] = True
            session["user_can_edit"] = {}
            session["user_can_edit"]["local"] = True
            session["user_login"] = "FileEditUser"
            return redirect(url_for('home.index'))
    else:
        return redirect(url_for('home.index'))

@module.route('/catedit-login-confirm', methods=["GET", "POST"])
def login_confirm():
    """
        Function called if the user is new or revoked the auth token
    """
    if not session.get("user_logged", False):
        if request.method == "POST":
            return github.authorize(
                scope="repo",
                redirect_uri=url_for('home.login_callback', _external=True)
            )
    else:
        return redirect(url_for('home.index'))

@module.route('/catedit-callback')
@github.authorized_handler
def login_callback(oauth_code):
    """
        Function that handles callback from Github after succesful login
    """
    session.permanent = False
    if session.get("user_code", None) == None:
    # That means we got here using github callback and not the login form
        session["user_code"] = oauth_code
    logger.debug(session["user_code"])
    session["user_logged"] = True
    session["user_login"] = "auth-error"
    try:
        logger.debug(
            "after login: "
            + str(github.get("rate_limit")["resources"])
        )
        session["user_login"] = github.get("user")["login"]
    except GitHubError as ghe:
        logger.error(
            "GitHubError trying to get the user login"
        )
        logger.error(ghe.response.text)
    try:
        repo_list = []
        repo_list = github.get("user/repos")
        logger.debug(str(github.get("rate_limit")["resources"]))
        for repo in repo_list:
            logger.debug(repo["name"])
        user_repos_name = [repo["name"] for repo in repo_list]
        logger.debug(
            str(user_repos_name) + " "
            + str(app.config["PERSISTENCE_CONFIG"]
                            ["REPOSITORY_LIST"])
        )
        session["user_repositories"] = list(
            set(user_repos_name).intersection(
                app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
            )
        )
        session["user_can_edit"] = {}
        for repo in session["user_repositories"]:
            if repo in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
                session["user_can_edit"][repo] = True
        logger.debug(session["user_can_edit"])
    except GitHubError as ghe:
        logger.error(
            "GitHubError trying to get the list of repository for user "
            + session["user_login"]
        )
        logger.error(ghe.response.text)
    return redirect(url_for('home.index'))

@module.route('/catedit-logout')
def logout():
    """
        Function that manages authentication (Github), logout

        Note: if you want to switch github users, you will have to logout of
        Github, else when logging back in, github will send the app the
        same oauth code
    """
    session["user_logged"] = None
    session["user_code"] = None
    session["user_login"] = None
    session["user_can_edit"] = None
    session["modified_categories"] = {
        repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                       ["REPOSITORY_LIST"]
    }
    session["deleted_categories"] = {
        repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                       ["REPOSITORY_LIST"]
    }
    return redirect(url_for('home.index'))
