"""
utils.py:
Module that groups utility functions that are used by views, partly because
most of them do requests to the Github API and as such must be cached
"""

from catedit import app, github, cache
from catedit.models import Category
from flask import redirect, url_for, session, abort
from flask.ext.github import GitHubError
from datetime import datetime
from rdflib import Graph
from base64 import b64decode
from io import StringIO

logger = app.logger

def check_user_status(repository):
    """
        Function to call at the beginning of every view that would require
        authentication and editing privilege to work properly (basically
        everything save login and index page)
    """
    if not session.get("user_logged", None):
        return redirect(url_for("home.index"))
    if not session.get("user_can_edit", {}).get(repository, False):
        return redirect(url_for("home.index"))

@cache.memoize(timeout=3600)
def get_comments(request_data):
    """
        Function that takes a dict with the following two values:
            * type: either "issues" or "commits"
            * id: the id of the issue of commit to get comments from
        then builds a comment list with each comment being a dict with the
        following format:
        {
            "author": author of the comment
            "body": body of the comment
            "date": date of the comment format dd/mm/yy hh:mm
        }
    """
    github_comments_data = []

    if request_data["type"] == "commits":
        commits_list = get_commits(request_data["repository"])
        if request_data["id"] not in [commit["id"] for commit in commits_list]:
            abort(404)
    elif request_data["type"] == "issues":
        issues_list = get_issues(request_data["repository"])
        if request_data["id"] not in [issue["id"] for issue in issues_list]:
            abort(404)

    try:
        github_comments_data = github.get(
            "repos/"
            + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
            + request_data["repository"] + "/"
            + request_data["type"] + "/"
            + request_data["id"] + "/comments?per_page=100"
        )

    except GitHubError as ghe:
        logger.error(
            "Error trying to get comments with following data: "
            + str(request_data)
        )
        logger.error(ghe.response.text)

    comment_list = []
    for comment in github_comments_data:
        comment_dict = {
            "author": comment["user"]["login"],
            "body": comment["body"],
            "date": convert_github_date(
                comment["created_at"]
            )
        }
        comment_list.append(comment_dict)

    discussion_data = {}
    try:
        discussion_data = github.get(
            "repos/"
            + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
            + request_data["repository"] + "/"
            + request_data["type"] + "/"
            + request_data["id"]
        )
    except GitHubError as ghe:
        logger.error(
            "Error trying to get the or issue of id " + request_data["id"]
        )
        logger.error(
            "endpoint: " + "repos/"
            + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
            + request_data["repository"] + "/"
            + request_data["type"] + "/"
            + request_data["id"]
        )
        logger.error(ghe.response.text)

    thread_author = ""
    thread_opening_date = ""
    thread_title = ""
    thread_opening_post = ""

    if request_data["type"] == "commits":
        thread_author = discussion_data.get("author",{}).get("login", "")
        thread_opening_date = convert_github_date(
            discussion_data.get(
                "commit",
                {}
            ).get(
                "author",
                {}
            ).get("date","")
        )
        thread_title = discussion_data.get("commit",{}).get("message","")
    elif request_data["type"] == "issues":
        thread_author = discussion_data.get("user",{}).get("login", "")
        thread_opening_date = convert_github_date(
            discussion_data.get("created_at", "0001-01-01T00:00:00Z")
        )
        thread_title = discussion_data.get("title", "")
        thread_opening_post = discussion_data.get("body", "")

    logger.debug(str(github.get("rate_limit")["resources"]))
    thread_dict = {
        "author": thread_author,
        "title": thread_title,
        "opening_date": thread_opening_date,
        "comment_list": comment_list,
        "opening_post": thread_opening_post
    }
    return thread_dict


def post_comment(request_data):
    """
        Function that posts a given comment to github.

        * repository is the repository to post in
        * type is either "issues" or "commits"
        * thread_id is the id of the issue or the commit to comment
        * thread_title is the title of the new issue, if we're posting a new
        issue
        * comment_body is the content of the comment
    """
    comment_data = {
        "body": request_data["comment_body"]
    }
    return_id = ""
    if request_data["thread_id"] != "new":
        try:
            github_response = github.post(
                "repos/"
                + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
                + request_data["repository"] + "/"
                + request_data["type"] + "/"
                + request_data["thread_id"]
                + "/comments",
                data=comment_data
            )
            return_id = request_data["thread_id"]
        except GitHubError as ghe:
            logger.error(
                "Error posting comment with following data: "
                + str(request_data)
            )
            logger.error(ghe.response.text)
    else:
        # We're posting a new issue
        comment_data["title"] = request_data["thread_title"]
        try:
            github_response = github.post(
                "repos/"
                + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
                + request_data["repository"] + "/"
                + request_data["type"],
                data=comment_data
            )
            return_id = str(github_response["number"])
        except GitHubError as ghe:
            logger.error(
                "Error posting new issue with following data: "
                + str(request_data)
            )
            logger.error(ghe.response.text)
    cache.clear()
    return return_id

@cache.memoize(timeout=3600)
def get_commits(repository):
    """
        Fuction that get the list of commits for a given repository. Returns a
        list of dict with the format:
        {
            id : commit id
            title : commit message
            author : commit author
            comment_count : commit comments count
        }
    """
    commits_data = []
    try:
        commits_data = github.get(
            "repos/"
            + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
            + repository
            + "/commits?per_page=100"
        )
    except GitHubError as ghe:
        logger.error("Error getting commits for repo " + repository)
        logger.error(ghe.response.text)
    changeset_list = [
        {
            "id": commit["sha"],
            "title": commit["commit"]["message"],
            "date": convert_github_date(
                commit["commit"]["committer"]["date"],
            ),
            "author": commit["commit"]["committer"]["name"],
            "comment_count": commit["commit"]["comment_count"],
        }
        for commit in commits_data
    ]

    logger.debug(str(github.get("rate_limit")["resources"]))
    return changeset_list

@cache.memoize(timeout=3600)
def get_issues(repository):
    """
        Fuction that get the list of issues for a given repository. Returns a
        list of dict with the format:
        {
            id: issue id
            title: issue title
            author: issue author
            opening_date: issue opening date
            last_updated: last update date
            comment_count: comments count
        }
    """
    issues_data = []
    try:
        issues_data = github.get(
        "repos/"
        + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
        + repository
        + "/issues?per_page=100")
    except GitHubError as ghe:
        logger.error("Error getting issues for repo " + repository)
        logger.error(ghe.response.text)

    discussion_list = [
        {
            "id": str(issue["number"]),
            "title": issue["title"],
            "author": issue["user"]["login"],
            "opening_date": convert_github_date(issue["created_at"]),
            "last_updated": convert_github_date(issue["updated_at"]),
            "comment_count": issue["comments"],
        }
        for issue in issues_data
    ]

    logger.debug(str(github.get("rate_limit")["resources"]))
    return discussion_list

@cache.memoize(timeout=3600)
def get_category_list(repository, changeset_id):
    """
        Get the category list as it was following the changeset of
        id changeset_id
    """
    commits_list = get_commits(repository)
    if changeset_id not in [commit["id"] for commit in commits_list]:
        abort(404)

    # First step
    commit_data = {}
    try:
        commit_data = github.get(
            "repos/"
            + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
            + repository + "/commits/"
            + changeset_id
        )
    except GitHubError as ghe:
        logger.error("Error trying to get the commit of id " + changeset_id)
        logger.error(ghe.response.text)

    logger.debug(str(github.get("rate_limit")["resources"]))

    tree_sha = commit_data.get("commit", {}).get("tree", {}).get("sha", "")

    # Second step
    tree_data = {}
    try:
        tree_data = github.get(
            "repos/"
            + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
            + repository + "/git/trees/"
            + commit_data["commit"]["tree"]["sha"]
            + "?recursive=1"
        )
    except GitHubError as ghe:
        logger.error("Error trying to get the tree of sha " + tree_sha)
        logger.error(ghe.response.text)

    logger.debug(str(github.get("rate_limit")["resources"]))
    logger.debug(tree_data)

    # Third step and fourth step
    cat_list = []
    for blob in tree_data.get("tree", []):
        if app.config["PERSISTENCE_CONFIG"]["CATEGORIES_PATH"] in blob["path"]:
            blob_data = {}
            try:
                blob_data = github.get(
                    "repos/"
                    + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"]
                    + "/" + repository + "/git/blobs/"
                    + blob["sha"]
                )
            except GitHubError as ghe:
                logger.error(
                    "Error trying to get the blob of sha " + blob["sha"]
                )
                logger.error(ghe.response.text)

            cat_graph = Graph()
            cat_graph.parse(
                source=StringIO(
                    str(b64decode(blob_data["content"]), "utf-8")
                ),
                format="turtle"
            )
            category = Category(graph=cat_graph)
            cat_list.append(
                {
                    "cat_label": category.label,
                    "cat_description": category.description,
                    "cat_id": category.cat_id,
                    "cat_properties": category.properties,
                    "state": "original"
                }
            )

    logger.debug(str(github.get("rate_limit")["resources"]))
    return cat_list

def convert_github_date(date):
    """
        Function that converts github date format to a
        "dd/mm/yyyy à hh:mm" format
    """
    return datetime.strptime(
        date,
        "%Y-%m-%dT%H:%M:%SZ"
    ).strftime(
        "%d/%m/%Y à %H:%M"
    )
