"""
social.py:
The views functions that handle comment display and posting
"""

from catedit import app
from catedit.utils import make_differences_list, compare_categories
from catedit.views.forms import NewDiscussionForm, CommentForm
from catedit.views.utils import check_user_status_and_repo_access, \
                                get_comments, post_comment, get_commits, \
                                get_issues, get_category_list_for_commit

from flask import render_template, request, redirect, url_for, \
                  abort, Blueprint, session


module = Blueprint('social', __name__)
logger = app.logger


@module.route(
    "/<string:repository>/changesets",
    methods=["GET"],
    defaults={
        "commits_page": 1,
        "commits_per_page": 10,
    }
)
@module.route(
    "/<string:repository>/changesets"
    + "/page<int:commits_page>_perpage<int:commits_per_page>",
    methods=["GET"]
)
def changesets_index(repository, commits_page, commits_per_page):
    """
        View that displays every changeset and links to every thread
    """
    check_user_status_and_repo_access(repository)

    changeset_list, commits_pagination = get_commits(
        repository,
        commits_per_page,
        commits_page
    )

    session.pop("pagination_links", None)

    return render_template(
        "social/changesets_index.html",
        current_repository=repository,
        changeset_list=changeset_list,
        commits_pagination=commits_pagination,
        commits_per_page=commits_per_page,
    )

@module.route(
    "/<string:repository>/discussions",
    methods=["GET"],
    defaults={
        "discussions_page": 1,
        "discussions_per_page": 10,
    }
)
@module.route(
    "/<string:repository>/discussions"
    + "/page<int:discussions_page>_perpage<int:discussions_per_page>",
    methods=["GET"]
)
def discussions_index(repository, discussions_page, discussions_per_page):
    """
        View that displays thread in general discussion and links to
        every thread
    """
    check_user_status_and_repo_access(repository)

    discussions_list, discussions_pagination = get_issues(
        repository,
        discussions_per_page,
        discussions_page
    )
    if discussions_list == [] and discussions_page != 1:
        abort(404)

    return render_template(
        "social/discussions_index.html",
        current_repository=repository,
        discussions_list=discussions_list,
        discussions_pagination=discussions_pagination,
        discussions_per_page=discussions_per_page,
    )


@module.route("/<string:repository>/changesets/<string:changeset_id>",
              methods=["GET", "POST"],
              defaults={"per_page": 10, "page": 1, "details_cat_id": None})
@module.route(
    "/<string:repository>/changesets/<string:changeset_id>"
    + "/page/<int:page>-per_page-<int:per_page>",
    defaults={"details_cat_id": None},
    methods=["GET", "POST"]
)
@module.route("/<string:repository>/changesets/"
              + "<string:changeset_id>/details_<string:details_cat_id>",
              methods=["GET", "POST"],
              defaults={"per_page": 10, "page": 1})
@module.route("/<string:repository>/changesets/"
              + "<string:changeset_id>/page/<int:page>-"
              + "per_page-<int:per_page>-details_<string:details_cat_id>",
              methods=["GET", "POST"])
def changeset(repository, changeset_id, per_page, page, details_cat_id):
    """
        View that displays a snapshot of the repository as it was for a given
        changeset, and the related discussion to this changeset. Allows
        users to comment on it.

        For re-building the old category set we use the tree and blob data:
        1) first we get the commit of id changeset_id
        2) we get the tree associated with this commit
        3) we get all the blobs referenced by this tree
        4) we build categories from the blobs contents
        --> Will probably be incorporated into a function later on

    """
    check_user_status_and_repo_access(repository)

    comment_form = CommentForm()
    comments_list, pagination = get_comments(
        repository=repository,
        thread_type="commits",
        thread_id=changeset_id,
        page=page,
        per_page=per_page
    )
    if comments_list == [] and page != 1:
        abort(404)
    old_category_list = get_category_list_for_commit(
        repository,
        changeset_id,
        get_parent=True
    )
    new_category_list = get_category_list_for_commit(
        repository,
        changeset_id
    )

    category_differences_dict = {
        list_type: [
            (first_category.label, second_category.label, first_category.cat_id)
            for (first_category, second_category) in diff_list
        ] if list_type == "modifications" else [
            category.label
            for category in diff_list
        ]
        for list_type, diff_list in make_differences_list(
            old_category_list,
            new_category_list,
            repository
        ).items()
    }

    changes_details = None
    print(str(details_cat_id))
    if details_cat_id is not None:
        if (
        details_cat_id not in [category.cat_id
        for category in old_category_list]
        or
        details_cat_id not in [category.cat_id
        for category in new_category_list]
        ):
            abort(404)
        else:
            old_category = next(
                category
                for category in old_category_list
                if category.cat_id == details_cat_id
            )
            new_category = next(
                category
                for category in new_category_list
                if category.cat_id == details_cat_id
            )
            changes_details = compare_categories(old_category, new_category, repository)
            changes_details["cat_id"] = details_cat_id

    old_category_list_template = []
    for category in old_category_list:
        if category.label in category_differences_dict["deletions"]:
            cat_state = "deleted"
        elif category.label in category_differences_dict["additions"]:
            cat_state = "created"
        elif category.label in [cat_tuple[0] for cat_tuple in category_differences_dict["modifications"]]:
            cat_state = "modified"
        else:
            cat_state = "untouched"
        old_category_list_template.append({
            "cat_label": category.label,
            "cat_description": category.description,
            "cat_id": category.cat_id,
            "cat_properties": category.properties,
            "state": cat_state
        })

    new_category_list_template = []
    for category in new_category_list:
        if category.label in category_differences_dict["deletions"]:
            cat_state = "deleted"
        elif category.label in category_differences_dict["additions"]:
            cat_state = "created"
        elif category.label in [cat_tuple[0] for cat_tuple in category_differences_dict["modifications"]]:
            cat_state = "modified"
        else:
            cat_state = "untouched"
        new_category_list_template.append({
            "cat_label": category.label,
            "cat_description": category.description,
            "cat_id": category.cat_id,
            "cat_properties": category.properties,
            "state": cat_state
        })
    if request.method == "GET":
        return render_template(
            "social/changeset.html",
            old_cat_list=old_category_list_template,
            new_cat_list=new_category_list_template,
            differences=category_differences_dict,
            changes_details=changes_details,
            comments=comments_list,
            changeset_id=changeset_id,
            comment_form=comment_form,
            current_repository=repository,
            comments_pagination=pagination,
            comments_per_page=per_page
        )
    elif request.method == "POST" and comment_form.validate_on_submit():
        return_id = post_comment(
            repository=repository,
            thread_type="commits",
            thread_id=changeset_id,
            comment_body=comment_form.comment_field.data
        )
        return redirect(url_for(
            "social.changeset",
            repository=repository,
            changeset_id=return_id,
            page=1,
            per_page=per_page,
            changes_details=changes_details
        ))
    else:
        # Form didn't validate
        return render_template(
            "social/changeset.html",
            old_cat_list=old_category_list_template,
            new_cat_list=new_category_list_template,
            comments=comments_list,
            changeset_id=changeset_id,
            change_details=changes_details,
            comment_form=comment_form,
            current_repository=repository,
            comments_pagination=pagination,
            comments_per_page=per_page
        )


@module.route(
    "/<string:repository>/discussions/<string:discussion_id>",
    methods=["GET", "POST"],
    defaults={"per_page": 10, "page": 1}
)
@module.route(
    "/<string:repository>/discussions/<string:discussion_id>"
    + "/page/<int:page>-per_page-<int:per_page>",
    methods=["GET", "POST"]
)
def discussion(repository, discussion_id, per_page, page):
    """
        View that displays the discussion of a given id and allows users
        to post comments on it.
    """
    check_user_status_and_repo_access(repository)
    comment_form = None
    pagination = None
    if discussion_id == "new":
        comment_form = NewDiscussionForm()
        comments_list = []
    else:
        comment_form = CommentForm()
        comments_list, pagination = get_comments(
            repository=repository,
            thread_type="issues",
            thread_id=discussion_id,
            page=page,
            per_page=per_page
        )
        if comments_list == [] and page != 1:
            abort(404)

    if request.method == "GET":
        return render_template(
            "social/discussion.html",
            comments=comments_list,
            comment_form=comment_form,
            current_repository=repository,
            discussion_id=discussion_id,
            comments_pagination=pagination,
            comments_per_page=per_page
        )
    elif request.method == "POST" and comment_form.validate_on_submit():
        if discussion_id == "new":
            return_id = post_comment(
                repository=repository,
                thread_type="issues",
                thread_id=discussion_id,
                comment_body=comment_form.comment_field.data,
                thread_title=comment_form.discussion_title.data
            )
        else:
            return_id = post_comment(
                repository=repository,
                thread_type="issues",
                thread_id=discussion_id,
                comment_body=comment_form.comment_field.data
            )
        return redirect(url_for(
            "social.discussion",
            repository=repository,
            discussion_id=return_id,
            page=1,
            per_page=per_page
        ))
    else:
        # Form didn't validate
        return render_template(
            "social/discussion.html",
            comments=comments_list,
            comment_form=comment_form,
            current_repository=repository,
            discussion_id=discussion_id,
            comments_pagination=pagination,
            comments_per_page=per_page
        )
