"""
views.py:
The views functions that handle the front-end of the application
"""

from catedit import app, github
from catedit.models import Category
import catedit.persistence
from flask import render_template, request, redirect, url_for, session
from flask.ext.github import GitHubError
from flask_wtf import Form
from catedit.resources import CategoryAPI, CategoryChangesAPI
from wtforms import StringField, TextAreaField
from wtforms.validators import DataRequired
from rdflib import Graph
from io import StringIO

logger = app.logger


class NewCategoryMinimalForm(Form):
    """
        Custom form class for creating a category with the absolute minimal
        attributes (label and description)
    """
    label = StringField(
        "Nom de la categorie (obligatoire)",
        validators=[DataRequired()]
        )
    description = TextAreaField(
        "Description de la categorie (obligatoire)",
        validators=[DataRequired()]
        )


class CommitForm(Form):
    """
        Custom form class for commiting changes
    """
    commit_message = StringField(
        "Message de soumission (obligatoire)",
        validators=[DataRequired()]
        )


@app.route('/', methods=['GET'])
@app.route('/index', methods=['GER'])
def cat_index():
    return render_template("catindex.html")


@app.route('/catrecap/delete-modifs-<deleted_modifs_id>',
           defaults={'deleted_cat_id': None},
           methods=['POST'])
@app.route('/catrecap/delete-<deleted_cat_id>',
           defaults={'deleted_modifs_id': None},
           methods=['POST'])
@app.route('/catrecap',
           defaults={'deleted_cat_id': None, 'deleted_modifs_id': None},
           methods=['GET', 'POST'])
def cat_recap(deleted_cat_id, deleted_modifs_id):
    """
        View that has a list of all categories available. Template is
        catrecap.html, located in src/templates/

        Note: it also handles category deletion from the same page.
    """
    cat_api_instance = CategoryAPI()
    cat_changes_api_instance = CategoryChangesAPI()

    if request.method == "POST":
        if (session.get("user_logged", None) is not None and
                session.get("user_can_edit", False) is not False):
            if (deleted_modifs_id is None and deleted_cat_id is None):
                cat_changes_api_instance.delete()
            elif deleted_modifs_id is not None:
                logger.debug("Changes for category "
                             + deleted_modifs_id
                             + " will be deleted.")
                cat_changes_api_instance.delete(deleted_modifs_id)
            # We identify if we want to delete a category
            elif deleted_cat_id is not None:
                logger.debug("Category "
                             + deleted_cat_id
                             + " will be deleted.")
                cat_api_instance.delete(deleted_cat_id)
        return redirect(url_for('cat_recap'))
    elif request.method == "GET":

        deleted_cat_list = []
        modified_cat_list = []
        serialized_cat_list = []
        if session.get("user_logged", None) is not None:
            serialized_cat_list = cat_api_instance.get()
            cat_changes = cat_changes_api_instance.get()[0]
            modified_cat_list = cat_changes["modified_categories"]
            logger.debug(modified_cat_list)
            deleted_cat_list = cat_changes["deleted_categories"]
            logger.debug(deleted_cat_list)
        # logger.debug(serialized_cat_list)
        cat_list = []

        # first we find the untouched, edited and deleted categories
        for serialized_cat in serialized_cat_list:
            cat_rdf_graph = Graph()
            cat_rdf_graph.parse(source=StringIO(serialized_cat),
                                format='turtle')
            cat = Category(graph=cat_rdf_graph)

            logger.debug(str([cat["name"] for cat in modified_cat_list]))
            logger.debug(str([cat["name"] for cat in deleted_cat_list]))

            if cat.cat_id in [cat["name"] for cat in modified_cat_list]:
                cat_state = "modified"
            elif cat.cat_id in [cat["name"] for cat in deleted_cat_list]:
                cat_state = "deleted"
            else:
                cat_state = "untouched"

            cat_list.append({"cat_label": cat.label,
                             "cat_description": cat.description,
                             "cat_id": cat.cat_id,
                             "cat_properties": cat.properties,
                             "state": cat_state})

        # now we must find the not yet submitted categories that were created
        for modified_cat in modified_cat_list:
            if modified_cat["name"] not in [cat["cat_id"] for cat in cat_list]:
                new_cat_rdf_graph = Graph()
                new_cat_rdf_graph.parse(source=StringIO(
                                            modified_cat["content"]
                                        ),
                                        format='turtle')
                new_cat = Category(graph=new_cat_rdf_graph)
                cat_list.append({"cat_label": new_cat.label,
                                 "cat_description": new_cat.description,
                                 "cat_id": new_cat.cat_id,
                                 "cat_properties": new_cat.properties,
                                 "state": "created"})

            # logger.debug(c.properties)
        return render_template('catrecap.html',
                               cat_list=cat_list)

@app.route('/catmodifs/delete-modifs-<deleted_modifs_id>',
           defaults={'deleted_cat_id': None},
           methods=['POST'])
@app.route('/catmodifs/delete-<deleted_cat_id>',
           defaults={'deleted_modifs_id': None},
           methods=['POST'])
@app.route('/catmodifs',
           defaults={'deleted_cat_id': None, 'deleted_modifs_id': None},
           methods=['GET', 'POST'])
def cat_modifs(deleted_cat_id, deleted_modifs_id):
    """
        View that handles all the visualisation of changes for a user's
        session, links to the editor forms, allows the users to cancel their
        own changes and submits all their changes.
    """
    cat_list = []
    modified_cat_list = []
    created_cat_list = []
    deleted_cat_namelist = []
    modified_cat_namelist = []

    cat_api_instance = CategoryAPI()
    cat_changes_api_instance = CategoryChangesAPI()

    if deleted_cat_id is None and deleted_modifs_id is None:
        commit_form = CommitForm(request.form)

        # if it's a GET with no delete_cat_id or deleted_modifs_id, then we'll
        # display the changes
        if request.method == "GET":
            if session.get("user_logged", None) is not None:
                serialized_cat_list = cat_api_instance.get()
                changes_list = cat_changes_api_instance.get()[0]

                # Creating existing cat list
                for serialized_cat in serialized_cat_list:
                    cat_rdf_graph = Graph()
                    cat_rdf_graph.parse(source=StringIO(serialized_cat),
                                        format='turtle')
                    cat = Category(graph=cat_rdf_graph)

                    cat_list.append({"cat_label": cat.label,
                                     "cat_description": cat.description,
                                     "cat_id": cat.cat_id,
                                     "cat_properties": cat.properties})

                # Creating modified and created cat lists
                for modified_category in \
                        changes_list.get("modified_categories", []):
                    modified_cat_rdf_graph = Graph()
                    modified_cat_rdf_graph.parse(
                        source=StringIO(
                            modified_category["content"]
                        ),
                        format='turtle'
                    )
                    modified_cat = Category(graph=modified_cat_rdf_graph)
                    if modified_cat.cat_id in \
                            [existing_cat["cat_id"]
                                for existing_cat in cat_list]:
                        modified_cat_list.append(
                            {"cat_label": modified_cat.label,
                             "cat_description": modified_cat.description,
                             "cat_id": modified_cat.cat_id,
                             "cat_properties": modified_cat.properties})
                    else:
                        created_cat_list.append(
                            {"cat_label": modified_cat.label,
                             "cat_description": modified_cat.description,
                             "cat_id": modified_cat.cat_id,
                             "cat_properties": modified_cat.properties})

                # Creating deleted cat list
                deleted_cat_namelist = [
                    element["name"] for element in
                    changes_list.get("deleted_categories", [])
                ]
                modified_cat_namelist = [
                    element["name"] for element in
                    changes_list.get("modified_categories", [])
                ]
            return render_template('catmodifs.html',
                                   cat_list=cat_list,
                                   created_cat_list=created_cat_list,
                                   modified_cat_list=modified_cat_list,
                                   modified_cat_namelist=modified_cat_namelist,
                                   deleted_cat_namelist=deleted_cat_namelist,
                                   commit_form=commit_form)
        # If it's a POST with no delete_cat_id and delete_modifs_id, then we
        # will submit all the stored changes
        elif request.method == "POST":
            if commit_form.validate_on_submit():
                cat_api_instance.put()
                cat_changes_api_instance.delete()
            return redirect(url_for('cat_recap'))

    # One of the ids is not None (either deleting a category or a modification)
    else:
        # We only do that if we have a POST
        if request.method == "POST":
            # We identify if we want to delete a change
            if (session.get("user_logged", None) is not None and
                session.get("user_can_edit", False) is not False):
                if deleted_modifs_id is not None:
                    cat_changes_api_instance.delete(deleted_modifs_id)
                # We identify if we want to delete a category
                elif deleted_cat_id is not None:
                    cat_api_instance.delete(delete_cat_id)
        return redirect(url_for('cat_modifs'))


@app.route('/cateditor', methods=['GET', 'POST'])
@app.route('/cateditor/<string:cat_id>', methods=['GET', 'POST'])
def cat_editor(cat_id=None):
    """
        View that handles creation and edition of categories. Template is
        cateditor.html, located in src/templates
    """
    cat_api_instance = CategoryAPI()
    cat_changes_api_instance = CategoryChangesAPI()

    cat_list = []
    deleted_cat_list = []
    serialized_cat_list = []
    if session.get("user_logged", None) is not None:
        serialized_cat_list = cat_api_instance.get()
        cat_changes=cat_changes_api_instance.get()
        deleted_cat_list = [
            category["name"] for category in
            cat_changes_api_instance.get()[0]["deleted_categories"]
        ]
        modified_cat_list = [
            category["name"] for category in
            cat_changes_api_instance.get()[0]["modified_categories"]
        ]
    for serialized_cat in serialized_cat_list:
        cat_rdf_graph = Graph()
        cat_rdf_graph.parse(source=StringIO(serialized_cat),
                            format='turtle')
        cat = Category(graph=cat_rdf_graph)
        if cat.cat_id not in deleted_cat_list:
            cat_list.append({"cat_label": cat.label,
                             "cat_description": cat.description,
                             "cat_id": cat.cat_id,
                             "cat_properties": cat.properties})

    for modified_cat in modified_cat_list:
        if modified_cat["name"] not in [cat["cat_id"] for cat in cat_list]:
            modified_cat_rdf_graph = Graph()
            modified_cat_rdf_graph.parse(
                source=StringIO(modified_cat["content"]),
                format='turtle'
            )
            modified_cat = Category(graph=modified_cat_rdf_graph)
            cat_list.append({"cat_label": modified_cat.label,
                             "cat_description": modified_cat.description,
                             "cat_id": modified_cat.cat_id,
                             "cat_properties": modified_cat.properties})

    if cat_id is not None:
        specific_serialized_cat = ""
        changes_response = cat_changes_api_instance.get(cat_id)
        # that means the category was modified or deleted
        if changes_response != 404:
            if changes_response[0]["type"] is not "deleted":
                specific_serialized_cat = changes_response[0]["category"] \
                                                             ["content"]
        else:
            specific_serialized_cat = cat_api_instance.get(cat_id)
        logger.debug(specific_serialized_cat)
        cat_rdf_graph = Graph()
        cat_rdf_graph.parse(source=StringIO(specific_serialized_cat),
                            format='turtle')
        cat = Category(graph=cat_rdf_graph)

        setattr(NewCategoryMinimalForm,
                'label',
                StringField("Nom de la categorie",
                            validators=[DataRequired()],
                            default=cat.label))
        setattr(NewCategoryMinimalForm,
                'description',
                TextAreaField("Description de la categorie",
                              validators=[DataRequired()],
                              default=cat.description))
        logger.debug("CatForm fields preset to "
                     + cat.label + " and "
                     + cat.description)

        cat_form = NewCategoryMinimalForm(request.form)

        # GET + cat_id = Edit cat form
        if request.method == 'GET':
            return render_template('cateditor.html',
                                   cat_id=cat.cat_id,
                                   cat_properties=cat.properties,
                                   form=cat_form,
                                   cat_list=cat_list,
                                   deleted_cat_list=deleted_cat_list)

        # PUT + cat_id = Submit edited cat, only if cat is not already
        # in deleted categories
        if cat_form.validate_on_submit() and cat.cat_id not in \
                [element["name"] for element in
                    session.get("deleted_categories", [])]:
            if (session.get("user_logged", None) is not None and
                    session.get("user_can_edit", False) is not False):
                cat_api_instance.put(cat_id)
            return redirect(url_for('cat_recap'))
        else:
            return render_template('cateditor.html',
                                   cat_id=cat_id,
                                   cat_properties=cat.properties,
                                   form=cat_form,
                                   cat_list=cat_list,
                                   deleted_cat_list=deleted_cat_list)

    else:
        setattr(NewCategoryMinimalForm,
                'label',
                StringField("Nom de la categorie",
                            validators=[DataRequired()]))
        setattr(NewCategoryMinimalForm,
                'description',
                TextAreaField("Description de la categorie",
                              validators=[DataRequired()]))

        cat_form = NewCategoryMinimalForm(request.form)

        # GET seul = Create cat form
        if request.method == 'GET':
            return render_template('cateditor.html',
                                   form=cat_form,
                                   cat_list=cat_list,
                                   deleted_cat_list=deleted_cat_list)

        # POST seul = Submit created cat
        if cat_form.validate_on_submit():
            if (session.get("user_logged", None) is not None and
                    session.get("user_can_edit", False) is not False):
                cat_api_instance.post()
            return redirect(url_for('cat_recap'))
        else:
            return render_template('cateditor.html',
                                   form=cat_form,
                                   cat_list=cat_list,
                                   deleted_cat_list=deleted_cat_list)


@app.route('/catedit-github-login')
def github_login():
    """
        Function that manages authentication (Github), login

        Note: If Persistence is set to PersistenceToFile (categories stored
        in local files, used for debugging), creates a mock user named
        "FileEditUser"
    """
    if getattr(catedit.persistence,
               app.config["PERSISTENCE_METHOD"])().session_compliant is True:
        session["save_user_changes"] = True
        session["modified_categories"] = []
        session["deleted_categories"] = []
    if app.config["PERSISTENCE_METHOD"] == "PersistenceToGithub":
        return github.authorize(scope="repo", redirect_uri=url_for('github_callback', _external=True))
    elif app.config["PERSISTENCE_METHOD"] == "PersistenceToFile":
        session["user_logged"] = True
        session["user_can_edit"] = True
        session["user_login"] = "FileEditUser"
        return redirect(url_for('cat_index'))


@app.route('/catedit-github-callback')
@github.authorized_handler
def github_callback(oauth_code):
    """
        Function that handles callback from Github after succesful login
    """
    session.permanent = False
    session["user_code"] = oauth_code
    session["user_logged"] = True
    session["user_login"] = github.get("user")["login"]
    try:
        repo_list = []
        repo_list = github.get("user/repos")
        for repo in repo_list:
            logger.debug(repo["name"])
        session["user_can_edit"] = True
        if not any(repo["name"] == app.config["REPOSITORY_NAME"]
                   for repo in repo_list):
            session["user_can_edit"] = False
        logger.debug(session["user_can_edit"])
    except GitHubError:
        logger.debug("error getting repos!")

    logger.debug(session["user_login"])
    return redirect(url_for('cat_index'))


@github.access_token_getter
def token_getter():
    """
        Utility function for github-flask module to get user token when
        making authenticated requests
    """
    if session.get("user_logged", None):
        # logger.debug("I made an authentified request")
        return session["user_code"]


@app.route('/catedit-logout')
def logout():
    """
        Function that manages authentication (Github), logout

        Note: if you want to switch github users, you will have to logout of
        Github, else when logging back in, github will send the app the
        same oauth code
    """
    session["user_code"] = None
    session["user_logged"] = None
    session["user_login"] = None
    session["user_can_edit"] = None
    session["save_user_changes"] = None
    session["modified_categories"] = []
    session["deleted_categories"] = []
    return redirect(url_for('cat_recap'))
