"""
    utils.py:
    Module that lists utility functions used through the app
"""
from catedit import app
import logging

from flask import session
from rdflib import RDF, RDFS, URIRef
from rdflib.compare import to_isomorphic, graph_diff


logger = logging.getLogger(__name__)

def compare_categories(first_category, second_category, repository, with_details=True):
    """
        Compares 2 categories and generate a dict with 3 lists of
        differences:
        * "same_category" is a True/False attribute which is True if both
        categories have the same id
        * "same_content" is a True/False attribute which is True if both
        categories are the exact same
        * "only_in_first" are the properties we can only find
        in first_category
        * "only_in_second" are the properties we can only find
        in second_category
        * "in_both" are the properties that can be found in both
        categories. To be considered "in both" a property has to have the same
        predicate and object in both categories that must have the same id.

        Each "property" is a triple with predicate id (property key defined in
        config PROPERTY_LIST) and object (either id if we're referencing
        another category or the link/text in other cases)

        If both categories are the exact same, returns
    """
    compare_result = {}
    first_iso_cat = to_isomorphic(first_category.cat_graph)
    second_iso_cat = to_isomorphic(second_category.cat_graph)

    compare_result["same_id"] = (
        first_category.cat_id == second_category.cat_id
    )
    compare_result["same_content"] = (
        first_iso_cat == second_iso_cat
    )

    if not(compare_result["same_content"]) and with_details:
        rdf_in_both = []
        rdf_only_in_first = []
        rdf_only_in_second = []
        rdf_in_both, rdf_only_in_first, rdf_only_in_second = graph_diff(
            first_category.cat_graph,
            second_category.cat_graph
        )
        in_first = []
        in_both = []
        in_second = []
        logger.debug(rdf_in_both)
        logger.debug(rdf_only_in_first)
        logger.debug(rdf_only_in_second)
        for (final_list, diff_list) in [
                (in_both, rdf_in_both),
                (in_first, rdf_only_in_first),
                (in_second, rdf_only_in_second)
        ]:
            for triple in diff_list.triples((None, None, None)):
                if triple[1] == RDFS.label:
                    final_list.append(("label", triple[2].toPython()))
                elif triple[1] == RDF.Description:
                    final_list.append(("description", triple[2].toPython()))
                else:
                    for predicate in session["properties"][repository].keys():
                        if triple[1] == \
                        URIRef(session["properties"][repository][predicate]["rdflib_class"]):
                            if (session["properties"]
                                       [repository]
                                       [predicate]
                                       ["object_type"] == "uriref-link"
                            or session["properties"]
                                      [repository]
                                      [predicate]
                                      ["object_type"] == "literal"):
                                final_list.append(
                                    (
                                        predicate,
                                        triple[2].toPython()
                                    )
                                )
                            else:
                                final_list.append(
                                    (
                                        predicate,
                                        triple[2].toPython().split("#", 1)[1]
                                    )
                                )
        compare_result["only_in_first"] = in_first
        compare_result["only_in_second"] = in_second
        compare_result["in_both"] = in_both
    return compare_result



def make_differences_list(first_category_list, second_category_list, repository):
    """
        Compares 2 category lists and generates a dict that lists addition,
        modification and deletions from first_category_list to
        second_category_list

        * "additions": list of categories that were added
        * "modifications" : list of couples before/after of categories that
        were modified
        * "deletions": list of categories that were deleted
    """
    created_categories = []
    modified_categories = []
    deleted_categories = []

    for first_list_category in first_category_list:
        if first_list_category.cat_id not in [second_list_category.cat_id
        for second_list_category in second_category_list]:
            deleted_categories.append(first_list_category)
        else:
            for second_list_category in second_category_list:
                if first_list_category.cat_id == second_list_category.cat_id:
                    if not(compare_categories(
                            first_list_category,
                            second_list_category,
                            repository,
                            with_details=False
                    )["same_content"]):
                        modified_categories.append(
                            (first_list_category, second_list_category)
                        )

    for second_list_category in second_category_list:
        if second_list_category.cat_id not in [first_list_category.cat_id
        for first_list_category in first_category_list]:
            created_categories.append(second_list_category)
    return {
        "additions": created_categories,
        "modifications": modified_categories,
        "deletions": deleted_categories
    }
