"""
__init__.py:
module main file used to configure the Flask app
"""

from logging import FileHandler, Formatter
import os
from catedit.version import CURRENT_VERSION

from flask import Flask, session
from flask_wtf.csrf import CsrfProtect
from flask.ext.github import GitHub
from flask.ext.cache import Cache
from flask.ext.restful import Api

from catedit.settings import AppSettings

# Set up app
app = Flask(__name__)
app.config.from_object(AppSettings)
cache = Cache(app, config={"CACHE_TYPE": "simple"})
app_configured = False
try:
    from catedit.config import AppConfig
    app.config.from_object(AppConfig)
    app_configured = True
except ImportError:
    pass

if 'CATEDIT_SETTINGS' in os.environ:
    app.config.from_envvar('CATEDIT_SETTINGS')
    app_configured = True

if not app_configured:
    raise Exception("Catedit not configured")

# CSRF protection
CsrfProtect(app)

# Github
app.config["GITHUB_CLIENT_ID"] = app.config["PERSISTENCE_CONFIG"] \
                                           ["GITHUB_CLIENT_ID"]
app.config["GITHUB_CLIENT_SECRET"] = app.config["PERSISTENCE_CONFIG"] \
                                               ["GITHUB_CLIENT_SECRET"]
github = GitHub(app)

# Api
api = Api(app)

# Version

app.config["CURRENT_VERSION"] = CURRENT_VERSION

# Views and APIs
from catedit.views import cat_editor, cat_recap, github_login,\
                          github_callback, logout

from catedit.resources import CategoryAPI, CategoryChangesAPI


api.add_resource(CategoryAPI,
                 '/category/<string:cat_id>',
                 '/category',
                 endpoint='category')
api.add_resource(CategoryChangesAPI,
                 '/category-changes/<string:cat_id>',
                 '/category-changes',
                 endpoint='category_changes')


# Set up logging
if app.config["LOGGING_CONFIG"]["IS_LOGGING"]:
    file_handler = FileHandler(filename=app.config["LOGGING_CONFIG"]
                                                  ["LOG_FILE_PATH"])
    file_handler.setFormatter(Formatter(
        '%(asctime)s %(levelname)s: %(message)s '
        '[in %(pathname)s:%(lineno)d]',
        '%Y-%m-%d %H:%M:%S'
    ))
    app.logger.addHandler(file_handler)
    app.logger.setLevel(app.config["LOGGING_CONFIG"]["LOGGING_LEVEL"])

# Session management

app.secret_key = app.config["SECRET_KEY"]
