import catedit
import unittest
from catedit.models import Category
from rdflib import Graph
from io import StringIO
from flask import session

class CateditTestCase(unittest.TestCase):

    def setUp(self):
        catedit.app.config['TESTING'] = True
        catedit.app.config.from_object(catedit.settings.AppSettings)
        catedit.app.config['PERSISTENCE_CONFIG'] = {
            "METHOD": "PersistenceToFile",
            "FILE_SAVE_DIRECTORY": "../../files/",
            "REPOSITORY_LIST": ["local"]
        }
        self.app = catedit.app.test_client()
        self.category_api = catedit.resources.CategoryAPI()
        self.category_changes_api = catedit.resources.CategoryChangesAPI()

    def tearDown(self):
        pass

    def test_1_1_model_create_category(self):
        pass

    def test_1_2_model_access_data(self):
        pass

    def test_1_3_model_editing_category(self):
        pass

    def test_1_4_model_save_load(self):
        pass

    def test_2_1_api_empty(self):
        with catedit.app.test_request_context():
            session["modified_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }
            session["deleted_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }
            print("Testing empty category list")
            assert self.category_api.get(repository="local") == ([], 200)
            print("Testing empty category changes list")
            assert self.category_changes_api.get(repository="local") == (
                {
                    "modified_categories": {},
                    "deleted_categories": {}
                },
                200
            )
            print("Testing getting dummy category id")
            assert self.category_api.get(
                repository="local",
                cat_id="nil"
            ) == 404
            print("Testing getting dummy category change id")
            assert self.category_changes_api.get(
                repository="local",
                modified_cat_id="nil"
            ) == ({
                "type": "untouched",
                "category": "nil"
            }, 200)

    def test_2_2_api_creation(self):
        with catedit.app.test_request_context():
            session["modified_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }
            session["deleted_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }

            print("Testing posting new category in changelist")
            cat_data = {
                "label": "new_label",
                "description": "new_description",
                "properties": [],
            }
            (cat_id, cat_serial, code) = self.category_api.post(
                repository="local",
                cat_data=cat_data
            )
            cat_graph = Graph()
            cat_graph.parse(source=StringIO(cat_serial), format='turtle')
            cat_object = Category(graph=cat_graph)
            assert cat_object.label == "new_label"
            assert cat_object.description == "new_description"
            assert cat_object.properties == []
            assert code == 201
            print("Checking if new category can be found in changelist")
            assert self.category_changes_api.get(
                repository="local",
                modified_cat_id=cat_id
            ) != 404

    def test_2_3_api_edit_changes(self):
        with catedit.app.test_request_context():
            session["modified_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }
            session["deleted_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }

            print("Creating new category changes that will be edited")
            cat_data = {
                "label": "new_label",
                "description": "new_description",
                "properties": [],
            }
            (cat_id, cat_serial, code) = self.category_api.post(
                repository="local",
                cat_data=cat_data
            )
            assert code == 201

            print("Testing editing new category")
            edited_cat_data = {
                "label": "new_label_edited",
                "description": "new_description_edited",
                "properties": [("comment","edited_category")],
            }
            code = self.category_api.put(
                repository="local",
                cat_id=cat_id,
                cat_data=edited_cat_data
            )
            assert code == 204

            (cat_changes, code) = self.category_changes_api.get(
                repository="local",
                modified_cat_id=cat_id
            )

            cat_graph = Graph()
            cat_graph.parse(
                source=StringIO(cat_changes["category"][cat_id]),
                format='turtle'
            )
            cat_object = Category(graph=cat_graph)
            assert cat_object.label == "new_label_edited"
            assert cat_object.description == "new_description_edited"
            assert cat_object.properties == [("comment","edited_category")]

    def test_2_4_api_delete_changes(self):
        with catedit.app.test_request_context():
            session["modified_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }
            session["deleted_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }

            print("Creating new category changes that will be deleted")
            cat_data = {
                "label": "new_label",
                "description": "new_description",
                "properties": [],
            }
            (cat_id, cat_serial, code) = self.category_api.post(
                repository="local",
                cat_data=cat_data
            )
            assert session["modified_categories"]["local"] != {}
            assert code == 201

            print("Deleting changes for created category")
            code = self.category_changes_api.delete(
                repository="local",
                modified_cat_id=cat_id
            )
            assert code == 204
            assert session["modified_categories"]["local"] == {}
            assert session["deleted_categories"]["local"] == {}

    def test_2_5_api_submit_edit_delete_category(self):
        with catedit.app.test_request_context():
            session["modified_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }
            session["deleted_categories"] = {
                repo: {} for repo in catedit.app.config["PERSISTENCE_CONFIG"]
                                                       ["REPOSITORY_LIST"]
            }

            print("Creating new categories")
            cat_data_1 = {
                "label": "new_label_1",
                "description": "new_description_1",
                "properties": [],
            }
            (cat_id_1, cat_serial_1, code) = self.category_api.post(
                repository="local",
                cat_data=cat_data_1
            )
            assert code == 201

            cat_data_2 = {
                "label": "new_label_2",
                "description": "new_description_2",
                "properties": [],
            }
            (cat_id_2, cat_serial_2, code) = self.category_api.post(
                repository="local",
                cat_data=cat_data_2
            )
            assert code == 201

            print("Submitting changes")
            code = self.category_api.put(
                repository="local"
            )
            assert code == 204

            print("Deleting categories")
            (categories, code) = self.category_api.get(
                repository="local"
            )
            for category_content in categories:
                cat_graph = Graph()
                cat_graph.parse(
                    source=StringIO(category_content),
                    format='turtle'
                )
                cat_object = Category(graph=cat_graph)

                assert cat_object.label == "new_label_1" or \
                       cat_object.label == "new_label_2"
                assert cat_object.description == "new_description_1" or \
                       cat_object.description == "new_description_2"
                code = self.category_api.delete(
                    repository="local",
                    deleted_cat_id=cat_object.cat_id
                )
                assert code == 204
            assert session["deleted_categories"] != {}

            print("Submitting deletions")
            code = self.category_api.put(
                repository="local"
            )
            assert code == 204
            assert session["modified_categories"]["local"] == {}
            assert session["deleted_categories"]["local"] == {}

    def test_3_views(self):
        print("Starting views.py tests...")
        print("Completed views.py tests!")

if __name__ == '__main__':
    unittest.main()
